#include "ccarstate.h"
#include "ccar.h"

#include <wx/arrimpl.cpp>
WX_DEFINE_OBJARRAY(cCarStateArray);

//-----------------------------------------------------------------------------
// cLogtype and cWheeltype

void cLogtype::Set(wxString name, wxString unit, wxString code, bool perwheel, bool integer, bool symmetric)
{
  m_Name = name;
  m_Unit = unit;
  m_Code = code;
  m_IsPerWheel = perwheel;
  m_IsInteger = integer;
  m_IsSymmetric = symmetric;
}


void cWheeltype::Set(wxString name, wxString code)
{
  m_Name = name;
  m_Code = code;
}

//-----------------------------------------------------------------------------
// Static members

cLogtype cCarState::s_Logtypes[LOGTYPE_LAST];
cWheeltype cCarState::s_Wheeltypes[WHEELTYPE_LAST];
bool cCarState::s_TypesDefined = false;
bool cCarState::s_SpeedInMph = false;

//-----------------------------------------------------------------------------
//

void cCarState::DefineTypes()
{
  wxASSERT(!s_TypesDefined);
  s_Logtypes[LOGTYPE_SPEED]          .Set( _T("Speed"),                       _T("km/h"), _T("spd"), false, false, false);
  s_Logtypes[LOGTYPE_TIMEINLAP]      .Set( _T("Time"),                        _T("s"),    _T("tim"), false, false, false);
  s_Logtypes[LOGTYPE_RPM]            .Set( _T("Rpm"),                         _T("rpm"),  _T("rpm"), false, false, false);
  s_Logtypes[LOGTYPE_GEAR]           .Set( _T("Gear"),                        _T(""),     _T("ger"), false, true,  false);
  s_Logtypes[LOGTYPE_STEER]          .Set( _T("Steer"),                       _T("deg"),  _T("str"), false, false, true);
  s_Logtypes[LOGTYPE_THROTTLE]       .Set( _T("Throttle"),                    _T("%"),    _T("thr"), false, false, false);
  s_Logtypes[LOGTYPE_BRAKE]          .Set( _T("Brake"),                       _T("%"),    _T("brk"), false, false, false);
  s_Logtypes[LOGTYPE_SLIPRATIO]      .Set( _T("Slip ratio"),                  _T(""),     _T("slr"), true,  false, false);
  s_Logtypes[LOGTYPE_CAMBER]         .Set( _T("Camber"),                      _T("deg"),  _T("cmb"), true,  false, true);
  s_Logtypes[LOGTYPE_SUSPTRAVEL]     .Set( _T("Suspension travel remaining"), _T("mm"),   _T("stv"), true,  false, false);
  s_Logtypes[LOGTYPE_FORCE_LON]      .Set( _T("Longitudinal wheel force"),    _T("N"),    _T("low"), true,  false, true);
  s_Logtypes[LOGTYPE_FORCE_LAT]      .Set( _T("Lateral wheel force"),         _T("N"),    _T("law"), true,  false, true);
  s_Logtypes[LOGTYPE_TYRELOAD]       .Set( _T("Tyre load"),                   _T("N"),    _T("tld"), true,  false, true);
  s_Logtypes[LOGTYPE_GLONG]          .Set( _T("Longitudinal G"),              _T("g"),    _T("log"), false, false, true);
  s_Logtypes[LOGTYPE_GLAT]           .Set( _T("Lateral G"),                   _T("g"),    _T("lag"), false, false, true);
  s_Logtypes[LOGTYPE_GTOTAL]         .Set( _T("Total G"),                     _T("g"),    _T("tog"), false, false, true);
  s_Logtypes[LOGTYPE_WHEELANGLE]     .Set( _T("Wheel angle"),                 _T("deg"),  _T("wan"), true,  false, true);
  s_Logtypes[LOGTYPE_SLIPANGLE]      .Set( _T("Slip angle"),                  _T("deg"),  _T("san"), false, false, true);
  s_Logtypes[LOGTYPE_CLUTCH]         .Set( _T("Clutch"),                      _T("%"),    _T("clu"), false, false, false);
  s_Logtypes[LOGTYPE_WHEELSPEED_VERT].Set( _T("Vertical wheel speed"),        _T("mm/s"), _T("vws"), true,  false, true);
  s_Logtypes[LOGTYPE_WHEELSPEED_ROT] .Set( _T("Rotational wheel speed"),      _T("km/h"), _T("rws"), true,  false, false);
  s_Logtypes[LOGTYPE_POWER]          .Set( _T("Power"),                       _T("kW"),   _T("pwr"), true,  false, true);

  s_Wheeltypes[WHEELTYPE_FRONT_LEFT] .Set( _T("Front Left"),   _T("fl"));
  s_Wheeltypes[WHEELTYPE_FRONT_RIGHT].Set( _T("Front Right"),  _T("fr"));
  s_Wheeltypes[WHEELTYPE_REAR_LEFT]  .Set( _T("Rear Left"),    _T("rl"));
  s_Wheeltypes[WHEELTYPE_REAR_RIGHT] .Set( _T("Rear Right"),   _T("rr"));
  s_Wheeltypes[WHEELTYPE_AVG]        .Set( _T("Avg"),          _T("av"));
  s_Wheeltypes[WHEELTYPE_AVG_FRONT]  .Set( _T("Avg Front"),    _T("avf"));
  s_Wheeltypes[WHEELTYPE_AVG_REAR]   .Set( _T("Avg Rear"),     _T("avr"));
  s_Wheeltypes[WHEELTYPE_MIN]        .Set( _T("Min"),          _T("min"));
  s_Wheeltypes[WHEELTYPE_MIN_FRONT]  .Set( _T("Min Front"),    _T("mif"));
  s_Wheeltypes[WHEELTYPE_MIN_REAR]   .Set( _T("Min Rear"),     _T("mir"));
  s_Wheeltypes[WHEELTYPE_MAX]        .Set( _T("Max"),          _T("max"));
  s_Wheeltypes[WHEELTYPE_MAX_FRONT]  .Set( _T("Max Front"),    _T("maf"));
  s_Wheeltypes[WHEELTYPE_MAX_REAR]   .Set( _T("Max Rear"),     _T("mar"));
  s_Wheeltypes[WHEELTYPE_DELTA_ALL]  .Set( _T("Delta F-R"),    _T("daa"));
  s_Wheeltypes[WHEELTYPE_DELTA_FRONT].Set( _T("Delta FL-FR"),  _T("daf"));
  s_Wheeltypes[WHEELTYPE_DELTA_REAR] .Set( _T("Delta RL-RR"),  _T("dar"));

  s_TypesDefined = true;
}

//-----------------------------------------------------------------------------

cCarState::cCarState(cCar* car)
: m_Car(car),
  m_Radius(0.0f),
  m_Glong(0.0f),
  m_Glat(0.0f),
  m_SlipAngle(0.0f)
{
  m_WheelSpeed[0] = 0.0f;
  m_WheelSpeed[1] = 0.0f;
  m_WheelSpeed[2] = 0.0f;
  m_WheelSpeed[3] = 0.0f;
}

//-----------------------------------------------------------------------------

cCarState::~cCarState()
{
}

//-----------------------------------------------------------------------------
// Read the various aspects of the car state

float cCarState::GetLogData(int log, int wheel) const
{
  wxASSERT(s_TypesDefined);
  wxASSERT(log >= LOGTYPE_FIRST);
  wxASSERT(log < LOGTYPE_LAST);
  wxASSERT(wheel >= WHEELTYPE_FIRST);
  wxASSERT(wheel < WHEELTYPE_LAST);

  if ((wheel >= 4) && (IsPerWheel(log))) {
    // "compound" wheel types: read the value for each wheel
    float val_fl = GetLogData(log, WHEELTYPE_FRONT_LEFT);
    float val_fr = GetLogData(log, WHEELTYPE_FRONT_RIGHT);
    float val_rl = GetLogData(log, WHEELTYPE_REAR_LEFT);
    float val_rr = GetLogData(log, WHEELTYPE_REAR_RIGHT);

    // combine the values
    float result;
    switch (wheel) {
      case WHEELTYPE_AVG        : return (val_fl + val_fr + val_rl + val_rr) / 4.0f;
      case WHEELTYPE_AVG_FRONT  : return (val_fl + val_fr) / 2.0f;
      case WHEELTYPE_AVG_REAR   : return (val_rl + val_rr) / 2.0f;

      case WHEELTYPE_MIN        :
        result = val_fl;
        if (val_fr < result) result = val_fr;
        if (val_rl < result) result = val_rl;
        if (val_rr < result) result = val_rr;
        return result;

      case WHEELTYPE_MIN_FRONT  : return (val_fl < val_fr) ? val_fl : val_fr;
      case WHEELTYPE_MIN_REAR   : return (val_rl < val_rr) ? val_rl : val_rr;

      case WHEELTYPE_MAX        :
        result = val_fl;
        if (val_fr > result) result = val_fr;
        if (val_rl > result) result = val_rl;
        if (val_rr > result) result = val_rr;
        return result;

      case WHEELTYPE_MAX_FRONT  : return (val_fl > val_fr) ? val_fl : val_fr;
      case WHEELTYPE_MAX_REAR   : return (val_rl > val_rr) ? val_rl : val_rr;

      case WHEELTYPE_DELTA_ALL    : return ((val_fl + val_fr) - (val_rl + val_rr)) / 2.0f;
      case WHEELTYPE_DELTA_FRONT  : return val_fl - val_fr;
      case WHEELTYPE_DELTA_REAR   : return val_rl - val_rr;

      default :
        // unknown wheel type; should not occur
        wxFAIL;
        return 0.0f; // defensive
    }
  }

  // "simple" wheel types
  switch (log) {
    case LOGTYPE_SPEED :
      // convert from m/s to km/h or to mph
      return (cCarState::s_SpeedInMph) ? (m_Speed * 3.6f / 1.609f) : (m_Speed * 3.6f);

    case LOGTYPE_TIMEINLAP  : return m_TimeInLap;
    case LOGTYPE_RPM        : return m_Rpm * RADPSEC2RPM; // from radian/s to rpm
    case LOGTYPE_GEAR       : return m_Gear - 1; // reverse = -1, 0 = neutral, 1 = first gear etc.
    case LOGTYPE_STEER      : return m_Steer * RAD2DEG;
    case LOGTYPE_THROTTLE   : return m_Throttle * 100.0f;
    case LOGTYPE_BRAKE      : return m_Brake * 100.0f;

    case LOGTYPE_SLIPRATIO  :
    {
      float speed = GetActualWheelSpeed(wheel);
      if (speed < EPSILON) return 0.0f;
      return (m_AngVel[wheel] * m_Car->m_Radius[wheel]) / speed - 1.0f;
    }

    case LOGTYPE_CAMBER     : return m_Camber[wheel] * RAD2DEG;
    case LOGTYPE_SUSPTRAVEL : return (m_Car->m_MaxDeflect[wheel] - m_Deflect[wheel]) * 1000.0f; // convert to mm
    case LOGTYPE_FORCE_LON  : return m_ForceLong[wheel];
    case LOGTYPE_FORCE_LAT  : return m_ForceLat[wheel];
    case LOGTYPE_TYRELOAD   : return m_Load[wheel];
    case LOGTYPE_GLONG      : return m_Glong / ONE_G;
    case LOGTYPE_GLAT       : return m_Glat / ONE_G;
    case LOGTYPE_GTOTAL     : return sqrt(m_Glong * m_Glong + m_Glat * m_Glat) / ONE_G;
    case LOGTYPE_WHEELANGLE : return m_WheelAngle[wheel] * RAD2DEG;
    case LOGTYPE_SLIPANGLE  : return m_SlipAngle * RAD2DEG;
    case LOGTYPE_CLUTCH     : return m_Clutch * 100.0f;
    case LOGTYPE_WHEELSPEED_VERT : return m_WheelSpeed[wheel] * 1000.0f; // convert to mm/s

    case LOGTYPE_WHEELSPEED_ROT :
    {
      float speed = m_AngVel[wheel] * m_Car->m_Radius[wheel];
      return (cCarState::s_SpeedInMph) ? (speed * 3.6f / 1.609f) : (speed * 3.6f);
    }

    case LOGTYPE_POWER :
      return GetActualWheelSpeed(wheel) * m_ForceLong[wheel] * 0.001f; // convert to kW (= 1000 Nm/s)

    default :
      // unknown logdata type; should not occur
      wxFAIL;
      return 0.0f; // defensive
  }
}

//-----------------------------------------------------------------------------
// Calculate the speed of a wheel in real-world coordinates
// Speed is corrected for turn radius (outside wheels move faster)

float cCarState::GetActualWheelSpeed(int wheel) const
{
  float correction = 1.0f; // correction factor
  if (fabs(m_Radius) > EPSILON) correction -= m_Car->m_WheelX[wheel] / m_Radius;
  return m_Speed * correction;
}

//-----------------------------------------------------------------------------
// Tell if the values are symmetric around the Y axis

bool cCarState::IsSymmetric(int log, int wheel)
{
  wxASSERT(s_TypesDefined);
  if (s_Logtypes[log].m_IsSymmetric) return true;

  switch (wheel) {
    case WHEELTYPE_DELTA_ALL :
    case WHEELTYPE_DELTA_FRONT :
    case WHEELTYPE_DELTA_REAR :
      return true;

    default :
      return false;
  }
}

//-----------------------------------------------------------------------------
//

wxString cCarState::GetLogtypeName(int log)
{
  wxASSERT(s_TypesDefined);
  wxASSERT(log >= LOGTYPE_FIRST);
  wxASSERT(log < LOGTYPE_LAST);

  return s_Logtypes[log].m_Name;
}


wxString cCarState::GetLogtypeCode(int log)
{
  wxASSERT(s_TypesDefined);
  wxASSERT(log >= LOGTYPE_FIRST);
  wxASSERT(log < LOGTYPE_LAST);

  return s_Logtypes[log].m_Code;
}


wxString cCarState::GetLogtypeUnits(int log)
{
  wxASSERT(s_TypesDefined);
  wxASSERT(log >= LOGTYPE_FIRST);
  wxASSERT(log < LOGTYPE_LAST);

  wxString units = s_Logtypes[log].m_Unit;
  if ((cCarState::s_SpeedInMph) && (units = _T("km/h"))) units = _T("mph");
  return units;
};

//-----------------------------------------------------------------------------
// Find the logtype belonging to a display name

int cCarState::FindLogtypeName(const wxString& name)
{
  wxASSERT(s_TypesDefined);
  for (int i = LOGTYPE_FIRST; i < LOGTYPE_LAST; i++) {
    if (name.CmpNoCase(s_Logtypes[i].m_Name) == 0) return i;
  }

  return wxNOT_FOUND;
}

//-----------------------------------------------------------------------------
// Find the logtype belonging to a code

int cCarState::FindLogtypeCode(const wxString& code)
{
  wxASSERT(s_TypesDefined);
  for (int i = LOGTYPE_FIRST; i < LOGTYPE_LAST; i++) {
    if (code.CmpNoCase(s_Logtypes[i].m_Code) == 0) return i;
  }

  return wxNOT_FOUND;
}

//-----------------------------------------------------------------------------
//

wxString cCarState::GetWheeltypeName(int wheel)
{
  wxASSERT(s_TypesDefined);
  wxASSERT(wheel >= WHEELTYPE_FIRST);
  wxASSERT(wheel < WHEELTYPE_LAST);

  return s_Wheeltypes[wheel].m_Name;
}

wxString cCarState::GetWheeltypeCode(int wheel)
{
  wxASSERT(s_TypesDefined);
  wxASSERT(wheel >= WHEELTYPE_FIRST);
  wxASSERT(wheel < WHEELTYPE_LAST);

  return s_Wheeltypes[wheel].m_Code;
}


//-----------------------------------------------------------------------------
// Find the wheeltype belonging to a display name

int cCarState::FindWheeltypeName(const wxString& name)
{
  wxASSERT(s_TypesDefined);
  for (int i = WHEELTYPE_FIRST; i < WHEELTYPE_LAST; i++) {
    if (name.CmpNoCase(cCarState::GetWheeltypeName(i)) == 0) return i;
  }

  return wxNOT_FOUND;
}

//-----------------------------------------------------------------------------
// Find the wheeltype belonging to a code

int cCarState::FindWheeltypeCode(const wxString& code)
{
  wxASSERT(s_TypesDefined);
  for (int i = WHEELTYPE_FIRST; i < WHEELTYPE_LAST; i++) {
    if (code.CmpNoCase(cCarState::GetWheeltypeCode(i)) == 0) return i;
  }

  return wxNOT_FOUND;
}

//-----------------------------------------------------------------------------
// Calculate the G forces and other derived data
// - prev, next = previous and next car state

void cCarState::CalculateDerivedData(const cCarState& prev, const cCarState& next)
{
  wxASSERT(next.m_TimeInLap - prev.m_TimeInLap > EPSILON);
  wxASSERT(fabs(1.0f - m_Fwd.Length()) < EPSILON);
  wxASSERT(fabs(1.0f - m_Right.Length()) < EPSILON);

  // calculate longitudinal acceleration (base value)
  cVector travel = (next.m_Pos - prev.m_Pos); // direction of car travel
  travel.Normalize();
  float glong = (next.m_Speed - prev.m_Speed) / (next.m_TimeInLap - prev.m_TimeInLap);

  // calculate lateral acceleration (base value)
  cVector center;             // center of turn circle
  cVector tocenter(0, 0, 0);  // direction of lateral force (= towards center)
  float glat = 0.0f;
  if (cVector::Center(prev.m_Pos, m_Pos, next.m_Pos, center)) {
    tocenter = center - m_Pos;
    m_Radius = tocenter.Length();
    wxASSERT(m_Radius > EPSILON);
    tocenter.Normalize();
    glat = m_Speed * m_Speed / m_Radius;
  }

  // calculate accelerations relative to position of car
  m_Glong = travel.Inner(m_Fwd) * glong + tocenter.Inner(m_Fwd) * glat;
  m_Glat = travel.Inner(m_Right) * glong + tocenter.Inner(m_Right) * glat;
  if (tocenter.Inner(m_Right) < 0) m_Radius = -m_Radius;

  // calculate slip angle
  if (((fabs(travel.x) > EPSILON) || (fabs(travel.y) > EPSILON)) &&
      ((fabs(m_Fwd.x) > EPSILON) || (fabs(m_Fwd.y) > EPSILON))) {
    m_SlipAngle = atan2(travel.y, travel.x) - atan2(m_Fwd.y, m_Fwd.x);
    if (m_SlipAngle < M_PI) m_SlipAngle += 2.0f * M_PI;
    if (m_SlipAngle > M_PI) m_SlipAngle -= 2.0f * M_PI;
  }

  // calculate vertical wheel speed
  for (size_t w = 0; w < 4; w++) {
    m_WheelSpeed[w] = (next.m_Deflect[w] - prev.m_Deflect[w]) / (next.m_TimeInLap - prev.m_TimeInLap);
  }
}
