#include "global.h"
#include <wx/window.h>
#include <wx/listctrl.h>
#include <wx/dcclient.h>

//-----------------------------------------------------------------------------
// Calculate the highest number with one significant digit, not greater than the given value

float FloorOneSD(float val)
{
  wxASSERT(val > 0); // not guaranteed to work for negative numbers

  float power = pow(10.0f, floor(log10(val))); // the next-lowest power of 10
  return val - fmod(val, power);
}

//-----------------------------------------------------------------------------
// Calculate the lowest number with one significant digit, not higher than the given value

float CeilOneSD(float val)
{
  wxASSERT(val > 0); // not guaranteed to work for negative numbers

  float power = pow(10.0f, floor(log10(val))); // the next-lowest power of 10
  return val - fmod(val, power) + power;
}

//-----------------------------------------------------------------------------
// Get the first significant digit

int FirstSD(float val)
{
  // convert to string in scientific notation, then take first char
  return wxString::Format("%e", fabs(val))[0] - '0';
}

//-----------------------------------------------------------------------------
// Format a time in mm:ss:ttt format
// - t = time in ms
// - shortformat = print without leading zeroes, and without minutes

wxString FormatTime(int t, bool shortformat)
{
  wxString result;

  if (shortformat && (t < 60000)) {
    result.Printf(_T("%d.%03d"), (t / 1000) % 60, t % 1000);
  }
  else {
    result.Printf(_T("%d:%02d.%03d"), t / 60000, (t / 1000) % 60, t % 1000);
  }

  if (t < 0) result = _T("-") + result;

  return result;
}

//-----------------------------------------------------------------------------
// Split a delimited list into its parts

void SplitStringList(const wxString& list, char separator, wxArrayString& parts)
{
  wxASSERT(parts.IsEmpty());

  wxString rest = list;
  while (!rest.IsEmpty()) {
    parts.Add(rest.BeforeFirst(separator));
    rest = rest.AfterFirst(separator);
  }
}

//-----------------------------------------------------------------------------
// Decode a string that is a list of key-value pairs

void DecodeKeyValueList(const wxString& list, wxArrayString& keys, wxArrayString& values)
{
  // split the list
  wxArrayString pairs;
  ::SplitStringList(list, _T(';'), pairs);

  // decode each pair
  for (size_t i = 0; i < pairs.GetCount(); i++) {
    wxString key = pairs[i].BeforeFirst(_T('='));
    key.MakeLower();
    key.Trim(true);
    key.Trim(false);
    keys.Add(key);

    wxString value = pairs[i].AfterFirst(_T('='));
    value.Trim(true);
    value.Trim(false);
    values.Add(value);
  }
}

//-----------------------------------------------------------------------------
// Set the width for all columns of a list control

void SetColumnWidths(wxListCtrl* list)
{
  // MS Windows sometimes gets the width of the last column wrong,
  // so we first add an extra, epmty column ...
  int count = list->GetColumnCount();
  list->InsertColumn(count, wxEmptyString);

  // ... then set the column widths ...
  for (int c = 0; c < count; c++) list->SetColumnWidth(c, wxLIST_AUTOSIZE_USEHEADER);

  // ... and then delete the extra column
  list->DeleteColumn(count);
}

//-----------------------------------------------------------------------------
// Convert LFS escape codes in a string to normal charaters

void UnEscapeLFS(wxString& string)
{
  // from http://en.lfsmanual.net/wiki/InSim_Tutorials#List_of_escape_codes
  string.Replace(_T("^v"), _T("|"));
  string.Replace(_T("^a"), _T("*"));
  string.Replace(_T("^c"), _T(":"));
  string.Replace(_T("^d"), _T("\\"));
  string.Replace(_T("^s"), _T("/"));
  string.Replace(_T("^q"), _T("?"));
  string.Replace(_T("^t"), _T("\""));
  string.Replace(_T("^l"), _T("<"));
  string.Replace(_T("^r"), _T(">"));
}

//-----------------------------------------------------------------------------
// Encode/Decode special characters

wxString Escape(const wxString& string, const wxString& special)
{
  wxString result = string;

  for (size_t i=0; i < special.Len(); i++) {
    result.Replace(special(i, 1), _T("\\") + special(i, 1));
  }

  return result;
}


wxString UnEscape(const wxString& string, const wxString& special)
{
  wxString result = string;

  for (size_t i=0; i < special.Len(); i++) {
    result.Replace(_T("\\") + special(i, 1), special(i, 1));
  }

  return result;
}

//-----------------------------------------------------------------------------
// Draw a string centered / right-aligned
// - dc = device context to draw in
// - text = string to draw
// - x = X coordinate of end of string
// - y = Y coordinate of top of string

void DrawTextCentered(wxDC& dc, const wxString& text, wxCoord x, wxCoord y)
{
  wxCoord sizeX, sizeY;
  dc.GetTextExtent(text, &sizeX, &sizeY);
  dc.DrawText(text, x - sizeX /2, y - sizeY / 2);
}


void DrawTextRightAligned(wxDC& dc, const wxString& text, wxCoord x, wxCoord y)
{
  wxCoord sizeX, sizeY;
  dc.GetTextExtent(text, &sizeX, &sizeY);
  dc.DrawText(text, x - sizeX, y);
}

//-----------------------------------------------------------------------------
// Draw a button outline
// - dc = device context to draw in
// - rect = rectangle for outline of button
// - highlight = draw it as a highlighted button (= mouse hovers over it)

void DrawButton(wxDC& dc, const wxRect& rect, bool highlight)
{
  dc.SetBrush(*wxTRANSPARENT_BRUSH);

  // draw 'base' of button in light grey
  dc.SetPen(wxPen(wxColour(LIGHT_GREY, LIGHT_GREY, LIGHT_GREY)));
  dc.DrawRectangle(rect.x, rect.y, rect.width, rect.height);

  // draw inner rectangle of button
  if (!highlight) dc.SetPen(*wxWHITE_PEN); // for highlighted buttons, draw in light grey
  dc.DrawRectangle(rect.x + 1, rect.y + 1, rect.width - 2, rect.height - 2);

  // draw outer rectangle in medium grey
  dc.SetPen(wxPen(wxColour(MEDIUM_GREY, MEDIUM_GREY, MEDIUM_GREY)));
  dc.DrawRoundedRectangle(rect.x, rect.y, rect.width, rect.height, 2);
}

//-----------------------------------------------------------------------------
// cObjChecker class (only for denbug build)

#ifdef __WXDEBUG__

int cObjChecker::s_ObjCount = 0;

cObjChecker::cObjChecker()
{
  s_ObjCount++;
  m_Marker = this + 1;
}


cObjChecker::~cObjChecker()
{
  m_Marker = NULL;
  s_ObjCount--;
  wxASSERT(s_ObjCount >= 0);
}

#endif
