#include "cproperties.h"
#include "cmgr.h"
#include <wx/dcclient.h>

//-----------------------------------------------------------------------------
// Property enumeration

enum ENUM_ROWS {
  ROW_PLAYERNAME = 0,
  ROW_CARNAME,
  ROW_LAPTIME,
  ROW_SPLITS,
  ROW_SECTORS,
  ROW_LFSVERSION,
  ROW_DRIVERAIDS,
  ROW_HLVC,
  ROW_BRAKES,
  ROW_RIDEHEIGHT,
  ROW_STIFFNESS,
  ROW_DAMPING_BUMP,
  ROW_DAMPING_REBOUND,
  ROW_ANTIROLL,
  ROW_GEARS,
  ROW_FINALDRIVE,
  ROW_TYRETYPE,
  ROW_TYRETEMP
};

//-----------------------------------------------------------------------------
// Event table

BEGIN_EVENT_TABLE(cProperties, cPane)
END_EVENT_TABLE()

//-----------------------------------------------------------------------------

cProperties::cProperties(wxWindow* parent)
: cPane(parent)
{
  m_List = new wxListView(this, -1, wxDefaultPosition, wxDefaultSize, wxLC_REPORT | wxNO_BORDER | wxLC_HRULES | wxLC_VRULES);

  m_TopSizer = new wxBoxSizer(wxVERTICAL);
  m_TopSizer->Add(m_List, wxSizerFlags(1).Expand().Proportion(1));
  SetSizer(m_TopSizer);

  // calculate font height
  wxClientDC dc(this);
  dc.SetFont(*wxNORMAL_FONT);
  wxCoord sizeX;
  dc.GetTextExtent(_T("Wg"), &sizeX, &m_FontHeight);

  // create rows for the properties
  m_List->InsertColumn(0, wxEmptyString);
  m_List->InsertItem(ROW_PLAYERNAME, _T("Player"));
  m_List->InsertItem(ROW_CARNAME, _T("Car"));
  m_List->InsertItem(ROW_LAPTIME, _T("Lap time"));
  m_List->InsertItem(ROW_SPLITS, _T("Split times"));
  m_List->InsertItem(ROW_SECTORS, _T("Sector times"));
  m_List->InsertItem(ROW_LFSVERSION, _T("LFS version"));
  m_List->InsertItem(ROW_DRIVERAIDS, _T("Driver aids"));
  m_List->InsertItem(ROW_HLVC, _T("HLVC"));

  m_List->InsertItem(ROW_BRAKES, _T("Brakes"));
  m_List->InsertItem(ROW_RIDEHEIGHT, _T("Ride height"));
  m_List->InsertItem(ROW_STIFFNESS, _T("Stiffness"));
  m_List->InsertItem(ROW_DAMPING_BUMP, _T("Bump damping"));
  m_List->InsertItem(ROW_DAMPING_REBOUND, _T("Rebound damping"));
  m_List->InsertItem(ROW_ANTIROLL, _T("Anti roll"));
  m_List->InsertItem(ROW_GEARS, _T("Gears"));
  m_List->InsertItem(ROW_FINALDRIVE, _T("Final drive"));
  m_List->InsertItem(ROW_TYRETYPE, _T("Tyre type"));
  m_List->InsertItem(ROW_TYRETEMP, _T("Tyre temperature (avg)"));

  ::SetColumnWidths(m_List);

  RedirectEvents(m_List, this, false, true); // redirect escape-key events to parent
  RedirectEvents(m_List, NULL, true, false); // redirect mouse-wheel events to window-at-pointer
}

//-----------------------------------------------------------------------------

cProperties::~cProperties()
{
}

//-----------------------------------------------------------------------------
// Re-read the properties from all the laps

void cProperties::UpdateAll()
{
  // delete exisiting columns
  for (int c = m_List->GetColumnCount() - 1; c > 0; c--) m_List->DeleteColumn(c);

  // re-create a column for each lap
  for (size_t l = 0; l < MGR->GetLapCount(); l++) {
    cLap* lap = MGR->GetLap(l);

    size_t col = l + 1;
    m_List->InsertColumn(col, lap->GetName());

    wxString buf;

    // general info
    m_List->SetItem(ROW_PLAYERNAME, col, lap->m_PlayerName);
    buf = lap->m_Car.m_Name + _T(" (") + cCar::Name2Code(lap->m_Car.m_Name) + _T(")");
    m_List->SetItem(ROW_CARNAME, col, buf);
    m_List->SetItem(ROW_LFSVERSION, col, lap->m_LfsVersion);
    m_List->SetItem(ROW_LAPTIME, col, ::FormatTime(lap->m_Split[lap->m_SplitCount - 1]));

    // split times
    buf = ::FormatTime(lap->m_Split[0]);
    for (int s = 1; s < lap->m_SplitCount - 1; s++) {
      buf += _T(", ") + ::FormatTime(lap->m_Split[s]);
    }
    m_List->SetItem(ROW_SPLITS, col, buf);

    // sector times
    buf = ::FormatTime(lap->m_Split[0]);
    for (int s = 1; s < lap->m_SplitCount; s++) {
      buf += _T(", ") + ::FormatTime(lap->m_Split[s] - lap->m_Split[s-1]);
    }
    m_List->SetItem(ROW_SECTORS, col, buf);

    // driver aids
    buf = wxEmptyString;
    if (lap->m_PlayerFlags & 2)   buf += _T("Gear cut, ");
    if (lap->m_PlayerFlags & 4)   buf += _T("Gear blip, ");
    if (lap->m_PlayerFlags & 8)   buf += _T("Auto shift, ");
    if (lap->m_PlayerFlags & 16)  buf += _T("Shifter, ");
    if (lap->m_PlayerFlags & 64)  buf += _T("Brake help, ");
    if (lap->m_PlayerFlags & 128) buf += _T("Axis clutch, ");
    buf = buf.BeforeLast(',');
    m_List->SetItem(ROW_DRIVERAIDS, col, buf);

    // HLVC
    switch (lap->m_Hlvc) {
      case 1 : buf = _T("Legal"); break;
      case 2 : buf = _T("Illegal"); break;
      default : buf = _T("Unknown"); break;
    }
    m_List->SetItem(ROW_HLVC, col, buf);

    // car setup data
    float frontBrake = lap->m_Car.m_Brake[0];
    float totalBrake = frontBrake + lap->m_Car.m_Brake[2];
    buf.Printf(_T("%.0f, %.0f%% front"), totalBrake / 2, frontBrake * 100 / totalBrake);
    m_List->SetItem(ROW_BRAKES, col, buf);

    buf.Printf(_T("%.3f rear, %.3f front"),
        lap->m_Car.m_MaxDeflect[2],
        lap->m_Car.m_MaxDeflect[0]);
    m_List->SetItem(ROW_RIDEHEIGHT, col, buf);

    buf.Printf(_T("%.1f rear, %.1f front"),
        lap->m_Car.m_Spring[2] / 1000,
        lap->m_Car.m_Spring[0] / 1000);
    m_List->SetItem(ROW_STIFFNESS, col, buf);

    buf.Printf(_T("%.1f rear, %.1f front"),
        lap->m_Car.m_Damp_Bump[2] / 1000,
        lap->m_Car.m_Damp_Bump[0] / 1000);
    m_List->SetItem(ROW_DAMPING_BUMP, col, buf);

    buf.Printf(_T("%.1f rear, %.1f front"),
        lap->m_Car.m_Damp_Rebound[2] / 1000,
        lap->m_Car.m_Damp_Rebound[0] / 1000);
    m_List->SetItem(ROW_DAMPING_REBOUND, col, buf);

    buf.Printf(_T("%.1f rear, %.1f front"),
        lap->m_Car.m_AntiRoll_Rear / 1000,
        lap->m_Car.m_AntiRoll_Front / 1000);
    m_List->SetItem(ROW_ANTIROLL, col, buf);

    buf = wxEmptyString;
    for (int g = 0; g < lap->m_Car.m_GearCount; g++) {
      if (g > 0) buf += _T(", ");
      if (g < 6) {
        buf += wxString::Format(_T("%.2f"), lap->m_Car.m_GearRatio[g]);
      }
      else {
        buf += _T("?");
      }
    }
    m_List->SetItem(ROW_GEARS, col, buf);

    buf.Printf(_T("%.2f"), lap->m_Car.m_FinalDrive);
    m_List->SetItem(ROW_FINALDRIVE, col, buf);

    if (lap->m_Car.m_TyreType[2] == lap->m_Car.m_TyreType[0]) {
      // same tyre types on front and rear
      buf.Printf(_T("%s"), cCar::s_TyreTypeNames[lap->m_Car.m_TyreType[2]].c_str());
    }
    else {
      // different tyre types
      buf.Printf(_T("%s rear, %s front"),
          cCar::s_TyreTypeNames[lap->m_Car.m_TyreType[2]].c_str(),
          cCar::s_TyreTypeNames[lap->m_Car.m_TyreType[0]].c_str());
    }
    m_List->SetItem(ROW_TYRETYPE, col, buf);

    buf.Printf(_T("%.1f | %.1f rear, %.1f | %.1f front"),
        lap->m_AvgTemp[2], lap->m_AvgTemp[3],
        lap->m_AvgTemp[0], lap->m_AvgTemp[1]);
    m_List->SetItem(ROW_TYRETEMP, col, buf);
  }

  ::SetColumnWidths(m_List);
}

//-----------------------------------------------------------------------------

wxSize cProperties::DoGetBestSize() const
{
  // calculate total width
  int width = 0;
  for (int c = 0; c < m_List->GetColumnCount(); c++) width += m_List->GetColumnWidth(c);
  if (m_List->GetColumnCount() == 1) width *= 2;
  width += m_List->GetColumnCount() * 2;

  // calculate total height
  int height = (m_List->GetItemCount() + 1) * (m_FontHeight + 2);

  return wxSize(width, height);
}
