#include "cgraphview.h"
#include "cgraph_y.h"
#include "cgraphtools_y.h"
#include "cgraph_xy.h"
#include "cgraphtools_xy.h"
#include "cgraph_h.h"
#include "cgraphtools_h.h"
#include "cruler.h"
#include "cmgr.h"

//-----------------------------------------------------------------------------
// Event table

BEGIN_EVENT_TABLE(cGraphView, cPane)
  EVT_SCROLLWIN(cGraphView::OnScroll)
  EVT_SIZE(cGraphView::OnSize)
END_EVENT_TABLE()

//-----------------------------------------------------------------------------

cGraphView::cGraphView(wxWindow* parent, graph_t type, wxString settings)
: cPane(parent)
{
  m_Type = type;

  SetBackgroundColour(*wxWHITE);

  switch (m_Type) {
    case GRAPHTYPE_Y :
      m_Graph = new cGraphY(this);
      m_Toolbar = new cGraphToolsY(this, (cGraphY*)m_Graph);
      break;

    case GRAPHTYPE_XY :
      m_Graph = new cGraphXY(this);
      m_Toolbar = new cGraphToolsXY(this, (cGraphXY*)m_Graph);
    break;

    case GRAPHTYPE_H :
      m_Graph = new cGraphH(this);
      m_Toolbar = new cGraphToolsH(this, (cGraphH*)m_Graph);
    break;

    default :
      wxFAIL;
      m_Graph = NULL;
      m_Toolbar = NULL;
      m_RulerX = NULL;
      m_RulerY = NULL;
      return;
  }
  wxASSERT(m_Graph->GetType() == m_Type);
  m_RulerX = new cRuler(this, wxBOTTOM);
  m_RulerY = new cRuler(this, wxLEFT);
  m_Graph->SetRulerX(m_RulerX);
  m_Graph->SetRulerY(m_RulerY);

  RedirectEvents(m_Graph, NULL, true, false); // redirect scrollwheel events to the window at the pointer
  // graph gets all mouse-wheel and escape-key events
  RedirectEvents(this, m_Graph, true, true);
  RedirectEvents(m_RulerX, m_Graph, true, true);
  RedirectEvents(m_RulerY, m_Graph, true, true);
  RedirectEvents(m_Toolbar, m_Graph, true, true);

  // populate sizers
  m_GrrSizer = new wxFlexGridSizer(2, 2, 0, 0);
  m_GrrSizer->Add(m_RulerY, wxSizerFlags(1).Expand().Border(wxRIGHT, 2));
  m_GrrSizer->Add(m_Graph, wxSizerFlags(1).Expand());
  m_GrrSizer->AddStretchSpacer(1); // empty slot, for bottom-left corner
  m_GrrSizer->Add(m_RulerX, wxSizerFlags(1).Expand().Border(wxTOP, 2));
  m_GrrSizer->AddGrowableCol(1, 1);
  m_GrrSizer->AddGrowableRow(0, 1);

  m_TopSizer = new wxBoxSizer(wxVERTICAL);
  m_TopSizer->Add(m_Toolbar, wxSizerFlags(1).Expand().Proportion(0));
  m_TopSizer->Add(m_GrrSizer, wxSizerFlags(1).Expand().Border(wxLEFT | wxRIGHT | wxBOTTOM, BASE_MARGIN));

  SetSizer(m_TopSizer);

  // pass the graph settings
  if (!settings.IsEmpty()) LoadSettings(settings);

  // add the currently loaded laps
  for (size_t index = 0; index < MGR->GetLapCount(); index++) {
    AddLap(MGR->GetLap(index));
  }
}

//-----------------------------------------------------------------------------

cGraphView::~cGraphView()
{
}

//-----------------------------------------------------------------------------

void cGraphView::UpdateAll()
{
  CHECK_THIS;
  CHECK_PTR(m_Graph);

  m_Graph->UpdateAll(); // will also refresh the rulers
  m_Toolbar->UpdateAll();
}

//-----------------------------------------------------------------------------

void cGraphView::AddLap(cLap* lap)
{
  CHECK_THIS;
  CHECK_PTR(m_Graph);
  m_Graph->AddCurve(lap);
}

//-----------------------------------------------------------------------------

void cGraphView::DeleteLap(cLap* lap)
{
  CHECK_THIS;
  CHECK_PTR(m_Graph);
  m_Graph->DeleteCurve(lap);
}

//-----------------------------------------------------------------------------

void cGraphView::DeleteAllLaps()
{
  CHECK_THIS;
  CHECK_PTR(m_Graph);
  m_Graph->DeleteAllCurves();
}

//-----------------------------------------------------------------------------

void cGraphView::OnScroll(wxScrollWinEvent& event)
{
  // remove crosshair before parent class performs scrolling
  m_Graph->RemoveCrossHair();

  // synchronize graphs after scrolling
  if (event.GetOrientation() == wxHORIZONTAL) RequestSync();

  event.Skip(); // let parent class do scrolling
}

//-----------------------------------------------------------------------------

void cGraphView::ShowXruler(bool show)
{
  if (m_Type != GRAPHTYPE_Y) return; // only valid for Y graphs
  if (show == m_RulerX->IsShown()) return; // no change

  m_RulerX->Show(show);
  m_GrrSizer->Layout();
  Refresh();
}

//-----------------------------------------------------------------------------
// Tell the graph view what its position is

void cGraphView::GraphPositionIs(bool top, bool bottom)
{
  m_Graph->GraphPositionIs(top, bottom);
  m_Toolbar->GraphPositionIs(top, bottom);
}

//-----------------------------------------------------------------------------

void cGraphView::OnSize(wxSizeEvent& WXUNUSED(event))
{
  FitInside();
}

//-----------------------------------------------------------------------------
// Loading and saving the configuration settings

wxString cGraphView::SaveSettings() const
{
  CHECK_THIS;
  CHECK_PTR(m_Graph);
  return m_Graph->SaveSettings();
}

void cGraphView::LoadSettings(const wxString& settings)
{
  m_Graph->LoadSettings(settings);
  m_Toolbar->UpdateAll();
}

//-----------------------------------------------------------------------------
// Enable or disable the tooltips

void cGraphView::EnableToolTips(bool flag)
{
  m_Graph->EnableToolTips(flag);
  m_RulerX->EnableToolTips(flag);
  m_RulerY->EnableToolTips(flag);
}
