#ifndef _CGRAPH_H_
#define _CGRAPH_H_

#include "global.h"
#include "ccurve.h"
#include <wx/window.h>
#include <wx/scrolwin.h>
#include <wx/pen.h>
#include <wx/menu.h>
#include <wx/choice.h>
#include <wx/tooltip.h>

class cRuler;

//-----------------------------------------------------------------------------

typedef enum {
  GRAPHTYPE_FIRST = 0,
  GRAPHTYPE_NONE = GRAPHTYPE_FIRST,
  GRAPHTYPE_Y,   // normal graph
  GRAPHTYPE_XY,  // XY plot
  GRAPHTYPE_H,   // histogram
  GRAPHTYPE_LAST
} graph_t;

//-----------------------------------------------------------------------------
// modes for drawing a rubberband box (for zooming in on a part of the graph)

typedef enum {
  RBBMODE_NONE = 0, // off, or not allowed
  RBBMODE_PREPARE,  // ready to start rbb-dragging (mouse click detected)
  RBBMODE_XONLY,    // only select in the horizontal direction
  RBBMODE_BOTH      // select in both X and Y direction
} rbbmode_t;

//-----------------------------------------------------------------------------
// A graph

class cGraph : public wxWindow
{
private:
  wxMenu m_Context;               // context menu
  wxToolTip* m_ToolTip;           // the window's tooltip
  bool m_ToolTips;                // enable tooltips?
  bool m_NeedToSync;              // do we need to sync other graphs?

  // event handlers
  void OnKey(wxKeyEvent& event);
  void OnIdle(wxIdleEvent& event);
  void OnMenuClick(wxCommandEvent& event);
  void OnMouseLeftClick(wxMouseEvent& event);
  void OnMouseLeftRelease(wxMouseEvent& event);
  void OnMouseLeftDclick(wxMouseEvent& event);
  void OnMouseRightClick(wxMouseEvent& event);
  void OnMouseMiddleClick(wxMouseEvent& event);
  void OnMouseMove(wxMouseEvent& event);
  void OnMouseWheel(wxMouseEvent& event);
  void OnSize(wxSizeEvent& event);

  // misc
  static void CalcStep(float range, int size, bool integer, float& grid, int& tick);

  DECLARE_EVENT_TABLE()

  void AdjustDimensions();
  void AdjustBbox();

protected:
  cGraphView* m_Parent;           // the parent window
  graph_t m_Type;                 // the type of graph
  static wxPen s_GridPen;         // the pen for drawing grid lines
  static wxPen s_AxisPen;         // the pen for drawing the Y-axis of the grid

  bool m_Connect;                 // connect the dots?
  cRuler* m_RulerX;               // the ruler for the X axis
  cRuler* m_RulerY;               // the ruler for the Y axis
  cCurveArray m_Curves;           // the curves

  int FindCurve(cLap* lap) const;

  float m_SelStart;               // distance-in-lap where selected part of lap starts
  float m_SelEnd;                 // distance-in-lap where selected part of lap end
  float m_CursorDist;             // distance-in-lap for the position of the track cursor (< 0 = no cursor)

  // graph dimensioning
  float m_MinX, m_MaxX, m_MinY, m_MaxY; // global bounding box (inclusive)
  float m_ScaleX, m_ScaleY;       // scale factor (translates curve values to pixels)
  float m_ZoomX, m_ZoomY;         // user's zoom factor
  float m_GridStepX, m_GridStepY; // step (= value difference) between the grid lines
  int m_RulerTickX, m_RulerTickY; // number of tick marks on the ruler between grid lines

  int m_FitDimensions;            // the dimension (wxHORIZONTAL and/or wxVERTICAL) on which FitSelection() works

  bool SetZoomX(float zoom);
  bool SetZoomY(float zoom);

  virtual float GetSelectionStart() = 0;
  virtual float GetSelectionEnd() = 0;

  // rubberband-box drawing
  rbbmode_t m_RbbMode;            // current rbb-drawing mode?
  wxPoint m_RbbOrigin;            // starting point of the box (the "fixed" corner)
  wxPoint m_RbbDestination;       // the "moving" corner

  void DrawRubberbandBox();

  // drawing the crosshair
  bool m_CrossHairMode;           // active crosshair drawing?
  bool m_CrossHairVisible;        // crosshair currently drawn?
  wxPoint m_CrossHairPos;         // position of current crosshair

  void DrawTrackCursor(bool scroll);
  void RemoveTrackCursor();
  void ClearToolTip() { m_ToolTip->SetTip(wxEmptyString); }

  wxString FormatValueX(float val);
  wxString FormatValueY(float val);

  // callbacks
  virtual rbbmode_t GetRbbMode(wxMouseEvent& WXUNUSED(event)) { return RBBMODE_NONE; }
  virtual void OnAdjustBbox() {}  // for post-processing of changes on the bounding box
  virtual void OnMouseClickRelease(wxPoint WXUNUSED(pos)) {} // for click+release without dragging

public:
  ENABLE_OBJCHECKER

  cGraph(cGraphView* parent, graph_t type);
  virtual ~cGraph();

  graph_t GetType() const { CHECK_THIS; return m_Type; }

  // get borders of bounding box
  float GetMinX() { return m_MinX; }
  float GetMaxX() { return m_MaxX; }
  float GetMinY() { return m_MinY; }
  float GetMaxY() { return m_MaxY; }

  void SetRulerX(cRuler* ruler);
  void SetRulerY(cRuler* ruler);

  bool GetConnect() const { return m_Connect; }
  void SetConnect(bool connect);

  void DrawCrossHair(wxPoint pos);
  void DrawCrossHairAtPointer();
  void RemoveCrossHair();

  void RequestSync() { m_NeedToSync = true; }
  virtual void DoSync() {}

  // manage the set of curves
  inline size_t GetCurveCount() const { return m_Curves.GetCount(); }
  inline cCurve* GetCurve(size_t index) { return &(m_Curves[index]); }
  virtual void AddCurve(cLap* lap) = 0;
  virtual void DeleteCurve(cLap* lap) = 0;
  virtual void DeleteAllCurves() = 0;

  // help methods for use in drawing
  virtual wxCoord Value2CoordX(float value);
  virtual wxCoord Value2CoordY(float value);
  virtual float Coord2ValueX(wxCoord coord);
  virtual float Coord2ValueY(wxCoord coord);
  virtual float cGraph::GetGraphOriginX();
  virtual float cGraph::GetGraphOriginY();
  void SetGraphOriginX(float value);
  void SetGraphOriginY(float value);
  virtual bool IsIntegerX() = 0;
  virtual bool IsIntegerY() = 0;

  // interface with rulers
  float GetGridStartX() { return Coord2ValueX(0); }
  float GetGridEndX() { return Coord2ValueX(GetClientSize().GetWidth()); }
  float GetGridStepX() { return m_GridStepX; }
  int GetRulerTicksX() { return m_RulerTickX; }
  float GetGridEndY() { return Coord2ValueY(0); }
  float GetGridStepY() { return m_GridStepY; }
  int GetRulerTicksY() { return m_RulerTickY; }

  void GraphPositionIs(bool top, bool bottom);

  float GetZoomX() const { return m_ZoomX; }
  float GetZoomY() const { return m_ZoomY; }
  void ZoomX(float zoom);
  void ZoomY(float zoom);
  void ZoomReset(int dimension = (wxHORIZONTAL | wxVERTICAL));
  void FitSelection();

  void ShowCrossHair(bool show);

  void UpdateAll();

  virtual void DoSetTrackCursorPos(cGraphView* view, float distance, bool scroll) = 0;
  virtual void DoSetTrackSelection(float start, float end) = 0;
  virtual void DoSynchronise(cGraph* WXUNUSED(graph)) {}

  virtual wxString GetCurveValueAtPointer(cLap* WXUNUSED(lap)) { return wxEmptyString; }

  virtual wxString SaveSettings() const = 0;
  virtual void LoadSettings(const wxString& settings) = 0;

  void EnableToolTips(bool flag);
  virtual wxString GetUnitsX() = 0;
  virtual wxString GetUnitsY() = 0;
};

#endif
