#include "cvector.h"

//-----------------------------------------------------------------------------

cVector::cVector()
: x(0.0f), y(0.0f), z(0.0f)
{
}

cVector::cVector(float initX, float initY, float initZ)
: x(initX), y(initY), z(initZ)
{
}

cVector::cVector(const cVector& other)
: x(other.x), y(other.y), z(other.z)
{
}

//-----------------------------------------------------------------------------

cVector::~cVector()
{
}

//-----------------------------------------------------------------------------
// Vector addition, subtraction and multiplication

cVector cVector::operator+ (const cVector& other) const
{
  return cVector(x + other.x, y + other.y, z + other.z);
}

cVector cVector::operator- (const cVector& other) const
{
  return cVector(x - other.x, y - other.y, z - other.z);
}

cVector cVector::operator* (float factor) const
{
  return cVector(x * factor, y * factor, z * factor);
}

cVector cVector::operator/ (float factor) const
{
  return cVector(x / factor, y / factor, z / factor);
}

//-----------------------------------------------------------------------------
// Scale the vector to length = 1

void cVector::Normalize()
{
  float len = Length();
  if (len < EPSILON) return;

  x = x / len;
  y = y / len;
  z = z / len;
}

//-----------------------------------------------------------------------------
// Inner and outer vector products

float cVector::Inner(const cVector& other) const
{
  return (x * other.x) + (y * other.y) + (z * other.z);
}

cVector cVector::Outer(const cVector& other) const
{
  return cVector(
      y * other.z - z * other.y,
      z * other.x - x * other.z,
      x * other.y - y * other.x);
}

//-----------------------------------------------------------------------------
// Calculate the center of the circle, given 3 points at its perimeter
// Returns false if calculation not possible (a, b, and c are on a line), otherwise true

bool cVector::Center(const cVector& a, const cVector& b, const cVector& c, cVector& center)
{
  // formula given in http://en.wikipedia.org/wiki/Circle#Calculating_the_parameters_of_a_circle

  float den = 2 * (a - b).Outer(b - c).SquareLength();
  if (den < EPSILON) return false;

  float alpha = (b - c).SquareLength() * ((a - b).Inner(a - c)) / den;
  float beta = (a - c).SquareLength() * ((b - a).Inner(b - c)) / den;
  float gamma = (a - b).SquareLength() * ((c - a).Inner(c - b)) / den;

  center = (a * alpha) + (b * beta) + (c * gamma);

  return true;
}
