#include "ctrackmap.h"
#include "cmgr.h"
#include "clap.h"
#include <wx/dcclient.h>
#include <wx/cursor.h>
#include <wx/config.h>

// max. number of line segments that the track is plotted in
#define MAX_SEGMENTS 1000

// margin between map and border of window
#define MAP_MARGIN 8

// minimum length of a drag-selected part of the track
#define MIN_SELECTION_LENGTH 25

// distance (in meters) for scrolling with the mouse wheel
#define SCROLL_DIST 50

//-----------------------------------------------------------------------------
// Menu identifiers

enum ENUM_MENU_IDS
{
  ID_MENU_RESETZOOM = 100,
  ID_MENU_SHOWDISTANCE,
  ID_MENU_SHOWSECTORBUTTONS,
  ID_MENU_LAST
};

//-----------------------------------------------------------------------------
// Event table

BEGIN_EVENT_TABLE(cTrackMap, cPane)
  EVT_LEFT_DCLICK(cTrackMap::OnMouseLeftDoubleClick)
  EVT_LEFT_DOWN(cTrackMap::OnMouseLeftClick)
  EVT_LEFT_UP(cTrackMap::OnMouseLeftRelease)
  EVT_MIDDLE_DOWN(cTrackMap::OnMouseMiddleClick)
  EVT_RIGHT_DOWN(cTrackMap::OnMouseRightClick)
  EVT_MENU(wxID_ANY, cTrackMap::OnMenuClick)
  EVT_MOTION(cTrackMap::OnMouseMove)
  EVT_MOUSEWHEEL(cTrackMap::OnMouseWheel)
  EVT_ENTER_WINDOW(cTrackMap::OnMouseEntering)
  EVT_LEAVE_WINDOW(cTrackMap::OnMouseLeaving)
  EVT_PAINT(cTrackMap::OnPaint)
  EVT_SIZE(cTrackMap::OnSize)
END_EVENT_TABLE()

//-----------------------------------------------------------------------------
// Static members

wxPen cTrackMap::s_Pen_Sel(wxColour(MEDIUM_GREY, MEDIUM_GREY, MEDIUM_GREY), 2);
wxPen cTrackMap::s_Pen_UnSel(wxColour(LIGHT_GREY, LIGHT_GREY, LIGHT_GREY), 2);
wxPen cTrackMap::s_ButtonPen(wxColour(LIGHT_GREY, LIGHT_GREY, LIGHT_GREY));

//-----------------------------------------------------------------------------

cTrackMap::cTrackMap(wxWindow* parent)
: cPane(parent)
{
  m_DragMode = false;
  m_ClickPos = -1;
  m_CursorDrawn = false;

  m_ShowSectorToolbar = true;
  m_SelectedButton = -1;

  m_CursorDist = -1;
  m_ShowDistance = false;

  SetBackgroundColour(*wxWHITE);

  m_Context.Append(ID_MENU_RESETZOOM, _T("Reset zoom"));
  m_Context.AppendSeparator();
  m_Context.AppendCheckItem(ID_MENU_SHOWSECTORBUTTONS, _T("Show sector buttons"));
  m_Context.AppendCheckItem(ID_MENU_SHOWDISTANCE, _T("Show cursor distance"));

  AdjustDimensions();
}

//-----------------------------------------------------------------------------

cTrackMap::~cTrackMap()
{
}

//-----------------------------------------------------------------------------
// Re-calculate the dimensioning

void cTrackMap::AdjustDimensions()
{
  wxCoord sizeX, sizeY;
  wxClientDC dc(this);
  dc.SetFont(*wxNORMAL_FONT);

  int width, height; // room available for the track map itself
  GetClientSize(&width, &height);

  dc.GetTextExtent(LARGEST_VALUE_TEXT, &sizeX, &sizeY);
  m_ValueStartY = height - sizeY;
  if (m_ShowDistance) height -= sizeY + BASE_MARGIN; // room for distance value at the bottom

  if (m_ShowSectorToolbar) {
    dc.GetTextExtent(_T("99"), &sizeX, &sizeY); // max size of button labels
    ::Maximise(sizeX, sizeY);                   // make square buttons
    m_ButtonSize = sizeX + 8;                   // 4 pixels at each side, for padding and button border

    width -= m_ButtonSize + BASE_MARGIN;        // room for sector buttons at the right
    m_ButtonStart.x = width;
    m_ButtonStart.y = BASE_MARGIN;
  }

  width -= 2 * MAP_MARGIN;
  height -= 2 * MAP_MARGIN;

  if (MGR->IsEmpty()) {
    // no laps loaded
    m_MinX = 0.0f;
    m_MaxY = 1.0f;
    m_Scale = 1.0f;
    m_MarginX = MAP_MARGIN;
    m_MarginY = MAP_MARGIN;
    m_SelStart = 0.0f;
    m_SelEnd = 0.0f;
    return;
  }

  cLap* lap = MGR->GetLap(0); // lap to plot the trajectory of
  wxASSERT(lap->GetStateCount() > 0);
  m_MinX = lap->GetMinX();
  m_MaxY = lap->GetMaxY();
  float rangeX = lap->GetMaxX() - m_MinX;
  float rangeY = m_MaxY - lap->GetMinY();
  float scaleX = width / rangeX;
  float scaleY = height / rangeY;
  m_Scale = (scaleX < scaleY) ? scaleX : scaleY;
  m_MarginX = MAP_MARGIN + (width - (wxCoord)(rangeX * m_Scale)) / 2;
  m_MarginY = MAP_MARGIN + (height - (wxCoord)(rangeY * m_Scale)) / 2;
  wxLogDebug(_T("m_MarginY = %d"), m_MarginY);
}

//-----------------------------------------------------------------------------
// Plot the trackmap

void cTrackMap::OnPaint(wxPaintEvent& WXUNUSED(event))
{
  wxPaintDC dc(this);
  dc.SetFont(*wxNORMAL_FONT);

  if (MGR->IsEmpty()) {
    // empty track map
    m_SelStart = IMPOSSIBLY_LOW_VALUE;
    m_SelEnd = IMPOSSIBLY_HIGH_VALUE;
    m_ClickPos = -1;
    return;
  }

  DoPaint(dc);

  if (m_ShowSectorToolbar) {
    // draw the sector buttons
    dc.SetPen(s_ButtonPen);
    dc.SetBrush(*wxTRANSPARENT_BRUSH);

    for (int i = 0; i <= m_SectorCount; i++) {
      DrawSectorButton(dc, i, (i == m_SelectedButton));

      wxString label = _T("A"); // label for "all track" button
      if (i > 0) label.Printf(_T("%d"), i);

      wxCoord x = m_ButtonStart.x;
      wxCoord y = Button2CoordY(i);
      ::DrawTextCentered(dc, label, x + m_ButtonSize / 2 + 1, y + m_ButtonSize / 2);
    }
  }

  // re-plot the cursor
  if ((m_CursorDrawn) && IsExposed(m_CursorX, m_CursorY)) {
    m_CursorDrawn = false;
    PutTrackCursor(dc, m_CursorDist);
  }
}

void cTrackMap::DoPaint(wxDC& dc)
{
  if (MGR->IsEmpty()) return;

  dc.SetLogicalFunction(wxCOPY);

  cLap* lap = MGR->GetLap(0); // lap to plot the trajectory of
  wxASSERT(lap->GetStateCount() > 0);

  // get screen coordinates of first state
  const cCarState* state = lap->GetState(0);
  wxCoord lastX = Track2CoordX(state->GetPosX());
  wxCoord lastY = Track2CoordY(state->GetPosY());

  // plot a marker for the start/finish line
  dc.SetPen(s_Pen_Sel);
  dc.SetBrush(wxBrush(dc.GetPen().GetColour()));
  dc.DrawCircle(lastX, lastY, 3);

  // plot the lap's trajectory
  size_t step = 1 + (lap->GetStateCount() / MAX_SEGMENTS);
  for (size_t s = step; s < lap->GetStateCount(); s += step) {
    state = lap->GetState(s);
    wxCoord stateX = Track2CoordX(state->GetPosX());
    wxCoord stateY = Track2CoordY(state->GetPosY());

    if ((state->GetDistance() < m_SelStart) || (state->GetDistance() > m_SelEnd)) {
      dc.SetPen(s_Pen_UnSel);
    }
    else {
      dc.SetPen(s_Pen_Sel);
    }
    dc.DrawLine(lastX, lastY, stateX, stateY);

    lastX = stateX;
    lastY = stateY;
  }

  // plot the distance value
  if (m_ShowDistance && (m_CursorDist >= 0.0f)) {
    dc.SetFont(*wxNORMAL_FONT);
    wxString value;
    value.Printf(_T("%.0f"), m_CursorDist);
    DrawTextRightAligned(dc, value, GetClientSize().GetWidth() - BASE_MARGIN, m_ValueStartY);
  }
}

//-----------------------------------------------------------------------------
// Convert track coordinates to window coordinates

wxCoord cTrackMap::Track2CoordX(float tc)
{
  return m_MarginX + (wxCoord)((tc - m_MinX) * m_Scale);
}

wxCoord cTrackMap::Track2CoordY(float tc)
{
  return m_MarginY + (wxCoord)((m_MaxY - tc) * m_Scale);
}

//-----------------------------------------------------------------------------
// Place the cursor at a new position
// - distance = distance_in_lap (in m) that corresponds to the cursor position
// - scroll = scroll if needed to get the cursor in the visible part
// If distance < 0 then cursor is removed

void cTrackMap::DoSetTrackCursorPos(cGraphView* WXUNUSED(view), float distance, bool WXUNUSED(scroll))
{
  if (MGR->IsEmpty()) return; // empty display

  // draw the cursor at its new position
  wxClientDC dc(this);
  RemoveTrackCursor(dc);
  PutTrackCursor(dc, distance);

  // update value
  if (m_ShowDistance) {
    RefreshRect(wxRect(0, m_ValueStartY, GetClientSize().GetWidth(), GetClientSize().GetHeight()));
    Update(); // repaint immediately
  }
}

//-----------------------------------------------------------------------------
// Draw the cursor at a new position
// - dc = device context to draw in
// - distance = distance-in-lap for cursor position

void cTrackMap::PutTrackCursor(wxDC& dc, float distance)
{
  wxASSERT(!m_CursorDrawn);

  m_CursorDist = distance;
  if (distance < 0) return;

  cLap* lap = MGR->GetLap(0);
  float x, y;
  lap->GetPositionAt(distance, x, y);

  m_CursorX = Track2CoordX(x);
  m_CursorY = Track2CoordY(y);
  DrawTrackCursor(dc, m_CursorX, m_CursorY);
  m_CursorDrawn = true;
}

//-----------------------------------------------------------------------------
// Remove the cursor from its current position

void cTrackMap::RemoveTrackCursor(wxDC& dc)
{
  if (!m_CursorDrawn) return;
  DrawTrackCursor(dc, m_CursorX, m_CursorY);
  m_CursorDrawn = false;
}

//-----------------------------------------------------------------------------
// Actually draw the cursor
// - dc = device context to draw in
// - x, y = coordinates of the cursor position

void cTrackMap::DrawTrackCursor(wxDC& dc, wxCoord x, wxCoord y)
{
  dc.SetPen(*wxWHITE_PEN);
  dc.SetBrush(*wxWHITE_BRUSH);
  dc.SetLogicalFunction(wxINVERT);
  dc.DrawCircle(x, y, 3);
}

//-----------------------------------------------------------------------------

void cTrackMap::OnSize(wxSizeEvent& event)
{
  AdjustDimensions();
  Refresh(); // force full redraw when window is resized

  event.Skip();
}

//-----------------------------------------------------------------------------

void cTrackMap::OnMouseLeftClick(wxMouseEvent& event)
{
  event.Skip(); // always pass left-clicks to default event handler
  if (MGR->IsEmpty()) return; // empty track map

  int button = -1;
  m_ClickPos = DecodeMouse(event, &button);

  if (button < 0) return; // click on the trajectory is handled at release

  // handle click on sector button
  if (button == 0) {
    // first button = select whole track
    m_SelStart = 0.0f;
    m_SelEnd = MGR->GetTrackLength();
  }
  else {
    // second button = first sector, third button = second sector, etc.
    wxASSERT(button <= m_SectorCount);
    m_SelStart = (button == 1) ? 0.0f : m_SectorEnd[button - 2];
    m_SelEnd = m_SectorEnd[button - 1];
  }

  // re-draw the trajectory
  wxClientDC dc(this);
  DoPaint(dc);

  // synchronise other panes/graphs
  ::SetTrackSelection_Send(m_SelStart, m_SelEnd);
}

//-----------------------------------------------------------------------------

void cTrackMap::OnMouseLeftDoubleClick(wxMouseEvent& event)
{
  int button = -1;
  DecodeMouse(event, &button);
  if (button < 0) ResetZoom(); // reset zoom if not on a button
}

//-----------------------------------------------------------------------------

void cTrackMap::OnMouseMove(wxMouseEvent& event)
{
  int button = -1;
  float dist = DecodeMouse(event, &button);
  if (m_ShowSectorToolbar) SelectButton(button);

  if (dist < 0) {
    // mouse not near track line
    SetCursor(wxNullCursor); // indicates that clicks will be ignored
    return;
  }

  SetCursor(wxCursor(wxCURSOR_HAND)); // indicates that clicks will be recognised

  if (!event.LeftIsDown()) return; // not drag-selecting
  if (m_ClickPos < 0) return; // last click was not near the track line
  if (fabs(dist - m_ClickPos) < MIN_SELECTION_LENGTH) return; // too near to start of drag

  // enter drag mode
  m_DragMode = true;
  wxClientDC dc(this);
  RemoveTrackCursor(dc);

  // re-draw the trajectory
  if (dist < m_ClickPos) {
    m_SelStart = dist;
    m_SelEnd = m_ClickPos;
  }
  else {
    m_SelStart = m_ClickPos;
    m_SelEnd = dist;
  }
  DoPaint(dc);
}

//-----------------------------------------------------------------------------

void cTrackMap::OnMouseLeftRelease(wxMouseEvent& event)
{
  event.Skip(); // always pass left-clicks to default event handler
  if (MGR->IsEmpty()) return; // empty track map
  if (m_ClickPos < 0) return; // last click was not near the track line

  if (m_DragMode) {
    // ending drag-select - set the track selection in the graphs
    m_DragMode = false;
    if (m_SelStart < m_SelEnd) {
      ::SetTrackSelection_Send(m_SelStart, m_SelEnd);
    } else {
      ::SetTrackSelection_Send(m_SelEnd, m_SelStart);
    }
    // set cursor half-way in the selected part
    ::SetTrackCursorPos_Send(((m_SelStart + m_SelEnd) / 2.0f), true);
  }
  else {
    // simple click+release
    if (event.ShiftDown()) {
      // modify selected part
      if (m_ClickPos < m_SelStart) {
        m_SelEnd = m_SelStart;
        m_SelStart = m_ClickPos;
      }
      else {
        m_SelEnd = m_ClickPos;
      }
      ::SetTrackSelection_Send(m_SelStart, m_SelEnd);
    }
    else {
      // set track cursor at this point on the track
      ::SetTrackCursorPos_Send(m_ClickPos, true);
    }
  }

  m_ClickPos = -1; // 'forget' last click
}

//-----------------------------------------------------------------------------
// Return the distance-in-lap that corresponds to a mouse position

float cTrackMap::DecodeMouse(wxMouseEvent& event, int* button)
{
  if (MGR->IsEmpty()) return -1;

  // get logical position of click
  wxClientDC dc(this);
  wxPoint pos = event.GetLogicalPosition(dc);

  // check the sector buttons
  if (button != NULL) {
    *button = -1;
    if ((pos.x >= m_ButtonStart.x) &&
        (pos.x < m_ButtonStart.x + m_ButtonSize) &&
        (pos.y >= m_ButtonStart.y) &&
        (pos.y < m_ButtonStart.y + (m_SectorCount + 1) * m_ButtonSize)) {
      *button = Coord2ButtonY(pos.y);
      return -1.0f;
    }
  }

  // find state that is nearest to this position
  float nearest_dist = -1; // distance-in-lap of state that is nearest to mouse position
  float nearest_pix = IMPOSSIBLY_HIGH_VALUE; // distance in pixels to that state
  cLap* lap = MGR->GetLap(0);
  for (size_t s = 0; s < lap->GetStateCount(); s++) {
    const cCarState* state = lap->GetState(s);
    float dX = (float)(Track2CoordX(state->GetPosX()) - pos.x);
    float dY = (float)(Track2CoordY(state->GetPosY()) - pos.y);
    float d = sqrt((dX * dX) + (dY * dY));
    if (d < nearest_pix) {
      nearest_pix = d;
      nearest_dist = state->GetDistance();
    }
  }

  if (nearest_pix > CLICK_DETECT_THRESHOLD) return -1.0f;

  return nearest_dist;
}

//-----------------------------------------------------------------------------

void cTrackMap::DoSetTrackSelection(float start, float end)
{
  m_SelStart = start;
  m_SelEnd = end;

  if (MGR->IsEmpty()) return; // empty display

  wxClientDC dc(this);
  RemoveTrackCursor(dc);
  DoPaint(dc); // re-draw the trajectory
  PutTrackCursor(dc, m_CursorDist);
}

//-----------------------------------------------------------------------------

void cTrackMap::ResetZoom()
{
  if (MGR->IsEmpty()) return;

  m_SelStart = 0.0f;
  m_SelEnd = MGR->GetTrackLength();

  // re-draw the trajectory
  wxClientDC dc(this);
  DoPaint(dc);

  // synchronise other panes/graphs
  ::SetTrackSelection_Send(m_SelStart, m_SelEnd);
}

//-----------------------------------------------------------------------------

void cTrackMap::OnMouseWheel(wxMouseEvent& event)
{
  if (MGR->IsEmpty()) return;

  float diff;
  if (event.GetWheelRotation() > 0) {
    diff = -SCROLL_DIST;
    if (m_SelStart + diff < 0.0f) diff = -m_SelStart;
  }
  else {
    diff = SCROLL_DIST;
    if (m_SelEnd + diff > MGR->GetTrackLength()) diff = MGR->GetTrackLength() - m_SelEnd;
  }
  ::SetTrackSelection_Send(m_SelStart + diff, m_SelEnd + diff);
  if (m_CursorDist >= 0.0f) ::SetTrackCursorPos_Send(m_CursorDist + diff, false);
}

//-----------------------------------------------------------------------------

void cTrackMap::UpdateAll()
{
  wxString track;
  if (!MGR->IsEmpty()) track = MGR->GetTrackCode();
  if (m_TrackCode != track) {
    // clear current track data
    m_SectorCount = 0;
    m_TrackCode = track;
  }

  if ((!m_TrackCode.IsEmpty()) && (m_SectorCount == 0)) {
    // load new track data
    for (size_t i = 0; i < MGR->GetLapCount(); i++) {
      cLap* lap = MGR->GetLap(i);
      if (!lap->IsComplete()) continue; // search for a completed lap
      if (lap->GetSplitCount() > MAX_SECTORS) continue; // defensive

      // derive sectors from split times
      m_SectorCount = lap->GetSplitCount();
      for (int s = 0; s < m_SectorCount - 1; s++) {
        m_SectorEnd[s] = lap->GetDistanceAt(lap->GetSplit(s));
      }
      m_SectorEnd[m_SectorCount - 1] = lap->GetTrackLength();
    }
  }

  AdjustDimensions();

  SetMinSize(wxSize(100, 100)); // TODO: set a more sensible size?
  Refresh();
}

//-----------------------------------------------------------------------------

void cTrackMap::OnMouseRightClick(wxMouseEvent& WXUNUSED(event))
{
  if (m_DragMode) return; // don't show context menu while a drag is going on

  m_Context.Enable(ID_MENU_RESETZOOM,
      (!MGR->IsEmpty()) &&
      ((m_SelStart > 0.0f) || ((m_SelEnd > 0.0f) && (m_SelEnd < MGR->GetTrackLength()))));
  m_Context.Check(ID_MENU_SHOWDISTANCE, m_ShowDistance);
  m_Context.Check(ID_MENU_SHOWSECTORBUTTONS, m_ShowSectorToolbar);
  PopupMenu(&m_Context);
}

//-----------------------------------------------------------------------------
// Handle a click in the context menu

void cTrackMap::OnMenuClick(wxCommandEvent& event)
{
  switch (event.GetId()) {
    case ID_MENU_RESETZOOM :
      ResetZoom();
      break;

    case ID_MENU_SHOWDISTANCE :
      m_ShowDistance = !m_ShowDistance;
      AdjustDimensions();
      Refresh();
      break;

    case ID_MENU_SHOWSECTORBUTTONS :
      m_ShowSectorToolbar = !m_ShowSectorToolbar;
      if (!m_ShowSectorToolbar) m_SelectedButton = -1;
      AdjustDimensions();
      Refresh();
      break;

    default :
      wxFAIL;
  }
}

//-----------------------------------------------------------------------------
// Loading and saving the configuration settings

void cTrackMap::LoadConfig(wxRegConfig* config, const wxString& key)
{
  config->Read(key + _T("/showdistance"), &m_ShowDistance);
  config->Read(key + _T("/sectors/showtoolbar"), &m_ShowSectorToolbar);
}


void cTrackMap::SaveConfig(wxRegConfig* config, const wxString& key)
{
  config->Write(key + _T("/showdistance"), m_ShowDistance);
  config->Write(key + _T("/sectors/showtoolbar"), m_ShowSectorToolbar);
}

//-----------------------------------------------------------------------------
// Convert button number to window coordinate and vice versa

wxCoord cTrackMap::Button2CoordY(int button) const
{
  return m_ButtonStart.y + button * (m_ButtonSize + 2);
}

int cTrackMap::Coord2ButtonY(wxCoord coord) const
{
  return (coord - m_ButtonStart.y) / (m_ButtonSize + 2);
}

//-----------------------------------------------------------------------------
// Set the 'selected' button
// - button = button number (-1 = none)

void cTrackMap::SelectButton(int button)
{
  wxASSERT(m_ShowSectorToolbar);
  wxASSERT(button <= m_SectorCount); // button number must be valid

  if (button == m_SelectedButton) return; // no change

  wxClientDC dc(this);

  // remove current selection
  if (m_SelectedButton >= 0) DrawSectorButton(dc, m_SelectedButton, false);

  m_SelectedButton = button;

  // draw new selection
  if (m_SelectedButton >= 0) DrawSectorButton(dc, m_SelectedButton, true);
}

//-----------------------------------------------------------------------------
// Draw the 'selected' box around a button, with the current pen
// - dc = current device context
// - button = button number

void cTrackMap::DrawSectorButton(wxDC& dc, int button, bool highlight)
{
  wxASSERT(m_ShowSectorToolbar);
  wxASSERT((button >= 0) && (button <= m_SectorCount)); // button number must be valid

  ::DrawButton(dc,
      wxRect(m_ButtonStart.x, Button2CoordY(button), m_ButtonSize, m_ButtonSize),
      highlight);
}

//-----------------------------------------------------------------------------

void cTrackMap::OnMouseEntering(wxMouseEvent& event)
{
  int button = -1;
  DecodeMouse(event, &button);
  if (m_ShowSectorToolbar) SelectButton(button);
}

void cTrackMap::OnMouseLeaving(wxMouseEvent& WXUNUSED(event))
{
  if (m_ShowSectorToolbar) SelectButton(-1);
}
