#include "cmgr.h"
#include "ctrackpath.h"
#include <wx/filename.h>

#define REL_PATH_SMX wxFILE_SEP_PATH + _T("smx")

//-----------------------------------------------------------------------------

cMgr::cMgr()
{
  m_TrackPath = NULL;
  ClearAll();
  m_LapNaming = LAPNAME_FILE;

  cCarState::DefineTypes();
}

//-----------------------------------------------------------------------------

cMgr::~cMgr()
{
  delete m_TrackPath;
  s_Mgr = NULL;
}

//----------------------------------------------------------------------------

cMgr* cMgr::GetMgr()
{
  // singleton pattern
  if (s_Mgr == NULL) s_Mgr = new cMgr;
  return s_Mgr;
}

//----------------------------------------------------------------------------

cMgr* cMgr::s_Mgr = 0;

//-----------------------------------------------------------------------------
// Comparison function, to be used in sorting the list of laps

int cMgr::Compare(cLap** a, cLap** b)
{
  // sort on display name, case-insensitive
  return ((*a)->GetName().CmpNoCase((*b)->GetName()));
}

//-----------------------------------------------------------------------------
// Change the lap naming scheme

void cMgr::SetLapNaming(lapname_t naming)
{
  wxASSERT(naming != LAPNAME_NONE);
  if (m_LapNaming == naming) return; // no change

  m_LapNaming = naming;
  for (size_t l = 0; l < m_Laps.GetCount(); l++) {
    m_Laps[l].SetName(m_LapNaming);
  }

  SortLaps(); // re-sort the laps
}

//-----------------------------------------------------------------------------
// Add a lap (= RAF file) that has been loaded
// - lap = lap to add
// - error = receives the error description of lap can't be added

bool cMgr::AddLap(cLap* lap, wxString& error)
{
  wxASSERT(lap->IsLoaded());

  // check if lap is for the same track as the others
  if ((m_Laps.GetCount() > 0) && (lap->GetTrackCode() != GetTrackCode())) {
    error = wxString::Format(_T("File contains data from a different track (%s)"), lap->GetTrackCode().c_str());
    return false;
  }

  // add the lap
  m_Laps.Add(lap);
  lap->SetName(m_LapNaming);
  SortLaps(); // keep the laps sorted

  // first lap only: sort out the track data
  if (m_Laps.GetCount() == 1) {
    // try to load track path
    wxASSERT(m_TrackPath == NULL); // no track path loaded yet
    wxString file = lap->GetFileName();
    wxString dir = file.BeforeLast(wxFILE_SEP_PATH).BeforeLast(wxFILE_SEP_PATH) + REL_PATH_SMX;
    wxString suffix = wxFILE_SEP_PATH + lap->GetTrackCode() + _T(".pth");

    // first try: relative to path of RAF file
    wxString pthFile = dir + suffix;

    // second try: m_SmxDir
    if (!wxDirExists(m_SmxDir)) m_SmxDir = wxEmptyString; // directory no longer exists
    if ((!wxFileExists(pthFile)) && (!m_SmxDir.IsEmpty())) {
      pthFile = m_SmxDir + suffix;
    }

    if (wxFileExists(pthFile)) {
      m_TrackPath = new cTrackPath();
      if (m_TrackPath->Load(pthFile)) {
        // load successful
        if (m_SmxDir.IsEmpty())
            m_SmxDir = pthFile.BeforeLast(wxFILE_SEP_PATH); // remember directory
      }
      else {
        // load failed
        delete m_TrackPath;
        m_TrackPath = NULL;
      }
    }
    else {
      wxLogDebug(_T("Could not find track outline file \"%s.pth\""), lap->GetTrackCode().c_str());
    }
  }

  return true;
}

//-----------------------------------------------------------------------------

void cMgr::DeleteLap(cLap* lap)
{
  wxASSERT(lap != NULL);
  int index = m_Laps.Index(*lap);
  wxASSERT(index != wxNOT_FOUND);

  m_Laps.RemoveAt(index);
  if (m_Laps.IsEmpty()) ClearAll();
}

//-----------------------------------------------------------------------------
// Check if a lap file is already loaded

bool cMgr::IsLoaded(const wxString& fileName)
{
  for (size_t l = 0; l < m_Laps.GetCount(); l++) {
    if (fileName == m_Laps[l].GetFileName()) return true;
  }
  return false;
}

//-----------------------------------------------------------------------------
// Delete all data

void cMgr::ClearAll()
{
  m_Laps.Clear();
  delete m_TrackPath;
  m_TrackPath = NULL;
}
