#include "clap.h"
#include "ccolourlist.h"
#include "ctrack.h"
#include <math.h>
#include <wx/filename.h>
#include <wx/stopwatch.h>

// time interval between car states (0.01 second)
#define TIME_INTERVAL 0.01f

#define GCALC_RANGE 5

// magic values for determining FWD/RWD
#define WD_THRESHOLD_PEDAL 0.95f
#define WD_THRESHOLD_FORCE 200.0f

//-----------------------------------------------------------------------------
// Define an array of cLap

#include <wx/arrimpl.cpp>
WX_DEFINE_OBJARRAY(cLapArray);

//-----------------------------------------------------------------------------

cLap::cLap(const wxString& fileName)
{
  m_FileName = fileName;
  wxASSERT(wxFileExists(m_FileName));
  m_IsComplete = false;

  m_Naming = LAPNAME_NONE;
  SetName(LAPNAME_FILE);

  m_TrackLength = 0.0f;
  m_Status = LAPSTATUS_CREATED;
  m_Shown = false;

  m_MinX = IMPOSSIBLY_HIGH_VALUE;
  m_MaxX = IMPOSSIBLY_LOW_VALUE;
  m_MinY = IMPOSSIBLY_HIGH_VALUE;
  m_MaxY = IMPOSSIBLY_LOW_VALUE;

  m_Colour = cColourList::Alloc();
}

//-----------------------------------------------------------------------------

cLap::~cLap()
{
  cColourList::Free(m_Colour);
}

//-----------------------------------------------------------------------------
// Get the display name
// - naming = naming scheme to use

wxString cLap::GetName(lapname_t naming) const
{
  CHECK_THIS;
  wxASSERT(naming != LAPNAME_NONE);
  wxASSERT(!m_FileName.IsEmpty());

  if (naming == m_Naming) return m_Name;

  wxString name;
  switch (naming) {
    case LAPNAME_FILE :
      wxFileName::SplitPath(m_FileName, NULL, NULL, &name, NULL);
      break;

    case LAPNAME_TIME :
      wxASSERT(IsLoaded());
      name = ::FormatTime(m_Split[m_SplitCount - 1]);
      break;

    case LAPNAME_TIME_DRIVER :
      wxASSERT(IsLoaded());
      name.Printf(_T("%s  %s"),
          ::FormatTime(m_Split[m_SplitCount - 1]).c_str(),
          m_PlayerName.c_str());
      break;

    case LAPNAME_TIME_CAR_DRIVER :
      wxASSERT(IsLoaded());
      name.Printf(_T("%s  %s - %s"),
          ::FormatTime(m_Split[m_SplitCount - 1]).c_str(),
          cCar::Name2Code(m_Car.m_Name).c_str(),
          m_PlayerName.c_str());
      break;

    default:
      wxFAIL;
  }

  return name;
}


void cLap::SetName(lapname_t naming)
{
  wxASSERT(naming != LAPNAME_NONE);
  m_Name = GetName(naming);
  m_Naming = naming;
}

//-----------------------------------------------------------------------------
// Start the loading of a RAF file
// - fileName = full filename
// Returns false if unable to load

bool cLap::LoadStart()
{
  wxASSERT(!IsOpened());
  wxASSERT(m_Status == LAPSTATUS_CREATED);
  m_Status = LAPSTATUS_LOADING;

  if (!Open(m_FileName)) return LoadFailed(_T("Could not open file"));

  // NB: RAF file format is documented on http://www.liveforspeed.net/?page=RAF

  // ----- initialization -----
  for (int i = 0; i < 4; i++) m_AvgTemp[i] = 0.0f;

  // ----- read the header -----
  wxString str;
  ReadString(str, 6); // read fixed header string 'LFSRAF'
  if (str != "LFSRAF") return LoadFailed(_T("Not a RAF file"));
  Skip(2); // game version, game revision
  wxInt8 format;
  ReadByte(format);
  if (format != 2) return LoadFailed(wxString::Format(_T("Unknown RAF format version (%d)"), format));
  Skip(3); // empty

  ReadWord(m_HeaderSize);
  ReadWord(m_BlockSize);
  ReadWord(m_WheelBlockSize);
  ReadWord(m_WheelBlockOffset);
  // check block sizes (for the current format version)
  wxASSERT(m_HeaderSize == 1024);
  wxASSERT(m_BlockSize == 192);
  wxASSERT(m_WheelBlockSize == 32);
  wxASSERT(m_WheelBlockOffset == 64);

  ReadInt(m_BlockCount);

  // sanity check on block count: max laptime between 0 and 30 minutes
  if ((m_BlockCount <= 0) || (m_BlockCount > 100*60*30))
      return LoadFailed(wxString::Format(_T("File format error (block count = %d)"), m_BlockCount));

  ReadString(m_TrackCode, 4); // will get empty string for S2 patch W and earlier
  ReadFloat(m_TrackLength);
  ReadString(m_PlayerName, 32);
  UnEscapeLFS(m_PlayerName);

  ReadString(m_Car.m_Name, 32);
  ReadString(m_TrackName, 32);
  ReadString(m_ConfigName, 16);

  // determine track code, and canonicalize track+config name
  wxString code = cTrack::Name2Code(m_TrackName, m_ConfigName);
  if ((code.IsEmpty()) && (m_TrackCode.IsEmpty()))
      return LoadFailed(wxString::Format(_T("Unknown track (%s %s)"), m_TrackName.c_str(), m_ConfigName.c_str()));

  if (m_TrackCode.IsEmpty()) m_TrackCode = code;
  // NB: since S2 patch X the track code can be read from the RAF file, but
  // we still need to decode it, to support older versions and to canonicalize the track+config name
  // However, we'll try to detect any differences
  wxASSERT(code == m_TrackCode);

  // sanity check on track length
  if (m_TrackLength <= 0.0f)
      return LoadFailed(wxString::Format(_T("Can't handle files for track %s"), m_TrackCode.c_str()));

  m_FullTrackName = m_TrackName + _T(" " ) + m_ConfigName;

  ReadString(m_Weather, 16);
  ReadString(m_LfsVersion, 8);
  ReadByte(m_PlayerFlags);

  wxInt8 wheels;
  ReadByte(wheels);
  if (wheels != 4) return LoadFailed(_T("Can only handle cars with 4 wheels"));

  ReadByte(m_Hlvc);
  ReadByte(m_SplitCount);
  for (int i = 0; i < 4; i++) ReadInt(m_Split[i]);

  // ----- read static car data -----
  GoTo(188);
  Skip(4 + 4); // mass, unsprung mass
  ReadFloat(m_Car.m_AntiRoll_Rear);
  ReadFloat(m_Car.m_AntiRoll_Front);
  ReadFloat(m_Car.m_FinalDrive);
  ReadByte(m_Car.m_GearCount);
  Skip(3);
  for (int i = 0; i < 6; i++) ReadFloat(m_Car.m_GearRatio[i]);

  // ----- read static wheel data -----
  for (int w = 0; w < 4; w++) {
    GoTo(512 + w * 128);

    float x, y;
    ReadFloat(x);
    ReadFloat(y);
    Skip(4); // Z position

    m_Wheelmap[w] = 0;
    if (x > 0) m_Wheelmap[w] += 1; // this is a right wheel
    if (y < 0) m_Wheelmap[w] += 2; // this is a rear wheel

    m_Car.m_WheelX[m_Wheelmap[w]] = x;
    m_Car.m_WheelY[m_Wheelmap[w]] = y;

    ReadFloat(m_Car.m_Radius[m_Wheelmap[w]]);
    Skip(4); // width
    ReadFloat(m_Car.m_MaxDeflect[m_Wheelmap[w]]);
    Skip(4 + 1);
    ReadByte(m_Car.m_TyreType[m_Wheelmap[w]]);
    Skip(2);
    ReadFloat(m_Car.m_Spring[m_Wheelmap[w]]);
    ReadFloat(m_Car.m_Damp_Bump[m_Wheelmap[w]]);
    ReadFloat(m_Car.m_Damp_Rebound[m_Wheelmap[w]]);
    ReadFloat(m_Car.m_Brake[m_Wheelmap[w]]);
  }

  // check validity of wheel mapping
  wxASSERT(m_Wheelmap[0] != m_Wheelmap[1]);
  wxASSERT(m_Wheelmap[0] != m_Wheelmap[2]);
  wxASSERT(m_Wheelmap[0] != m_Wheelmap[3]);
  wxASSERT(m_Wheelmap[1] != m_Wheelmap[2]);
  wxASSERT(m_Wheelmap[1] != m_Wheelmap[3]);
  wxASSERT(m_Wheelmap[2] != m_Wheelmap[3]);

  // ----- prepare to read the states -----
  m_PrevTime = IMPOSSIBLY_LOW_VALUE;
  m_Log.Alloc(m_BlockCount);
  m_BlockIndex = 0;

  m_FwdStates_Y = 0;
  m_FwdStates_N = 0;
  m_RwdStates_Y = 0;
  m_RwdStates_N = 0;

  m_Status = LAPSTATUS_LOADING;
  return true;
}

//-----------------------------------------------------------------------------
// Load the next batch of data
// - ms = allowed time (in ms) to go on reading
// Returns false if error or ready

bool cLap::LoadNext(int ms)
{
  wxASSERT(m_Status == LAPSTATUS_LOADING);
  wxStopWatch sw;

  // load more states
  while (m_BlockIndex < m_BlockCount) {
    if (Eof()) return LoadFailed(_T("Unexpected end-of-file"));

    GoTo(m_HeaderSize + m_BlockIndex * m_BlockSize); // go to start of block

    cCarState* state = new cCarState(&m_Car);
    ReadFloat(state->m_Throttle);
    ReadFloat(state->m_Brake);
    ReadFloat(state->m_Steer);
    ReadFloat(state->m_Clutch);
    Skip(4); // handbrake
    ReadByte(state->m_Gear);
    Skip(3); // empty
    ReadFloat(state->m_Speed);
    Skip(4); // car distance

    wxInt32 pos;
    ReadInt(pos); state->m_Pos.x = (float)pos / 65536.0f;
    ReadInt(pos); state->m_Pos.y = (float)pos / 65536.0f;
    ReadInt(pos); state->m_Pos.z = (float)pos / 65536.0f;

    ReadFloat(state->m_Rpm);
    ReadFloat(state->m_Distance);

    if (m_BlockIndex == 0) {
      // initialise time-in-lap by estimating from distance and current speed
      state->m_TimeInLap = state->m_Distance / state->m_Speed;
    }
    else {
      state->m_TimeInLap = m_PrevTime + TIME_INTERVAL;
    }

    wxInt16 dir;
    ReadWord(dir); state->m_Right.x = (float)dir / 32767.0f;
    ReadWord(dir); state->m_Right.y = (float)dir / 32767.0f;
    ReadWord(dir); state->m_Right.z = (float)dir / 32767.0f;
    ReadWord(dir); state->m_Fwd.x = (float)dir / 32767.0f;
    ReadWord(dir); state->m_Fwd.y = (float)dir / 32767.0f;
    ReadWord(dir); state->m_Fwd.z = (float)dir / 32767.0f;
    state->m_Right.Normalize(); // vectors should have unit length
    state->m_Fwd.Normalize();

    // read dynamic wheel data
    for (int w = 0; w < 4; w++) {
      GoTo(m_HeaderSize + m_BlockIndex * m_BlockSize + m_WheelBlockOffset + w * m_WheelBlockSize);
      ReadFloat(state->m_Deflect[m_Wheelmap[w]]);
      ReadFloat(state->m_WheelAngle[m_Wheelmap[w]]);
      ReadFloat(state->m_ForceLat[m_Wheelmap[w]]);
      ReadFloat(state->m_ForceLong[m_Wheelmap[w]]);
      ReadFloat(state->m_Load[m_Wheelmap[w]]);
      ReadFloat(state->m_AngVel[m_Wheelmap[w]]);
      ReadFloat(state->m_Camber[m_Wheelmap[w]]);
      wxInt8 temp;
      ReadByte(temp);
      m_AvgTemp[m_Wheelmap[w]] += temp;
    }

    wxASSERT(m_Log.GetCount() == (size_t)m_BlockIndex);
    if (Eof()) return LoadFailed(_T("Unexpected end-of-file"));
    m_Log.Add(state);

    // calculated state data
    ::Minimise(m_MinX, state->GetPosX());
    ::Maximise(m_MaxX, state->GetPosX());
    ::Minimise(m_MinY, state->GetPosY());
    ::Maximise(m_MaxY, state->GetPosY());

    if (state->m_Throttle > WD_THRESHOLD_PEDAL) {
      // in this state, the throttle is pushed
      // TODO: state only counts if the clutch isn't pressed (can be done after patch Y)
      if (state->m_ForceLong[0] > WD_THRESHOLD_FORCE) {
        // front left wheel is driven
        m_FwdStates_Y += 1;
      }
      else {
        m_FwdStates_N += 1;
      }
      if (state->m_ForceLong[2] > WD_THRESHOLD_FORCE) {
        // rear left wheel is driven
        m_RwdStates_Y += 1;
      }
      else {
        m_RwdStates_N += 1;
      }
    }

    // calculated state data that needs data from other states
    int calc, prev, next; // index of state to calculate for, and the previous and next states
    if (m_BlockIndex <= GCALC_RANGE) {
      // first couple of states: calculation not needed
    }
    else if (m_BlockIndex < m_BlockCount - 1) {
      // intermediate states: normal calculation
      next = m_BlockIndex;
      calc = m_BlockIndex - GCALC_RANGE;
      wxASSERT(calc >= 1);
      prev = calc - GCALC_RANGE;
      ::Maximise(prev, 0);
      m_Log[calc].CalculateDerivedData(m_Log[prev], m_Log[next]);
    }
    else if (m_BlockIndex == m_BlockCount - 1) {
      // last state: calculate for the last couple of states
      for (calc = m_BlockIndex - GCALC_RANGE; calc < m_BlockIndex; calc++) {
        prev = calc - GCALC_RANGE;
        ::Maximise(prev, 0); // defensive
        m_Log[calc].CalculateDerivedData(m_Log[prev], m_Log[m_BlockIndex]);
      }
    }

    // prepare for next block
    m_BlockIndex += 1;
    m_PrevTime = state->m_TimeInLap;

    if (sw.Time() > ms) return true; // time's up!
  }

  // done reading the file data
  Close();

  // determine completeness
  float laptime = GetLaptime();
  if ((laptime >= (m_BlockCount - 1) * TIME_INTERVAL) &&
      (laptime <= (m_BlockCount + 1) * TIME_INTERVAL)) {
    m_IsComplete = true;
  }

  // sanity check on m_Distance
  if (m_Log[0].m_Distance >= m_Log[m_Log.GetCount() - 1].m_Distance)
      return LoadFailed(_T("File format error (descending distance-in-lap)"));

  // take average of tyre temp
  for (int i = 0; i < 4; i++) m_AvgTemp[i] = m_AvgTemp[i] / m_Log.GetCount();

  // determine FWD/RWD
  m_Car.m_Fwd = (m_FwdStates_Y > m_FwdStates_N);
  m_Car.m_Rwd = (m_RwdStates_Y > m_RwdStates_N);
  wxASSERT(m_Car.m_Fwd || m_Car.m_Rwd); // sanity check

  // loading successful
  wxASSERT(m_LoadError.IsEmpty());
  wxASSERT(m_Status == LAPSTATUS_LOADING);
  m_Status = LAPSTATUS_LOAD_OK;
  m_Shown = true;

  return false; // reading complete
}

//-----------------------------------------------------------------------------
// Utility function to signal an error in loading
// - error = error description
// Always return false

bool cLap::LoadFailed(wxString error)
{
  wxASSERT(m_Status == LAPSTATUS_LOADING);
  m_Status = LAPSTATUS_LOAD_FAIL;
  m_LoadError = error;
  Close();
  return false;
}

//-----------------------------------------------------------------------------

size_t cLap::FindStateAt(float distance) const
{
  wxASSERT(m_Status == LAPSTATUS_LOAD_OK);

  // use binary search
  size_t lo = 0;
  size_t hi = m_Log.GetCount() - 1;

  if (distance <= GetState(lo)->GetDistance()) return lo;
  if (distance >= GetState(hi)->GetDistance()) return hi;

  wxASSERT(GetState(lo)->GetDistance() <= GetState(hi)->GetDistance());

  while (hi - lo >= 2) {
    size_t mid = (hi + lo) / 2;
    if (GetState(mid)->GetDistance() < distance) {
      lo = mid;
    }
    else {
      hi = mid;
    }
  }
  return lo;
}

//-----------------------------------------------------------------------------

void cLap::GetPositionAt(float distance, float& x, float& y) const
{
  wxASSERT(IsLoaded());

  size_t index = FindStateAt(distance);
  const cCarState* state1 = GetState(index);
  const cCarState* state2 = (index >= m_Log.GetCount() - 1) ? state1 : GetState(index + 1);

  float lambda = 0.0f; // interpolation scalar
  float dist1 = state1->GetDistance();
  float dist2 = state2->GetDistance();
  if (dist2 - dist1 > EPSILON) lambda = (distance - dist1) / (dist2 - dist1);

  x = (1.0f - lambda) * state1->GetPosX() + lambda * state2->GetPosX();
  y = (1.0f - lambda) * state1->GetPosY() + lambda * state2->GetPosY();
}

//-----------------------------------------------------------------------------

float cLap::GetLogDataAt(float distance, int log, int wheel) const
{
  wxASSERT(IsLoaded());
  wxASSERT(log >= LOGTYPE_FIRST);
  wxASSERT(log < LOGTYPE_LAST);
  wxASSERT(wheel >= WHEELTYPE_FIRST);
  wxASSERT(wheel < WHEELTYPE_LAST);

  size_t index = FindStateAt(distance);

  // distance and value of this state
  float dist1 = GetState(index)->GetDistance();
  float val1 = GetState(index)->GetLogData(log, wheel);

  if (index == m_Log.GetCount() - 1)
    return val1;

  // distance and value of the next state
  float dist2 = GetState(index + 1)->GetDistance();
  float val2 = GetState(index + 1)->GetLogData(log, wheel);

  // calculate value
  float lambda = 0.0f; // interpolation scalar
  if (dist2 - dist1 > EPSILON) lambda = (distance - dist1) / (dist2 - dist1);

  float result;
  if (cCarState::IsInteger(log)) {
    // integer logdata: take nearest state
    result = (lambda < 0.5f) ? val1 : val2;
  }
  else {
    // float logdata: calculate by interpolation
    result = (1.0f - lambda) * val1 + lambda * val2;
  }

  return result;
}

//-----------------------------------------------------------------------------
//

float cLap::GetDistanceAt(float time)
{
  wxASSERT(m_Status == LAPSTATUS_LOAD_OK);

  // use binary search
  size_t lo = 0;
  size_t hi = m_Log.GetCount() - 1;

  if (time <= GetState(lo)->GetTime()) return GetState(lo)->GetDistance();
  if (time >= GetState(hi)->GetTime()) return GetState(hi)->GetDistance();

  wxASSERT(GetState(lo)->GetTime() <= GetState(hi)->GetTime());

  while (hi - lo >= 2) {
    size_t mid = (hi + lo) / 2;
    if (GetState(mid)->GetTime() < time) {
      lo = mid;
    }
    else {
      hi = mid;
    }
  }

  float time1 = GetState(lo)->GetTime();
  float time2 = GetState(hi)->GetTime();
  float lambda = (time - time1) / (time2 - time1); // interpolation scalar
  return (1.0f - lambda) * GetState(lo)->GetDistance() + lambda * GetState(hi)->GetDistance();
}
