#include "cfiledlg.h"
#include "cfile.h"
#include "ccar.h"
#include "ctrack.h"
#include "cmgr.h"
#include <wx/stattext.h>
#include <wx/dir.h>
#include <wx/datetime.h>
#include <wx/dirdlg.h>
#include <wx/filefn.h>
#include <wx/msgdlg.h>
#include <wx/textdlg.h>
#include <wx/config.h>
#include <wx/filename.h>

// sorting indicators on columns (ascending/descending)
#define SORT_INDICATOR_ASC _T("> ")
#define SORT_INDICATOR_DESC _T("< ")

//-----------------------------------------------------------------------------
// Enums

// columns in the file list
enum ENUM_COLS {
  COL_FILENAME = 0,
  COL_CARCODE,
  COL_TRACKCODE,
  COL_LAPTIME,
  COL_DATE,
  COL_PLAYER
};

// control IDs
enum {
  ID_FILE_LIST = 0,
  ID_CHOICE_CAR,
  ID_CHOICE_TRACK,
  ID_BTN_FOLDER,
  ID_BTN_DELETE,
  ID_BTN_RENAME,
  ID_BTN_ALL
};

// menu identifiers
enum ENUM_MENU_IDS
{
  ID_MENU_RENAME = 100,
  ID_MENU_DELETE,
  ID_MENU_SELECTALL,
  ID_MENU_LAST
};


//-----------------------------------------------------------------------------
// Event table

BEGIN_EVENT_TABLE(cFileDlg, wxDialog)
  EVT_BUTTON(wxID_ANY, cFileDlg::OnButton)
  EVT_CHOICE(ID_CHOICE_CAR, cFileDlg::OnChangeCarFilter)
  EVT_CHOICE(ID_CHOICE_TRACK, cFileDlg::OnChangeTrackFilter)
  EVT_LIST_KEY_DOWN(ID_FILE_LIST, cFileDlg::OnKeyDown)
  EVT_LIST_COL_CLICK(ID_FILE_LIST, cFileDlg::OnColClick)
  EVT_LIST_ITEM_ACTIVATED(ID_FILE_LIST, cFileDlg::OnItemActivate)
  EVT_LIST_ITEM_DESELECTED(ID_FILE_LIST, cFileDlg::OnChangeSelection)
  EVT_LIST_ITEM_SELECTED(ID_FILE_LIST, cFileDlg::OnChangeSelection)
  EVT_LIST_ITEM_RIGHT_CLICK(ID_FILE_LIST, cFileDlg::OnItemRightClick)
  EVT_MENU(wxID_ANY, cFileDlg::OnMenuClick)
END_EVENT_TABLE()

//-----------------------------------------------------------------------------
// - parent = parent window
// - title = text for caption
// - dir = directory with RAF files
// - car = the car filter (empty if none)
// - track = the track filter (empty if none)

cFileDlg::cFileDlg(wxWindow* parent)
: wxDialog(parent, -1, wxEmptyString, wxDefaultPosition, wxDefaultSize, wxDEFAULT_DIALOG_STYLE | wxRESIZE_BORDER)
{
  m_SortKey = 1;

  // list of RAF files
  m_List = new wxListView(this, ID_FILE_LIST,
      wxDefaultPosition, wxDefaultSize, wxLC_REPORT | wxSIMPLE_BORDER);
  m_List->InsertColumn(COL_FILENAME, SORT_INDICATOR_ASC _T("File"));
  m_List->InsertColumn(COL_CARCODE, _T("Car"));
  m_List->InsertColumn(COL_TRACKCODE, _T("Track"));
  m_List->InsertColumn(COL_LAPTIME, _T("Laptime"));
  m_List->InsertColumn(COL_DATE, _T("Date"));
  m_List->InsertColumn(COL_PLAYER, _T("Player name"));

  // controls for filtering the list
  m_CarChoice = new wxChoice(this, ID_CHOICE_CAR, wxDefaultPosition, wxDefaultSize, 0, NULL);
  m_TrackChoice = new wxChoice(this, ID_CHOICE_TRACK, wxDefaultPosition, wxDefaultSize, 0, NULL);

  // other buttons
  m_AllBtn = new wxButton(this, ID_BTN_ALL, _T("&All"),
      wxDefaultPosition, wxDefaultSize, wxBU_EXACTFIT);
  m_FolderBtn = new wxButton(this, ID_BTN_FOLDER, _T("&Folder..."),
      wxDefaultPosition, wxDefaultSize, wxBU_EXACTFIT);
  m_DeleteBtn = new wxButton(this, ID_BTN_DELETE, _T("&Delete"),
      wxDefaultPosition, wxDefaultSize, wxBU_EXACTFIT);
  m_RenameBtn = new wxButton(this, ID_BTN_RENAME, _T("&Rename"),
      wxDefaultPosition, wxDefaultSize, wxBU_EXACTFIT);
  m_CancelBtn = new wxButton(this, wxID_CANCEL, _T("&Cancel"));
  m_OpenBtn = new wxButton(this, wxID_OK, _T("&Open"));
  m_OpenBtn->SetDefault();

  // populate sizers
  m_FilterSizer = new wxBoxSizer(wxHORIZONTAL);
  m_FilterSizer->Add(new wxStaticText(this, -1, _T("Car")), wxSizerFlags(1).Proportion(0).Center());
  m_FilterSizer->AddSpacer(BASE_MARGIN);
  m_FilterSizer->Add(m_CarChoice, wxSizerFlags(1).Proportion(0).Center());
  m_FilterSizer->AddSpacer(4 * BASE_MARGIN);
  m_FilterSizer->Add(new wxStaticText(this, -1, _T("Track")), wxSizerFlags(1).Proportion(0).Center());
  m_FilterSizer->AddSpacer(BASE_MARGIN);
  m_FilterSizer->Add(m_TrackChoice, wxSizerFlags(1).Proportion(0).Center());
  m_FilterSizer->AddStretchSpacer();
  m_FilterSizer->Add(m_AllBtn);

  m_ButtonSizer = new wxBoxSizer(wxHORIZONTAL);
  m_ButtonSizer->Add(m_FolderBtn);
  m_ButtonSizer->AddSpacer(BASE_MARGIN);
  m_ButtonSizer->Add(m_DeleteBtn);
  m_ButtonSizer->AddSpacer(BASE_MARGIN);
  m_ButtonSizer->Add(m_RenameBtn);
  m_ButtonSizer->AddStretchSpacer();
  m_ButtonSizer->Add(m_CancelBtn);
  m_ButtonSizer->AddSpacer(BASE_MARGIN);
  m_ButtonSizer->Add(m_OpenBtn);

  m_TopSizer = new wxBoxSizer(wxVERTICAL);
  m_TopSizer->Add(m_FilterSizer, wxSizerFlags(1).Expand().Proportion(0).Border(wxALL, BASE_MARGIN));
  m_TopSizer->Add(m_List, wxSizerFlags(1).Expand().Proportion(1).Border(wxALL, BASE_MARGIN));
  m_TopSizer->Add(m_ButtonSizer, wxSizerFlags(1).Expand().Proportion(0).Border(wxALL, BASE_MARGIN));

  SetSizer(m_TopSizer);

  EnableButtons();

  // context menu on list
  m_Context.Append(ID_MENU_RENAME, _T("Rename"));
  m_Context.Append(ID_MENU_DELETE, _T("Delete"));
  m_Context.AppendSeparator();
  m_Context.Append(ID_MENU_SELECTALL, _T("Select all\tCtrl+A"));
}

//-----------------------------------------------------------------------------

cFileDlg::~cFileDlg()
{
}

//-----------------------------------------------------------------------------
// Get the names of the selected files

void cFileDlg::GetPaths(wxArrayString& paths) const
{
  long item = m_List->GetFirstSelected();

  while (item >= 0) {
    cFileDlgItemData* data = (cFileDlgItemData*)m_List->GetItemData(item);
    paths.Add(data->filename);
    item = m_List->GetNextSelected(item);
  }
}

//-----------------------------------------------------------------------------
// Read the RAF files and fill the file list

void cFileDlg::ReadFiles()
{
  if (m_Dir.IsEmpty()) m_Dir = wxGetCwd();

  m_List->Freeze();

  // reset current data
  m_List->DeleteAllItems();
  for (size_t i = 0; i < m_ItemData.GetCount(); i++) delete m_ItemData[i];
  m_ItemData.Clear();

  wxArrayString carCodes;         // the list of car codes that have been found
  wxArrayString trackCodes;       // the list of track codes that have been found

  // enumerate files
  wxArrayString fileList;
  wxDir::GetAllFiles(m_Dir, &fileList, _T("*.raf"), wxDIR_FILES);
  cFile file;
  for (size_t f = 0; f < fileList.GetCount(); f++) {
    // read data from file
    if (!file.Open(fileList[f])) continue;
    wxString header;
    file.ReadString(header, 6);
    file.Skip(2); // game version, game revision
    wxInt8 format;
    file.ReadByte(format);
    if ((header != "LFSRAF") || (format != 2)) {
      // not an LFS RAF file, or unknown format version
      file.Close();
      continue;
    }
    file.GoTo(28);
    float trackLength;
    file.ReadFloat(trackLength);
    wxString playerName;
    file.ReadString(playerName, 32);
    UnEscapeLFS(playerName);
    wxString car;
    file.ReadString(car, 32);
    wxString track;
    file.ReadString(track, 32);
    wxString config;
    file.ReadString(config, 16);
    file.GoTo(171);
    wxInt8 splits;
    file.ReadByte(splits);
    file.GoTo(172 + (splits - 1) * 4);
    wxInt32 lapTime;
    file.ReadInt(lapTime);
    file.Close();

    // get car and track codes
    wxString carCode = cCar::Name2Code(car);
    if (carCodes.Index(carCode) == wxNOT_FOUND) carCodes.Add(carCode);
    wxString trackCode = cTrack::Name2Code(track, config);
    wxString trackDesc = trackCode + _T(" - ") + config;
    if ((!m_CurrentTrack.IsEmpty()) && (trackCode != m_CurrentTrack)) continue;
    if (trackCodes.Index(trackDesc) == wxNOT_FOUND) trackCodes.Add(trackDesc);

    // apply filters
    if ((!m_CarFilter.IsEmpty()) &&
        (carCode != m_CarFilter.BeforeFirst(_T(' ')))) continue;
    if ((m_CurrentTrack.IsEmpty()) && (!m_TrackFilter.IsEmpty()) &&
        (trackCode != m_TrackFilter.BeforeFirst(_T(' ')))) continue;

    // add file to list
    size_t item = m_List->GetItemCount();
    m_List->InsertItem(item, wxEmptyString);
    wxString lapName;
    wxFileName::SplitPath(fileList[f], NULL, NULL, &lapName, NULL);
    m_List->SetItem(item, COL_FILENAME, lapName);
    m_List->SetItem(item, COL_CARCODE, carCode);
    m_List->SetItem(item, COL_TRACKCODE, trackCode);
    m_List->SetItem(item, COL_LAPTIME, ::FormatTime(lapTime));
    wxDateTime fileDate(wxFileModificationTime(fileList[f]));
    m_List->SetItem(item, COL_DATE, fileDate.FormatISODate());
    m_List->SetItem(item, COL_PLAYER, playerName);

    // show files that shouldn't be loaded in grey:
    // 1. files that are already loaded
    // 2. files that can't be loaded
    if (((!m_CurrentTrack.IsEmpty()) && (MGR->IsLoaded(fileList[f]))) ||
        (trackLength <= 0.0f)) {
      m_List->SetItemTextColour(item, wxColour(MEDIUM_GREY, MEDIUM_GREY, MEDIUM_GREY));
    }

    // attach item data
    cFileDlgItemData* data = new cFileDlgItemData;
    data->filename = fileList[f];
    data->name = lapName;
    data->car = carCode;
    data->track = trackCode;
    data->laptime = lapTime;
    data->filedate = fileDate.FormatISODate();
    data->player = playerName;
    m_ItemData.Add(data);
    m_List->SetItemData(item, (long)data);
  }

  ::SetColumnWidths(m_List);
  DoSort();
  if (m_List->GetItemCount() > 0) m_List->Select(0);

  // re-fill car choice
  carCodes.Sort();
  carCodes.Insert(_T("(all)"), 0);
  m_CarChoice->Clear();
  int selected = 0;
  for (size_t c = 0; c < carCodes.GetCount(); c++) {
    m_CarChoice->Append(carCodes[c]);
    if (m_CarFilter == carCodes[c].BeforeFirst(_T(' '))) selected = c;
  }
  m_CarChoice->SetSelection(selected);

  // re-fill track choice
  m_TrackChoice->Clear();
  trackCodes.Sort();
  if (m_CurrentTrack.IsEmpty()) trackCodes.Insert(_T("(all)"), 0);
  selected = 0;
  for (size_t t = 0; t < trackCodes.GetCount(); t++) {
    m_TrackChoice->Append(trackCodes[t]);
    if (m_TrackFilter == trackCodes[t].BeforeFirst(_T(' '))) selected = t;
  }
  m_TrackChoice->SetSelection(selected);

  // ready
  EnableButtons();
  m_List->Thaw();
}

//-----------------------------------------------------------------------------
// Comparison function, called by wxListCtrl::SortItems

int wxCALLBACK cFileDlg_Compare(long item1, long item2, long key)
{
  cFileDlgItemData* data1 = (cFileDlgItemData*)item1;
  cFileDlgItemData* data2 = (cFileDlgItemData*)item2;
  wxASSERT(data1 != NULL);
  wxASSERT(data2 != NULL);

  int result = 0;

  int column = abs(key) - 1;
  switch (column) {
    case COL_FILENAME :
      result = data1->name.CmpNoCase(data2->name);
      break;
    case COL_CARCODE :
      result = data1->car.CmpNoCase(data2->car);
      break;
    case COL_TRACKCODE :
      result = data1->track.CmpNoCase(data2->track);
      break;
    case COL_LAPTIME :
      result = data1->laptime - data2->laptime;
      break;
    case COL_DATE :
      result = data1->filedate.CmpNoCase(data2->filedate);
      break;
    case COL_PLAYER :
      result = data1->player.CmpNoCase(data2->player);
      break;
    default :
      wxFAIL;
  }

  if (key < 0) result = -result; // reversed sorting order

  // if no difference then sort on lap name
  if (result == 0) result = data1->name.CmpNoCase(data2->name);

  return result;
}

//-----------------------------------------------------------------------------
// Sort the list of files on the current column

void cFileDlg::DoSort()
{
  if (m_List->GetItemCount() == 0) return; // nothing to do

  // un-select any selected item in the list
  long item = m_List->GetFirstSelected();
  while (item >= 0) {
    m_List->Select(item, false);
    item = m_List->GetNextSelected(item);
  }

  // do the sorting
  m_List->SortItems(cFileDlg_Compare, m_SortKey);
}

//-----------------------------------------------------------------------------
// A column in the list was clicked

void cFileDlg::OnColClick(wxListEvent& event)
{
  long col = event.GetColumn();

  if ((col < 0) || (col >= m_List->GetColumnCount())) {
    event.Skip(); // no valid column
    return;
  }

  int key = col + 1;
  if (abs(m_SortKey) == key) {
    // already sorted on this column - reverse order
    key = -key;
  }
  else {
    // sort on new column
    if (col == COL_DATE) m_SortKey = -m_SortKey; // for "Date" column: start with descending order
  }
  SetSortKey(key);
}

//-----------------------------------------------------------------------------
// Set the sort key
// - key = new sort key (see declaration of m_SortKey for description)

void cFileDlg::SetSortKey(int key)
{
  // remove the indicator of the current sorted column
  int oldcol = abs(m_SortKey) - 1;
  wxString text = GetColumnText(oldcol);
  if ((text.Left(2) == SORT_INDICATOR_ASC) || (text.Left(2) == SORT_INDICATOR_DESC)) {
    text = text.Mid(2);
    SetColumnText(oldcol, text);
  }

  m_SortKey = key;
  DoSort();

  // set the indicator of the new sorted column
  int newcol = abs(m_SortKey) - 1;
  text = GetColumnText(newcol);
  if (m_SortKey > 0) {
    text = SORT_INDICATOR_ASC + text;
  }
  else {
    text = SORT_INDICATOR_DESC + text;
  }
  SetColumnText(newcol, text);
  ::SetColumnWidths(m_List);
}

//-----------------------------------------------------------------------------
// One or more items in the list were activated (by double-click or ENTER)

void cFileDlg::OnItemActivate(wxListEvent& WXUNUSED(event))
{
  if (IsModal() && (m_List->GetFirstSelected() >= 0)) EndModal(wxID_OK);
}

//-----------------------------------------------------------------------------

void cFileDlg::OnButton(wxCommandEvent& event)
{
  switch (event.GetId()) {
    case ID_BTN_ALL :
      m_CarFilter = wxEmptyString;
      m_TrackFilter = wxEmptyString;
      ReadFiles();
      break;

    case ID_BTN_FOLDER :
      {
        wxString folder = m_Dir;
        wxDirDialog dirDlg(this, _T("Select RAF folder"), folder);
        if (dirDlg.ShowModal() != wxID_OK) return; // no choice was made

        folder = dirDlg.GetPath();
        if (folder == m_Dir) return; // no change

        m_Dir = folder;
        m_CarFilter = wxEmptyString;
        m_TrackFilter = wxEmptyString;
        ReadFiles();
        Layout();
      }
      break;

    case ID_BTN_DELETE :
      DeleteSelection();
      break;

    case ID_BTN_RENAME :
      RenameSelection();
      break;

    case wxID_OK :
      if (IsModal() && (m_List->GetFirstSelected() >= 0)) EndModal(wxID_OK);
      break;

    default :
      event.Skip();
      return;
  }
}

//-----------------------------------------------------------------------------

void cFileDlg::OnChangeCarFilter(wxCommandEvent& WXUNUSED(event))
{
  if (m_CarChoice->GetSelection() == 0) {
    m_CarFilter = wxEmptyString;
  }
  else {
    m_CarFilter = m_CarChoice->GetStringSelection().BeforeFirst(_T(' '));
  }
  ReadFiles();
}

//-----------------------------------------------------------------------------

void cFileDlg::OnChangeTrackFilter(wxCommandEvent& WXUNUSED(event))
{
  if (m_TrackChoice->GetSelection() == 0) {
    m_TrackFilter = wxEmptyString;
  }
  else {
    m_TrackFilter = m_TrackChoice->GetStringSelection().BeforeFirst(_T(' '));
  }
  ReadFiles();
}

//-----------------------------------------------------------------------------
// Prepare things and show the (modal) dialog
// - caption = dialog caption
// - directory = directory with RAF files
// - add = add files to already loaded files (or clear all and load new ones)?

int cFileDlg::DoShowModal(const wxString& caption, const wxString& directory, bool add)
{
  SetTitle(caption);
  m_Dir = directory;
  if (add) {
    m_OpenBtn->SetLabel(_T("&Add"));
    m_CurrentTrack = MGR->GetTrackCode();
  }
  else {
    m_OpenBtn->SetLabel(_T("&Open"));
    m_CurrentTrack = wxEmptyString;
  }
  ReadFiles();

  return ShowModal();
}

//-----------------------------------------------------------------------------
// Get/Set the text of a column of the file list

wxString cFileDlg::GetColumnText(int col) const
{
  wxListItem info;
  info.m_mask = wxLIST_MASK_TEXT;
  info.m_itemId = 0;

  if (!m_List->GetColumn(col, info)) return wxEmptyString;
  return info.m_text;
}

void cFileDlg::SetColumnText(int col, const wxString& text)
{
  wxListItem item;
  item.m_mask = wxLIST_MASK_TEXT;
  m_List->GetColumn(col, item);
  item.SetText(text);
  m_List->SetColumn(col, item);
}

//-----------------------------------------------------------------------------
// Enable or disable the controls

void cFileDlg::EnableButtons()
{
  wxArrayString selection;
  GetPaths(selection);

  m_DeleteBtn->Enable(selection.GetCount() > 0);
  m_RenameBtn->Enable(selection.GetCount() == 1);
}

//-----------------------------------------------------------------------------

void cFileDlg::OnKeyDown(wxListEvent& event)
{
  switch (event.GetKeyCode()) {
    case WXK_DELETE :
      // Delete = delete selected items
      DeleteSelection();
      break;

    case WXK_F2 :
      // F2 = rename selected item
      RenameSelection();
      break;

    case 'A' :
      // Ctrl-A = select all
      if (!wxGetKeyState(WXK_CONTROL)) {
        event.Skip();
        return;
      }
      SelectAll();
      break;

    default :
      event.Skip();
      return;
  }
}

//-----------------------------------------------------------------------------
// Delete all selected files (after confirmation by user)

void cFileDlg::DeleteSelection()
{
  // get names of selected files
  wxArrayString selection;
  GetPaths(selection);
  if (selection.IsEmpty()) return;

  // ask user to confirm deletion
  wxString prompt;
  if (selection.GetCount() == 1) {
    // mention name of the selected item
    prompt = wxString::Format(_T("Delete '%s'?"), m_List->GetItemText(m_List->GetFirstSelected()).c_str());
  }
  else {
    // mention number of selected items
    prompt = wxString::Format(_T("Delete %d files?"), selection.GetCount());
  }
  if (wxMessageBox(prompt, (_T("Warning")), wxYES_NO | wxNO_DEFAULT | wxICON_QUESTION, this) == wxNO) return;

  // delete files
  for (size_t f = 0; f < selection.GetCount(); f++) {
    ::wxRemoveFile(selection[f]);
  }

  // renew list
  ReadFiles();
}

//-----------------------------------------------------------------------------
// Rename the selected file

void cFileDlg::RenameSelection()
{
  // get names of selected file
  wxArrayString selection;
  GetPaths(selection);
  if (selection.GetCount() != 1) return;

  // get display name of selected file
  wxString dispName = m_List->GetItemText(m_List->GetFirstSelected());

  // prompt for new name
  wxTextEntryDialog dlg(this, _T("New name for file"), _T("Rename file"), dispName);
  if (dlg.ShowModal() == wxID_CANCEL) return;

  wxString newName = m_Dir + wxFILE_SEP_PATH + dlg.GetValue() + _T(".raf");
  if (::wxFileExists(newName) && (selection[0].CmpNoCase(newName) != 0)) {
    wxMessageBox(_T("File already exists"), (_T("Error renaming file")),
        wxOK | wxICON_ERROR, this);
    return;
  }

  // rename file
  ::wxRenameFile(selection[0], newName);

  // renew list
  ReadFiles();
}

//-----------------------------------------------------------------------------
// The right mouse button was clicked over an item in the list

void cFileDlg::OnItemRightClick(wxListEvent& WXUNUSED(event))
{
  wxArrayString selection;
  GetPaths(selection);

  m_Context.Enable(ID_MENU_RENAME, selection.GetCount() == 1);
  m_Context.Enable(ID_MENU_DELETE, !selection.IsEmpty());
  m_Context.Enable(ID_MENU_SELECTALL, !selection.IsEmpty());
  PopupMenu(&m_Context);
}

//-----------------------------------------------------------------------------
// Handle a click in the context menu

void cFileDlg::OnMenuClick(wxCommandEvent& event)
{
  switch (event.GetId()) {
    case ID_MENU_RENAME :
      RenameSelection();
      break;

    case ID_MENU_DELETE :
      DeleteSelection();
      break;

    case ID_MENU_SELECTALL :
      SelectAll();
      break;

    default :
      wxFAIL;
  }
}

//-----------------------------------------------------------------------------
// Loading and saving the configuration settings

void cFileDlg::LoadConfig(wxRegConfig* config, const wxString& key)
{
  config->Read(key + _T("/car_filter"), &m_CarFilter);
  config->Read(key + _T("/track_filter"), &m_TrackFilter);

  int sortkey;
  config->Read(key + _T("/sortkey"), &sortkey);
  SetSortKey(sortkey);

  int sizeX, sizeY;
  GetSize(&sizeX, &sizeY);
  config->Read(key + _T("/size/x"), &sizeX);
  config->Read(key + _T("/size/y"), &sizeY);
  SetSize(sizeX, sizeY);
}


void cFileDlg::SaveConfig(wxRegConfig* config, const wxString& key)
{
  config->Write(key + _T("/car_filter"), m_CarFilter);
  config->Write(key + _T("/track_filter"), m_TrackFilter);
  config->Write(key + _T("/size/x"), GetSize().GetWidth());
  config->Write(key + _T("/size/y"), GetSize().GetHeight());
  config->Write(key + _T("/sortkey"), m_SortKey);
}
