/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
*/

#ifndef INSIM
#define INSIM

#ifdef WIN32
#include <windows.h>
#else
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <fcntl.h>
#define closesocket(s) close(s)
#define SOCKET_ERROR -1
#define INVALID_SOCKET -1
typedef enum {FALSE, TRUE} BOOL; //notice, order matters
#endif
#include <time.h>
#include <stdio.h>

#include "events.h"
#include "version.h"

#define INSIM_TIMEOUT 120
#define INSIM_KEEPALIVE 30
#define INSIM_KEEPALIVE_MAX 60
#define PACKET_BUFFER_SIZE 512

struct socket_info_t
{
    /* socket */
    int s;
	/* How long before we timeout select */
	struct timeval select_timeout;
	/* Last time we sent a keepalive to LFS */
	time_t last_keepalive;
	/* Last time LFS contacted us */
	time_t last_contact;
	/* How long before we assume we've lost network connectivity with LFS */
	unsigned int timeout;
	/* How long before we send keepalive packets, in seconds */
	unsigned int keepalive;
};

struct lfs_info_t
{
	char *address;
	unsigned short port;
	char *admin_password;
	unsigned char flags;
	unsigned char node_secs;
	unsigned char prefix;
};

struct packet_buffer_t
{
    /* packet buffer - 512 should be more than enough */
	char buffer[PACKET_BUFFER_SIZE];
	/* number of bytes currently in buffer */
	unsigned int bytes;
};

struct internals_t
{
    /* path to config.lua and script directories */
    char *path;
    /* log file pointer */
    char *log;
    /* number of connection attempts */
    unsigned int connection_attempts;
    /* auto reconnect? */
    BOOL reconnect;
	/* Spam-factor */
	unsigned int verbose;
	/* do we quit, true = yes, false = no */
	BOOL quit;
};

struct insim_t
{
    struct socket_info_t socket;
    struct lfs_info_t lfs;
    struct internals_t internals;
    struct packet_buffer_t pbuf;
};

struct insim_t *insim_create();
int insim_destroy(struct insim_t *I);

int insim_connect(struct insim_t *I);
int insim_recv(struct insim_t *I);
int insim_disconnect(struct insim_t *I);

int insim_send(struct insim_t *I, const char *buf, unsigned int size);

int insim_init(struct insim_t *I);
int insim_close(struct insim_t *I);

#define ISF_RACE_TRACKING	1	// bit 0 : turns on race tracking
#define ISF_RESERVED_1		2	// bit 1 : spare
#define ISF_RESERVED_2		4	// bit 2 : spare
#define ISF_NO_WARNINGS		8	// bit 3 : turns off packet warnings
#define ISF_KEEP_ALIVE		16	// bit 4 : send keep alive packets
#define ISF_NLP_MCI			32	// bit 5 : send MCI instead of NLP

typedef unsigned char byte;
typedef short word;

struct IS_ISI // InSim Init - packet to initialise the InSim system
{
	byte	size;		// 44
	byte	type;		// ISP_ISI
	byte	reqI;		// If non-zero LFS will send an IS_VER packet
	byte	zero;		// 0

	word	udpport;	// Port for UDP replies from LFS (0 to 65535)
	word	flags;		// Bit flags for options (see below)

	byte	sp0;		// 0
	byte	prefix;		// Special host message prefix character
	word	interval;	// Time in ms between NLP or MCI (0 = none)

	char	admin[16];	// Admin password (if set in LFS)
	char	iname[16];	// A short name for your program
};

// IS_TINY - used for various requests, replies and reports
struct IS_TINY // General purpose 4 byte packet
{
	byte size;		// always 4
	byte type;		// always ISP_TINY
	byte reqI;		// 0 unless it is an info request or a reply to an info request
	byte subT;		// subtype, from TINY_ enumeration (e.g. TINY_RACE_END)
};

// IS_SMALL - used for various requests, replies and reports
struct IS_SMALL // General purpose 8 byte packet
{
    byte size;		// always 8
	byte type;		// always ISP_SMALL
	byte reqI;		// 0 unless it is an info request or a reply to an info request
	byte subT;		// subtype, from SMALL_ enumeration (e.g. SMALL_SSP)

	unsigned UVal;	// value (e.g. for SMALL_SSP this would be the OutSim packet rate)
};

struct IS_VER // VERsion
{
	byte	size;			// 20
	byte	type;			// ISP_VERSION
	byte	reqI;			// ReqI as received in the request packet
	byte	zero;
	char	version	[8];	// LFS version, e.g. 0.3G
	char	product	[6];	// Product : DEMO or S1
	word	insimver;		// InSim Version : increased when InSim packets change
};

enum // the second byte in all packets is one of these
{
	ISP_NONE,		//  0					: not used
	ISP_ISI,		//  1 - instruction		: insim initialise
	ISP_VER,		//  2 - info			: version info
	ISP_TINY,		//  3 - both ways		: multi purpose
	ISP_SMALL,		//  4 - both ways		: multi purpose
	ISP_STA,		//  5 - info			: state info
	ISP_SCH,		//  6 - instruction		: single character
	ISP_SFP,		//  7 - instruction		: state flags pack
	ISP_SCC,		//  8 - instruction		: set car camera
	ISP_CPP,		//  9 - both ways		: cam pos pack
	ISP_ISM,		// 10 - info			: start multiplayer
	ISP_MSO,		// 11 - info			: message out
	ISP_III,		// 12 - info			: hidden /i message
	ISP_MST,		// 13 - instruction		: type message or /command
	ISP_MTC,		// 14 - instruction		: message to a connection
	ISP_MOD,		// 15 - instruction		: set screen mode
	ISP_VTN,		// 16 - info			: vote notification
	ISP_RST,		// 17 - info			: new connection
	ISP_NCN,		// 18 - info			: new connection
	ISP_CNL,		// 19 - info			: connection left
	ISP_CPR,		// 20 - info			: connection renamed
	ISP_NPL,		// 21 - info			: new player (joined race)
	ISP_PLP,		// 22 - info			: player pit (keeps slot in race)
	ISP_PLL,		// 23 - info			: player leave (spectate - loses slot)
	ISP_LAP,		// 24 - info			: lap time
	ISP_SPX,		// 25 - info			: split x time
	ISP_PIT,		// 26 - info			: pit stop start
	ISP_PSF,		// 27 - info			: pit stop finish
	ISP_PLA,		// 28 - info			: pit lane enter / leave
	ISP_CCH,		// 29 - info			: camera changed
	ISP_PEN,		// 30 - info			: penalty given or cleared
	ISP_TOC,		// 31 - info			: take over car
	ISP_FLG,		// 32 - info			: flag (yellow or blue)
	ISP_PFL,		// 33 - info			: player flags (help flags)
	ISP_FIN,		// 34 - info			: finished race
	ISP_RES,		// 35 - info			: result confirmed
	ISP_REO,		// 36 - both ways		: reorder (info or instruction)
	ISP_NLP,		// 37 - info			: node and lap packet
	ISP_MCI,		// 38 - info			: multi car info
};

enum // the fourth byte of IS_TINY packets is one of these
{
	TINY_NONE,		//  0					: see "maintaining the connection"
	TINY_VER,		//  1 - info request	: get version
	TINY_CLOSE,		//  2 - instruction		: close insim
	TINY_PING,		//  3 - ping request	: external progam requesting a reply
	TINY_REPLY,		//  4 - ping reply		: reply to a ping request
	TINY_VTC,		//  5 - info			: vote cancelled
	TINY_SCP,		//  6 - info request	: send camera pos
	TINY_SST,		//  7 - info request	: send state info
	TINY_GTH,		//  8 - info request	: get time in hundredths (i.e. SMALL_RTP)
	TINY_MPE,		//  9 - info			: multi player end
	TINY_ISM,		// 10 - info request	: get multiplayer info (i.e. ISP_ISM)
	TINY_REN,		// 11 - info			: race end (return to game setup screen)
	TINY_CLR,		// 12 - info			: all players cleared from race
	TINY_NCN,		// 13 - info			: get all connections
	TINY_NPL,		// 14 - info			: get all players
	TINY_RES,		// 15 - info			: get all results
	TINY_NLP,		// 16 - info request	: send an IS_NLP packet
	TINY_MCI,		// 17 - info request	: send an IS_MCI packet
	TINY_REO,		// 18 - info request	: send an IS_REO packet
	TINY_RST,		// 19 - info request	: send an IS_RST packet
};

enum // the fourth byte of IS_SMALL packets is one of these
{
	SMALL_NONE,		//  0					: not used
	SMALL_SSP,		//  1 - instruction		: start sending positions
	SMALL_SSG,		//  2 - instruction		: start sending gauges
	SMALL_VTA,		//  3 - report			: vote action
	SMALL_TMS,		//  4 - instruction		: time stop
	SMALL_STP,		//  5 - instruction		: time step
	SMALL_RTP,		//  6 - info			: race time packet (reply to GTH)
	SMALL_NLI,		//  7 - instruction		: set node lap interval
};

#endif
