/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
*/

#include "config.h"

int load_conf(struct insim_t *I, lua_State *L)
{
    printf("Loading config\n");
    if (luaL_dofile(L, "config.lua") > 0)
        fprintf(stderr, "Config not OK (%d)\n", lua_error(L));

    lua_getglobal(L, "config");
	if (!lua_istable(L, -1))
	{
        fprintf(stderr, "Config: variable not found or not a table\n");
		return -1;
	}

    /* Read the luaLFS conf */
	lua_pushstring(L, "luaLFS");
    lua_gettable(L, -2);
	if (!lua_istable(L, -1))
	{
		fprintf(stderr, "Config: Error, luaLFS entries are not a table\n");
		return -1;
	}

    I->socket.select_timeout.tv_sec = conf_gettable_num(L, "polling", 5);
    I->socket.select_timeout.tv_usec = 0;

    I->socket.timeout = conf_gettable_num(L, "timeout", INSIM_TIMEOUT);
    if (I->socket.timeout <= 0)
        I->socket.timeout = INSIM_TIMEOUT;

    I->socket.keepalive = conf_gettable_num(L, "keepalive", 30);

    I->internals.connection_attempts = conf_gettable_num(L, "connection_attempts", 1);
    I->internals.reconnect = conf_gettable_num(L, "auto_reconnect", 0);

    I->internals.verbose = conf_gettable_num(L, "verbose", 0);

    /*
    Theoretically the timeout could occur during a period LFS is quiet, and therefore select is blocking
    We want to prevent that, so check that the keepalive + the maximum select timeout does not exceed the
    "safe" limit of INSIM_KEEPALIVE_MAX
    */
    if ((I->socket.keepalive <= 0) || ((I->socket.keepalive + I->socket.select_timeout.tv_sec) > INSIM_KEEPALIVE_MAX))
        I->socket.keepalive = INSIM_KEEPALIVE;

    /* go back to "root" of config */
    lua_pop(L, 1);

    /* Read the LFS conf */
	lua_pushstring(L, "LFS");
    lua_gettable(L, -2);
	if (!lua_istable(L, -1))
	{
		fprintf(stderr, "Config: Error, LFS entries are not a table\n");
		return -1;
	}

    char *p = NULL;
    size_t psize = 0;
    p = conf_gettable_str(L, "address");
    psize = strlen(p) + 1;

    /* copy the address into our own memory space, where we can ensure that the address stays */
    I->lfs.address = malloc(psize);
    memset(I->lfs.address, 0, psize);
    memcpy(I->lfs.address, p, psize - 1);

    I->lfs.port = conf_gettable_num(L, "port", 29999);

    p = conf_gettable_str(L, "password");
    psize = strlen(p) + 1;

    /* copy the address into our own memory space, where we can ensure that the address stays */
    I->lfs.admin_password = malloc(psize);
    memset(I->lfs.admin_password , 0, psize);
    memcpy(I->lfs.admin_password , p, psize - 1);

    I->lfs.flags = ISF_RACE_TRACKING | ISF_KEEP_ALIVE;
    I->lfs.node_secs = conf_gettable_num(L, "nodesecs", 0);

    char *tprefix = conf_gettable_str(L, "prefix");
    if (strlen(tprefix) == 1)
        I->lfs.prefix = (unsigned char)tprefix[0];

    time_t now;
    now = time(NULL);

    I->socket.last_contact = now;
    I->socket.last_keepalive = now;
	return 0;
}

int conf_gettable_num(lua_State *L, const char *name, int idefault)
{
    int r = 0;

    lua_pushstring(L, name);
	lua_gettable(L, -2);

	if (!lua_isnumber(L, -1))
	{
		fprintf(stderr, "Config: '%s' is either not number, or does not exist. Falling back to default of '%d'.\n", name, idefault);
		r = idefault;
	}
	else
	{
        r = lua_tonumber(L, -1);
	}

	/* back a level */
	lua_pop(L, 1);

	return r;
}

char *conf_gettable_str(lua_State *L, const char *name)
{
    char *r;
    char *sdefault = NULL;

    lua_pushstring(L, name);
	lua_gettable(L, -2);

	if (!lua_isstring(L, -1))
	{
		fprintf(stderr, "Config: '%s' is either not string, or does not exist. Falling back to default of '%s'.\n", name, sdefault);
		r = sdefault;
	}
	else
	{
        r = (char *)lua_tostring(L, -1);
	}

	/* back a level */
	lua_pop(L, 1);

	return r;
}
