/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
*/

#include "insim.h"
#include "ilua.h"

#ifndef WIN32
#include <unistd.h>
#endif
insim_t *insim_create()
{
	insim_t *I = malloc(sizeof(insim_t));
	if (I != NULL)
		return I;

	return NULL;
}

int insim_destroy(insim_t *I)
{
	if (I != NULL)
		free(I);
	return 0;
}

int insim_connect(insim_t *I)
{
	if (insim_init(I) < 0)
        return -1;

	struct ISI p;
	memset(&p, 0, sizeof(struct ISI));
	strcpy(p.id, "ISI");
	p.port = I->listen_port;
	p.flags = I->flags;
	p.nodesecs = I->node_secs;

	if (insim_send(I, (char *)&p, sizeof(struct ISI)) < 0)
        return -1;

	struct ISP v;
	memset(&v, 0, sizeof(struct ISP));
	strcpy((char *)&v.id, "VER");
	v.value = 0;
	if (insim_send(I, (char *)&v, sizeof(struct ISP)) < 0)
        return -1;

	return 0;
}

int insim_recv(insim_t *I)
{
	struct ISP a;
	memset(&a, 0, sizeof(struct ISP));
	strcpy((char *)&a.id, "ACK");
	a.value = 0;

	int rc;

	FD_ZERO(&(I->ssets));
	FD_SET(I->socket, &(I->ssets));

	rc = select(I->socket+1, &(I->ssets), NULL, NULL, &(I->select_timeout));

	if (rc < 0)
	{
	    fprintf(stderr, "insim_recv: Select error!\n");
	    return -1;
	}

    time_t now;
    now = time(NULL);

    if (difftime(now, I->last_keepalive) > I->keepalive)
    {
        /* tell LFS we're still here */
        insim_send(I, (char *)&a, sizeof(struct ISP));
        I->last_keepalive = now;
    }

	if (rc == 0)
	{
        // Timeout
        if (difftime(now, I->last_contact) > I->timeout)
        {
            fprintf(stderr, "insim_recv: Connection Timeout\n");
            return -1;
        }

        if (!I->gotversion)
        {
            fprintf(stderr, "insim_recv: No response from LFS yet. It's probably not running\n");
            return -1;
        }

	    return 1;
	}

	/* read socket for data */
	char buf[512];
	int retval = recv(I->socket, buf, 512, 0);
	if (retval > 0)
	{
	    I->last_contact = now;

		if (strcmp(buf, "VER") == 0)
		{
			struct ISV r;
			memcpy(&r, buf, sizeof(struct ISV));
			printf("Connected to LFS %s (%s), InSIM v%d\n", r.product, r.version, r.insimver);
			I->gotversion = TRUE;
            ilua_evt_fire(EVT_CONNECTED, NULL, 0);
		}
		else
		{
			    if (I->verbose)
                	printf("Receiving '%s'\n", buf);
			ilua_evt_fire(buf, buf, retval);
		}
	}

	return 1;
}

int insim_disconnect(insim_t *I)
{
	struct ISP p;
	memset(&p, 0, sizeof(struct ISP));
	strcpy((char *)&p.id, "ISC");
	p.value = 0;
	insim_send(I, (char *)&p, sizeof(struct ISP));
	insim_close(I);
	return (1);
}

int insim_send(insim_t *I, char *buf, int size)
{
	struct sockaddr_in temp_addr;
	unsigned int addr = inet_addr(I->send_address);

	memset(&temp_addr, 0, sizeof(temp_addr));
	memcpy(&temp_addr.sin_addr, &addr, sizeof(addr));
	temp_addr.sin_family = AF_INET;
	temp_addr.sin_port = htons(I->send_port);

	if (I->verbose)
        printf("Sending '%s'\n", buf);

	return sendto(I->socket, buf, size, 0, (struct sockaddr *)&temp_addr, sizeof(struct sockaddr));
}

int insim_init(insim_t *I)
{
	struct sockaddr_in saddr;
#ifdef WIN32
   	WSADATA wsadata;

	if (WSAStartup(0x202, &wsadata) == SOCKET_ERROR)
	{
		WSACleanup();
		return WSAGetLastError();
	}
#endif

	/* create our listen socket */
	I->socket = socket(AF_INET, SOCK_DGRAM, 0);
	if (I->socket == INVALID_SOCKET)
	{
		return -1;
	}

	/* make our socket non-blocking */
#ifdef WIN32
	unsigned long s_mode = 1;
	if (ioctlsocket(I->socket, FIONBIO, &s_mode) < 0)
	{
		return -1;
	}
#else
	int opts;
	opts = fcntl(I->socket, F_GETFL);
	if (opts < 0)
	{
		return -1;
	}

	opts = (opts | O_NONBLOCK);
	if (fcntl(I->socket, F_SETFL,opts) < 0)
	{
		return -1;
	}
#endif
	saddr.sin_family = AF_INET;
	saddr.sin_addr.s_addr = inet_addr(I->listen_address);
	saddr.sin_port = htons(I->listen_port);

	if (bind(I->socket, (struct sockaddr *)&saddr, sizeof(saddr)) == SOCKET_ERROR)
	{
		return -1;
	}
	return (1);
}

int insim_close(insim_t *I)
{
	if (I->socket != INVALID_SOCKET)
		closesocket(I->socket);

	I->socket = INVALID_SOCKET;
#ifdef WIN32
	WSACleanup();
#endif
	return INVALID_SOCKET;
}
