/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
*/

#include "config.h"

int load_conf(insim_t *I, lua_State *L)
{
    printf("Loading config\n");
    if (luaL_dofile(L, "config.lua") > 0)
        fprintf(stderr, "Config not OK (%d)\n", lua_error(L));

    lua_getglobal(L, "config");
	if (!lua_istable(L, -1))
	{
        fprintf(stderr, "Config: variable not found or not a table\n");
		return -1;
	}

    /* Read the luaLFS conf */
	lua_pushstring(L, "luaLFS");
    lua_gettable(L, -2);
	if (!lua_istable(L, -1))
	{
		fprintf(stderr, "Config: Error, luaLFS entries are not a table\n");
		return -1;
	}

    I->listen_address = conf_gettable_str(L, "address");
    I->listen_port = conf_gettable_num(L, "port");

    I->select_timeout.tv_sec = conf_gettable_num(L, "polling");
    I->select_timeout.tv_usec = 0;

    I->timeout = conf_gettable_num(L, "timeout");
    I->keepalive = conf_gettable_num(L, "keepalive");

    I->verbose = conf_gettable_num(L, "verbose");

    if (I->timeout <= 0)
        I->timeout = INSIM_TIMEOUT;

    /*
    Theoretically the timeout could occur during a period LFS is quiet, and therefore select is blocking
    We want to prevent that, so check that the keepalive + the maximum select timeout does not exceed the
    "safe" limit of INSIM_KEEPALIVE_MAX
    */
    if ((I->keepalive <= 0) || ((I->keepalive + I->select_timeout.tv_sec) > INSIM_KEEPALIVE_MAX))
        I->keepalive = INSIM_KEEPALIVE;

    /* go back to "root" of config */
    lua_pop(L, 1);

    /* Read the LFS conf */
	lua_pushstring(L, "LFS");
    lua_gettable(L, -2);
	if (!lua_istable(L, -1))
	{
		fprintf(stderr, "Config: Error, LFS entries are not a table\n");
		return -1;
	}

    I->send_address = conf_gettable_str(L, "address");
    I->send_port = conf_gettable_num(L, "port");
    I->admin_password = conf_gettable_str(L, "password");
    I->flags = ISF_RACE_TRACKING | ISF_KEEP_ALIVE;
    I->node_secs = conf_gettable_num(L, "nodesecs");

    time_t now;
    now = time(NULL);

    I->last_contact = now;
    I->last_keepalive = now;
	return (1);
}

int conf_gettable_num(lua_State *L, const char *name)
{
    int r = 0;
    int idefault = 0;

    lua_pushstring(L, name);
	lua_gettable(L, -2);

	if (!lua_isnumber(L, -1))
	{
		fprintf(stderr, "Config: '%s' is either not number, or does not exist. Falling back to default of '%d'.\n", name, idefault);
		r = 0;
	}
	else
	{
        r = lua_tonumber(L, -1);
	}

	/* back a level */
	lua_pop(L, 1);

	return r;
}

char *conf_gettable_str(lua_State *L, const char *name)
{
    char *r;
    char *sdefault = NULL;

    lua_pushstring(L, name);
	lua_gettable(L, -2);

	if (!lua_isstring(L, -1))
	{
		fprintf(stderr, "Config: '%s' is either not string, or does not exist. Falling back to default of '%s'.\n", name, sdefault);
		r = sdefault;
	}
	else
	{
        r = (char*) lua_tostring(L, -1);
	}

	/* back a level */
	lua_pop(L, 1);

	return r;
}
