/*
 * Copyright Karl Southern 2007
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
*/

#include "insim.h"

insim_t *insim_create()
{
	insim_t *I = malloc(sizeof(insim_t));
	if (I != NULL)
		return I;

	return NULL;
}

int insim_destroy(insim_t *I)
{
	if (I != NULL)
		free(I);
	return 0;
}

int insim_connect(insim_t *I)
{
	insim_init(I);

	struct ISI p;
	memset(&p, 0, sizeof(struct ISI));
	strcpy(p.id, "ISI");
	p.port = I->listen_port;
	p.flags = I->flags;
	p.nodesecs = I->node_secs;

	insim_send(I, (char *)&p, sizeof(struct ISI));

	struct ISP v;
	memset(&v, 0, sizeof(struct ISP));
	strcpy((char *)&v.id, "VER");
	v.value = 0;
	insim_send(I, (char *)&v, sizeof(struct ISP));
}

int insim_recv(insim_t *I)
{
	struct ISP a;
	memset(&a, 0, sizeof(struct ISP));
	strcpy((char *)&a.id, "ACK");
	a.value = 0;

	// check last time LFS sent us data, if > 2 mins, fail
	time_t now;
	now = time(NULL);
	if (difftime(now, I->last_contact) > INSIM_TIMEOUT)
	{
		printf("Connection Timeout\n");
		return -1;
	}

	// read socket for data
	char buf[512];
	int retval = recv(I->socket, buf, 512, 0);
	if (retval > 0)
	{
		if (strcmp(buf, "VER") == 0)
		{
			struct ISV r;
			memcpy(&r, buf, sizeof(struct ISV));
			printf("Connected to LFS %s (%s), InSIM v%d\n", r.product, r.version, r.insimver);

			ilua_evt_fire(EVT_CONNECTED, NULL);
		}
		else
		{
			ilua_evt_fire(buf, buf, retval);
		}
	}

	// tell LFS we're still here
	insim_send(I, (char *)&a, sizeof(struct ISP));

	// sleep, so we dont hammer the system
	Sleep(150);

	return 1;
}

int insim_disconnect(insim_t *I)
{
	struct ISP p;
	memset(&p, 0, sizeof(struct ISP));
	strcpy((char *)&p.id, "ISC");
	p.value = 0;
	insim_send(I, (char *)&p, sizeof(struct ISP));
	insim_close(I);
}

int insim_send(insim_t *I, char *buf, int size)
{
	struct sockaddr_in temp_addr;
	unsigned int addr = inet_addr(I->send_address);

	memset(&temp_addr, 0, sizeof(temp_addr));
	memcpy(&temp_addr.sin_addr, &addr, sizeof(addr));
	temp_addr.sin_family = AF_INET;
	temp_addr.sin_port = htons(I->send_port);

	return sendto(I->socket, buf, size, 0, (struct sockaddr *)&temp_addr, sizeof(struct sockaddr));
}

int insim_init(insim_t *I)
{
	struct sockaddr_in saddr;
#ifdef WIN32
   	WSADATA wsadata;

	if (WSAStartup(0x202, &wsadata) == SOCKET_ERROR)
	{
		WSACleanup();
		return WSAGetLastError();
	}
#endif

	/* create our listen socket */
	I->socket = socket(AF_INET, SOCK_DGRAM, 0);
	if (I->socket == INVALID_SOCKET)
	{
		return -1;
	}


	/* make our socket non-blocking */
#ifdef WIN32
	unsigned long s_mode = 1;
	if (ioctlsocket(I->socket, FIONBIO, &s_mode) < 0)
	{
		return -1;
	}
#else
	int opts;
	opts = fcntl(s, F_GETFL);
	if (opts < 0)
	{
		return -1;
	}

	opts = (opts | O_NONBLOCK);
	if (fcntl(s, F_SETFL,opts) < 0)
	{
		return -1;
	}
#endif

	saddr.sin_family = AF_INET;
	saddr.sin_addr.s_addr = inet_addr(I->listen_address);
	saddr.sin_port = htons(I->listen_port);

	if (bind(I->socket, (struct sockaddr *)&saddr, sizeof(saddr)) == SOCKET_ERROR)
	{
		return -1;
	}
}

int insim_close(insim_t *I)
{
	if (I->socket != INVALID_SOCKET)
		closesocket(I->socket);

	I->socket = INVALID_SOCKET;
#ifdef WIN32
	WSACleanup();
#endif
	return INVALID_SOCKET;
}
