// LFS_tcap.cpp : LFS telemetry capture using OutSim UDP packet.
//	Outputs to comma-delimited CSV files named LFS_tcap_yyyymmdd_hhmmss.csv
//	and binary files (replicating the OutSim packets) named LFS_tcap_yyyymmdd_hhmmss.bin.
//
// Uses my common-format CSV designed for benchmarking physics for five modern-era sims: AMS1, iRacing, LFS, rFactor2 and RRE.
//
// See Documents\Games\Racing\Sim Telemetry Channels.xlsx for CSV format and source channels for all sims.
//
// Configure OutSim by setting options in LFS root folder config file cfg.txt:
//
//		OutSim Mode 1			(generate telemetry while driving)
//		OutSim Opts ff			(all telemetry data)
//
// See LFS thread "Live telemetry data in test patch U9":	https://www.lfs.net/forum/thread/93701-Live-telemetry-data-in-test-patch-U9
// Requires LFS patch version U9 or higher:					https://www.lfs.net/forum/thread/93185
//

#pragma warning(disable : 4005)		// Suppress macro redefinition warning from Microsoft headers.
#pragma warning(disable : 4996)		// Suppress deprecated function warning from fprintf.

#include <math.h>					// Standard maths library.
#include <stdio.h>					// Standard input/output library - no iostream for me, printf all the way for console and CSV file output!
#include <time.h>					// Date and time library - used for generating unique CSV file names.
#include <winsock2.h>				// WinSock2 library (also needed to add WS2_32.lib to project dependencies).

#define BUFFER_SIZE 512				// Receive buffer size (basically maximum packet size in UDP).
#define HOST "127.0.0.1"			// Host to connect to.
#define PORT 30000					// Port to connect to the host through.

// Define types used by OutSimPack.
typedef unsigned char byte;
typedef unsigned short word;
typedef struct
{
	int X, Y, Z;
} Vec;
typedef struct
{
	float X, Y, Z;
} Vector;

// Include OutSimPack header.
#include "OutSimPack.h"

// Function prototypes.
void OutSimPacketReceived(const OutSimPack2 packet, FILE *tp);
Vector World2Local(Vector WorldVector, float Roll, float Pitch, float Yaw);

// External variables for derived channels that require calculation of inter-sample deltas
float PrevAngVelX = 0;				// AngVel.X from previous sample
float PrevAngVelY = 0;				// AngVel.Y from previous sample
float PrevAngVelZ = 0;				// AngVel.Z from previous sample
float PrevSuspDeflectLF = 0;		// SuspDeflectLF from previous sample
float PrevSuspDeflectRF = 0;		// SuspDeflectRF from previous sample
float PrevSuspDeflectLR = 0;		// SuspDeflectLR from previous sample
float PrevSuspDeflectRR = 0;		// SuspDeflectRR from previous sample
float PrevGroundSpeed = 0;			// GroundSpeed from previous sample
float GroundSpeed = 0;				// GroundSpeed for current sample

// Miscellaneous variables
int SessionTick = 0;				// Telemetry sample number - equivalent to iRacing's SessionTick channel

int main(int argc, char *argv[])
{
	FILE *tp;						// Telemetry CSV file.
	FILE *tp2;						// Telemetry binary file.
	time_t now;						// Current date/time (encoded).
	struct tm *dtp;					// Broken-down date/time structure.
	char file_name[29];				// CSV file name, format = LFS_tcap_yyyymmdd_hhmmss.csv
	char file_name2[29];			// bin file name, format = LFS_tcap_yyyymmdd_hhmmss.bin

	printf("Hello - LFS are you out there?\n");
	
	// Initialise WinSock version 2.2.
	WSADATA wsaData;
	if (WSAStartup(MAKEWORD(2, 2), &wsaData) != 0)
	{
		WSACleanup();
		printf("Error: Failed to init WinSock\n");
		return EXIT_FAILURE;
	}

	// Create UDP socket.
	SOCKET sock = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
	if (sock == INVALID_SOCKET)
	{
		WSACleanup();
		printf("Error: Could not create socket\n");
		return EXIT_FAILURE;
	}

	// Bind to receive UDP packets.
	sockaddr_in saddr;
	saddr.sin_family = AF_INET;
	saddr.sin_addr.s_addr = inet_addr(HOST);
	saddr.sin_port = htons(PORT);
	if (bind(sock, (sockaddr *)&saddr, sizeof(sockaddr)) == SOCKET_ERROR)
	{
		closesocket(sock);
		WSACleanup();
		printf("Error: Could not connect to LFS\n");
		return EXIT_FAILURE;
	}

	// Get current date/time
	now = time(NULL);
	dtp = localtime(&now);

	// Open CSV telemetry file, name format = LFS_tcap_yyyymmdd_hhmmss.csv
	sprintf(file_name, "LFS_tcap_%4d%02d%02d_%02d%02d%02d.csv", 1900 + dtp->tm_year, 1 + dtp->tm_mon, dtp->tm_mday, dtp->tm_hour, dtp->tm_min, dtp->tm_sec);
	tp = fopen(file_name, "wb");

	// Open bin telemetry file, name format = LFS_tcap_yyyymmdd_hhmmss.bin
	sprintf(file_name2, "LFS_tcap_%4d%02d%02d_%02d%02d%02d.bin", 1900 + dtp->tm_year, 1 + dtp->tm_mon, dtp->tm_mday, dtp->tm_hour, dtp->tm_min, dtp->tm_sec);
	tp2 = fopen(file_name2, "wb");

	// Write CSV header row 1 (channel names)			Col
	fprintf(tp, "SessionTime,");					//	  1
	fprintf(tp, "LapDist,");						//	  2
	fprintf(tp, "Speed,");							//	  3
	fprintf(tp, "Ground Acceleration,");			//	  4
	fprintf(tp, "VelocityX,");						//	  5
	fprintf(tp, "VelocityY,");						//	  6
	fprintf(tp, "VelocityZ,");						//	  7
	fprintf(tp, "LongAccel,");						//	  8
	fprintf(tp, "LatAccel,");						//	  9
	fprintf(tp, "VertAccel,");						//	 10
	fprintf(tp, "Roll,");							//	 11
	fprintf(tp, "Pitch,");							//	 12
	fprintf(tp, "YawNorth,");						//	 13
	fprintf(tp, "RollRate,");						//	 14
	fprintf(tp, "PitchRate,");						//	 15
	fprintf(tp, "YawRate,");						//	 16
	fprintf(tp, "Angular Acceleration X,");			//	 17
	fprintf(tp, "Angular Acceleration Y,");			//	 18
	fprintf(tp, "Angular Acceleration Z,");			//	 19
	fprintf(tp, "Gear,");							//	 20
	fprintf(tp, "RPM,");							//	 21
	fprintf(tp, "Force Long LF,");					//	 22
	fprintf(tp, "Force Long RF,");					//	 23
	fprintf(tp, "Force Long LR,");					//	 24
	fprintf(tp, "Force Long RR,");					//	 25
	fprintf(tp, "Force Lat LF,");					//	 26
	fprintf(tp, "Force Lat RF,");					//	 27
	fprintf(tp, "Force Lat LR,");					//	 28
	fprintf(tp, "Force Lat RR,");					//	 29
	fprintf(tp, "Vertical Load LF,");				//	 30
	fprintf(tp, "Vertical Load RF,");				//	 31
	fprintf(tp, "Vertical Load LR,");				//	 32
	fprintf(tp, "Vertical Load RR,");				//	 33
	fprintf(tp, "Steered Angle LF,");				//	 34
	fprintf(tp, "Steered Angle RF,");				//	 35
	fprintf(tp, "Steered Angle LR,");				//	 36
	fprintf(tp, "Steered Angle RR,");				//	 37
	fprintf(tp, "Patch Velocity Lateral LF,");		//	 38
	fprintf(tp, "Patch Velocity Lateral RF,");		//	 39
	fprintf(tp, "Patch Velocity Lateral LR,");		//	 40
	fprintf(tp, "Patch Velocity Lateral RR,");		//	 41
	fprintf(tp, "Patch Velocity Long LF,");			//	 42
	fprintf(tp, "Patch Velocity Long RF,");			//	 43
	fprintf(tp, "Patch Velocity Long LR,");			//	 44
	fprintf(tp, "Patch Velocity Long RR,");			//	 45
	fprintf(tp, "Slip Angle Sim LF,");				//	 46
	fprintf(tp, "Slip Angle Sim RF,");				//	 47
	fprintf(tp, "Slip Angle Sim LR,");				//	 48
	fprintf(tp, "Slip Angle Sim RR,");				//	 49
	fprintf(tp, "Throttle,");						//	 50
	fprintf(tp, "Brake,");							//	 51
	fprintf(tp, "SteeringWheelAngle,");				//	 52
	fprintf(tp, "Delta_Centre,");					//	 53
	fprintf(tp, "Clutch,");							//	 54
	fprintf(tp, "HandbrakeRaw,");					//	 55
	fprintf(tp, "AirTemp,");						//	 56
	fprintf(tp, "TrackTempCrew,");					//	 57
	fprintf(tp, "Lap,");							//	 58
	fprintf(tp, "Lap Elapsed Time,");				//	 59
	fprintf(tp, "Path Lateral Distance,");			//	 60
	fprintf(tp, "Lat,");							//	 61
	fprintf(tp, "Lon,");							//	 62
	fprintf(tp, "Alt,");							//	 63
	fprintf(tp, "WaterTemp,");						//	 64
	fprintf(tp, "OilTemp,");						//	 65
	fprintf(tp, "Engine Power,");					//	 66
	fprintf(tp, "Engine Torque,");					//	 67
	fprintf(tp, "FuelLevel,");						//	 68
	fprintf(tp, "SteeringWheelPctTorqueSign,");		//	 69
	fprintf(tp, "Output,");							//	 70
	fprintf(tp, "LFshockDefl,");					//	 71
	fprintf(tp, "RFshockDefl,");					//	 72
	fprintf(tp, "LRshockDefl,");					//	 73
	fprintf(tp, "RRshockDefl,");					//	 74
	fprintf(tp, "LFshockVel,");						//	 75
	fprintf(tp, "RFshockVel,");						//	 76
	fprintf(tp, "LRshockVel,");						//	 77
	fprintf(tp, "RRshockVel,");						//	 78
	fprintf(tp, "Camber LF,");						//	 79
	fprintf(tp, "Camber RF,");						//	 80
	fprintf(tp, "Camber LR,");						//	 81
	fprintf(tp, "Camber RR,");						//	 82
	fprintf(tp, "Slip_Angle_LF_CSV,");				//	 83
	fprintf(tp, "Slip_Angle_RF_CSV,");				//	 84
	fprintf(tp, "Slip_Angle_LR_CSV,");				//	 85
	fprintf(tp, "Slip_Angle_RR_CSV,");				//	 86
	fprintf(tp, "Slip_Angle_Front_CSV,");			//	 87
	fprintf(tp, "Slip_Angle_Rear_CSV,");			//	 88
	fprintf(tp, "LFSpeed,");						//	 89
	fprintf(tp, "RFSpeed,");						//	 90
	fprintf(tp, "LRSpeed,");						//	 91
	fprintf(tp, "RRSpeed,");						//	 92
	fprintf(tp, "Slip/Grip Fraction LF,");			//	 93
	fprintf(tp, "Slip/Grip Fraction RF,");			//	 94
	fprintf(tp, "Slip/Grip Fraction LR,");			//	 95
	fprintf(tp, "Slip/Grip Fraction RR,");			//	 96
	fprintf(tp, "Touching LF,");					//	 97
	fprintf(tp, "Touching RF,");					//	 98
	fprintf(tp, "Touching LR,");					//	 99
	fprintf(tp, "Touching RR,");					//	100
	fprintf(tp, "Slip Ratio LF,");					//	101
	fprintf(tp, "Slip Ratio RF,");					//	102
	fprintf(tp, "Slip Ratio LR,");					//	103
	fprintf(tp, "Slip Ratio RR,");					//	104
	fprintf(tp, "LFpressure,");						//	105
	fprintf(tp, "RFpressure,");						//	106
	fprintf(tp, "LRpressure,");						//	107
	fprintf(tp, "RRpressure,");						//	108
	fprintf(tp, "LFwearL,");						//	109
	fprintf(tp, "RFwearR,");						//	110
	fprintf(tp, "LRwearL,");						//	111
	fprintf(tp, "RRwearR,");						//	112
	fprintf(tp, "LFwearM,");						//	113
	fprintf(tp, "RFwearM,");						//	114
	fprintf(tp, "LRwearM,");						//	115
	fprintf(tp, "RRwearM,");						//	116
	fprintf(tp, "LFwearR,");						//	117
	fprintf(tp, "RFwearL,");						//	118
	fprintf(tp, "LRwearR,");						//	119
	fprintf(tp, "RRwearL,");						//	120
	fprintf(tp, "LFtempL,");						//	121
	fprintf(tp, "RFtempR,");						//	122
	fprintf(tp, "LRtempL,");						//	123
	fprintf(tp, "RRtempR,");						//	124
	fprintf(tp, "LFtempM,");						//	125
	fprintf(tp, "RFtempM,");						//	126
	fprintf(tp, "LRtempM,");						//	127
	fprintf(tp, "RRtempM,");						//	128
	fprintf(tp, "LFtempR,");						//	129
	fprintf(tp, "RFtempL,");						//	130
	fprintf(tp, "LRtempR,");						//	131
	fprintf(tp, "RRtempL,");						//	132
	fprintf(tp, "LFtempCL,");						//	133
	fprintf(tp, "RFtempCR,");						//	134
	fprintf(tp, "LRtempCL,");						//	135
	fprintf(tp, "RRtempCR,");						//	136
	fprintf(tp, "LFtempCM,");						//	137
	fprintf(tp, "RFtempCM,");						//	138
	fprintf(tp, "LRtempCM,");						//	139
	fprintf(tp, "RRtempCM,");						//	140
	fprintf(tp, "LFtempCR,");						//	141
	fprintf(tp, "RFtempCL,");						//	142
	fprintf(tp, "LRtempCR,");						//	143
	fprintf(tp, "RRtempCL,");						//	144
	fprintf(tp, "Air Temperature LF,");				//	145
	fprintf(tp, "Air Temperature RF,");				//	146
	fprintf(tp, "Air Temperature LR,");				//	147
	fprintf(tp, "Air Temperature RR,");				//	148
	fprintf(tp, "Brake Temperature LF,");			//	149
	fprintf(tp, "Brake Temperature RF,");			//	150
	fprintf(tp, "Brake Temperature LR,");			//	151
	fprintf(tp, "Brake Temperature RR,");			//	152
	fprintf(tp, "LFbrakeLinePress,");				//	153
	fprintf(tp, "RFbrakeLinePress,");				//	154
	fprintf(tp, "LRbrakeLinePress,");				//	155
	fprintf(tp, "RRbrakeLinePress,");				//	156
	fprintf(tp, "LFrideHeight LF,");				//	157
	fprintf(tp, "RFrideHeight RF,");				//	158
	fprintf(tp, "LRrideHeight LR,");				//	159
	fprintf(tp, "RRrideHeight RR,");				//	160
	fprintf(tp, "Downforce Front,");				//	161
	fprintf(tp, "Downforce Rear,");					//	162
	fprintf(tp, "Drag,");							//	163
	fprintf(tp, "SessionTick,");					//	164
	fprintf(tp, "Linear Velocity X (World),");		//	165
	fprintf(tp, "Linear Velocity Y (World),");		//	166
	fprintf(tp, "Linear Velocity Z (World),");		//	167
	fprintf(tp, "Linear Acceleration X (World),");	//	168
	fprintf(tp, "Linear Acceleration Y (World),");	//	169
	fprintf(tp, "Linear Acceleration Z (World),");	//	170
	fprintf(tp, "Linear Velocity X (SimV),");		//	171
	fprintf(tp, "Linear Velocity Y (SimV),");		//	172
	fprintf(tp, "Linear Velocity Z (SimV),");		//	173
	fprintf(tp, "Linear Acceleration X (SimV),");	//	174
	fprintf(tp, "Linear Acceleration Y (SimV),");	//	175
	fprintf(tp, "Linear Acceleration Z (SimV)\n");	//	176

	// Write CSV header row 2 (units)					Col		Channel Name
	fprintf(tp, "s,");								//	  1		Time
	fprintf(tp, "m,");								//	  2		Lap Distance
	fprintf(tp, "m/s,");							//	  3		Ground Speed
	fprintf(tp, "m/s^2,");							//	  4		Ground Acceleration
	fprintf(tp, "m/s,");							//	  5		Linear Velocity X
	fprintf(tp, "m/s,");							//	  6		Linear Velocity Y
	fprintf(tp, "m/s,");							//	  7		Linear Velocity Z
	fprintf(tp, "m/s^2,");							//	  8		Linear Acceleration X
	fprintf(tp, "m/s^2,");							//	  9		Linear Acceleration Y
	fprintf(tp, "m/s^2,");							//	 10		Linear Acceleration Z
	fprintf(tp, "rad,");							//	 11		Roll
	fprintf(tp, "rad,");							//	 12		Pitch
	fprintf(tp, "rad,");							//	 13		Yaw
	fprintf(tp, "rad/s,");							//	 14		Angular Velocity X
	fprintf(tp, "rad/s,");							//	 15		Angular Velocity Y
	fprintf(tp, "rad/s,");							//	 16		Angular Velocity Z
	fprintf(tp, "rad/s^2,");						//	 17		Angular Acceleration X
	fprintf(tp, "rad/s^2,");						//	 18		Angular Acceleration Y
	fprintf(tp, "rad/s^2,");						//	 19		Angular Acceleration Z
	fprintf(tp, "num,");							//	 20		Gear
	fprintf(tp, "RPM,");							//	 21		Engine Speed
	fprintf(tp, "N,");								//	 22		Force Long LF
	fprintf(tp, "N,");								//	 23		Force Long RF
	fprintf(tp, "N,");								//	 24		Force Long LR
	fprintf(tp, "N,");								//	 25		Force Long RR
	fprintf(tp, "N,");								//	 26		Force Lat LF
	fprintf(tp, "N,");								//	 27		Force Lat RF
	fprintf(tp, "N,");								//	 28		Force Lat LR
	fprintf(tp, "N,");								//	 29		Force Lat RR
	fprintf(tp, "N,");								//	 30		Vertical Load LF
	fprintf(tp, "N,");								//	 31		Vertical Load RF
	fprintf(tp, "N,");								//	 32		Vertical Load LR
	fprintf(tp, "N,");								//	 33		Vertical Load RR
	fprintf(tp, "rad,");							//	 34		Steered Angle LF
	fprintf(tp, "rad,");							//	 35		Steered Angle RF
	fprintf(tp, "rad,");							//	 36		Steered Angle LR
	fprintf(tp, "rad,");							//	 37		Steered Angle RR
	fprintf(tp, "m/s,");							//	 38		Patch Velocity Lateral LF
	fprintf(tp, "m/s,");							//	 39		Patch Velocity Lateral RF
	fprintf(tp, "m/s,");							//	 40		Patch Velocity Lateral LR
	fprintf(tp, "m/s,");							//	 41		Patch Velocity Lateral RR
	fprintf(tp, "m/s,");							//	 42		Patch Velocity Long LF
	fprintf(tp, "m/s,");							//	 43		Patch Velocity Long RF
	fprintf(tp, "m/s,");							//	 44		Patch Velocity Long LR
	fprintf(tp, "m/s,");							//	 45		Patch Velocity Long RR
	fprintf(tp, "deg,");							//	 46		Slip Angle Sim LF
	fprintf(tp, "deg,");							//	 47		Slip Angle Sim RF
	fprintf(tp, "deg,");							//	 48		Slip Angle Sim LR
	fprintf(tp, "deg,");							//	 49		Slip Angle Sim RR
	fprintf(tp, "ratio,");							//	 50		Throttle
	fprintf(tp, "ratio,");							//	 51		Brake
	fprintf(tp, "rad,");							//	 52		SWA
	fprintf(tp, "rad,");							//	 53		Delta Neutral
	fprintf(tp, "ratio,");							//	 54		Clutch
	fprintf(tp, "ratio,");							//	 55		Handbrake
	fprintf(tp, "Celsius,");						//	 56		Ambient Temperature
	fprintf(tp, "Celsius,");						//	 57		Track Temperature
	fprintf(tp, "num,");							//	 58		Lap Number
	fprintf(tp, "s,");								//	 59		Lap Elapsed Time
	fprintf(tp, "m,");								//	 60		Path Lateral Distance
	fprintf(tp, "m,");								//	 61		World Position X
	fprintf(tp, "m,");								//	 62		World Position Y
	fprintf(tp, "m,");								//	 63		World Position Z
	fprintf(tp, "Celsius,");						//	 64		Engine Coolant Temperature
	fprintf(tp, "Celsius,");						//	 65		Engine Oil Temperature
	fprintf(tp, "W,");								//	 66		Engine Power
	fprintf(tp, "Nm,");								//	 67		Engine Torque
	fprintf(tp, "ratio,");							//	 68		Fuel Level
	fprintf(tp, "Nm,");								//	 69		Steering Shaft Torque
	fprintf(tp, "ratio,");							//	 70		FFB Output
	fprintf(tp, "m,");								//	 71		Suspension Deflection LF
	fprintf(tp, "m,");								//	 72		Suspension Deflection RF
	fprintf(tp, "m,");								//	 73		Suspension Deflection LR
	fprintf(tp, "m,");								//	 74		Suspension Deflection RR
	fprintf(tp, "m/s,");							//	 75		Damper Velocity LF
	fprintf(tp, "m/s,");							//	 76		Damper Velocity RF
	fprintf(tp, "m/s,");							//	 77		Damper Velocity LR
	fprintf(tp, "m/s,");							//	 78		Damper Velocity RR
	fprintf(tp, "rad,");							//	 79		Camber LF
	fprintf(tp, "rad,");							//	 80		Camber RF
	fprintf(tp, "rad,");							//	 81		Camber LR
	fprintf(tp, "rad,");							//	 82		Camber RR
	fprintf(tp, "deg,");							//	 83		Slip Angle Computed LF
	fprintf(tp, "deg,");							//	 84		Slip Angle Computed RF
	fprintf(tp, "deg,");							//	 85		Slip Angle Computed LR
	fprintf(tp, "deg,");							//	 86		Slip Angle Computed RR
	fprintf(tp, "deg,");							//	 87		Slip Angle Computed FA
	fprintf(tp, "deg,");							//	 88		Slip Angle Computed RA
	fprintf(tp, "rad/s,");							//	 89		Angular Velocity LF
	fprintf(tp, "rad/s,");							//	 90		Angular Velocity RF
	fprintf(tp, "rad/s,");							//	 91		Angular Velocity LR
	fprintf(tp, "rad/s,");							//	 92		Angular Velocity RR
	fprintf(tp, "ratio,");							//	 93		Slip/Grip Fraction LF
	fprintf(tp, "ratio,");							//	 94		Slip/Grip Fraction RF
	fprintf(tp, "ratio,");							//	 95		Slip/Grip Fraction LR
	fprintf(tp, "ratio,");							//	 96		Slip/Grip Fraction RR
	fprintf(tp, "int,");							//	 97		Touching LF
	fprintf(tp, "int,");							//	 98		Touching RF
	fprintf(tp, "int,");							//	 99		Touching LR
	fprintf(tp, "int,");							//	100		Touching RR
	fprintf(tp, "ratio,");							//	101		Slip Ratio LF
	fprintf(tp, "ratio,");							//	102		Slip Ratio RF
	fprintf(tp, "ratio,");							//	103		Slip Ratio LR
	fprintf(tp, "ratio,");							//	104		Slip Ratio RR
	fprintf(tp, "Pa,");								//	105		Tyre Pressure LF
	fprintf(tp, "Pa,");								//	106		Tyre Pressure RF
	fprintf(tp, "Pa,");								//	107		Tyre Pressure LR
	fprintf(tp, "Pa,");								//	108		Tyre Pressure RR
	fprintf(tp, "ratio,");							//	109		Tyre Wear Outer LF
	fprintf(tp, "ratio,");							//	110		Tyre Wear Outer RF
	fprintf(tp, "ratio,");							//	111		Tyre Wear Outer LR
	fprintf(tp, "ratio,");							//	112		Tyre Wear Outer RR
	fprintf(tp, "ratio,");							//	113		Tyre Wear Middle LF
	fprintf(tp, "ratio,");							//	114		Tyre Wear Middle RF
	fprintf(tp, "ratio,");							//	115		Tyre Wear Middle LR
	fprintf(tp, "ratio,");							//	116		Tyre Wear Middle RR
	fprintf(tp, "ratio,");							//	117		Tyre Wear Inner LF
	fprintf(tp, "ratio,");							//	118		Tyre Wear Inner RF
	fprintf(tp, "ratio,");							//	119		Tyre Wear Inner LR
	fprintf(tp, "ratio,");							//	120		Tyre Wear Inner RR
	fprintf(tp, "Celsius,");						//	121		Tyre Surface Temp Outer LF
	fprintf(tp, "Celsius,");						//	122		Tyre Surface Temp Outer RF
	fprintf(tp, "Celsius,");						//	123		Tyre Surface Temp Outer LR
	fprintf(tp, "Celsius,");						//	124		Tyre Surface Temp Outer RR
	fprintf(tp, "Celsius,");						//	125		Tyre Surface Temp Centre LF
	fprintf(tp, "Celsius,");						//	126		Tyre Surface Temp Centre RF
	fprintf(tp, "Celsius,");						//	127		Tyre Surface Temp Centre LR
	fprintf(tp, "Celsius,");						//	128		Tyre Surface Temp Centre RR
	fprintf(tp, "Celsius,");						//	129		Tyre Surface Temp Inner LF
	fprintf(tp, "Celsius,");						//	130		Tyre Surface Temp Inner RF
	fprintf(tp, "Celsius,");						//	131		Tyre Surface Temp Inner LR
	fprintf(tp, "Celsius,");						//	132		Tyre Surface Temp Inner RR
	fprintf(tp, "Celsius,");						//	133		Tyre Core Temp Outer LF
	fprintf(tp, "Celsius,");						//	134		Tyre Core Temp Outer RF
	fprintf(tp, "Celsius,");						//	135		Tyre Core Temp Outer LR
	fprintf(tp, "Celsius,");						//	136		Tyre Core Temp Outer RR
	fprintf(tp, "Celsius,");						//	137		Tyre Core Temp Centre LF
	fprintf(tp, "Celsius,");						//	138		Tyre Core Temp Centre RF
	fprintf(tp, "Celsius,");						//	139		Tyre Core Temp Centre LR
	fprintf(tp, "Celsius,");						//	140		Tyre Core Temp Centre RR
	fprintf(tp, "Celsius,");						//	141		Tyre Core Temp Inner LF
	fprintf(tp, "Celsius,");						//	142		Tyre Core Temp Inner RF
	fprintf(tp, "Celsius,");						//	143		Tyre Core Temp Inner LR
	fprintf(tp, "Celsius,");						//	144		Tyre Core Temp Inner RR
	fprintf(tp, "Celsius,");						//	145		Air Temperature LF
	fprintf(tp, "Celsius,");						//	146		Air Temperature RF
	fprintf(tp, "Celsius,");						//	147		Air Temperature LR
	fprintf(tp, "Celsius,");						//	148		Air Temperature RR
	fprintf(tp, "Celsius,");						//	149		Brake Temperature LF
	fprintf(tp, "Celsius,");						//	150		Brake Temperature RF
	fprintf(tp, "Celsius,");						//	151		Brake Temperature LR
	fprintf(tp, "Celsius,");						//	152		Brake Temperature RR
	fprintf(tp, "Pa,");								//	153		Brake Pressure LF
	fprintf(tp, "Pa,");								//	154		Brake Pressure RF
	fprintf(tp, "Pa,");								//	155		Brake Pressure LR
	fprintf(tp, "Pa,");								//	156		Brake Pressure RR
	fprintf(tp, "m,");								//	157		Ride Height LF
	fprintf(tp, "m,");								//	158		Ride Height RF
	fprintf(tp, "m,");								//	159		Ride Height LR
	fprintf(tp, "m,");								//	160		Ride Height RR
	fprintf(tp, "N,");								//	161		Downforce Front
	fprintf(tp, "N,");								//	162		Downforce Rear
	fprintf(tp, "N,");								//	163		Drag
	fprintf(tp, "num,");							//	164		SessionTick
	fprintf(tp, "m/s,");							//	165		Linear Velocity X (World)
	fprintf(tp, "m/s,");							//	166		Linear Velocity Y (World)
	fprintf(tp, "m/s,");							//	167		Linear Velocity Z (World)
	fprintf(tp, "m/s^2,");							//	168		Linear Acceleration X (World)
	fprintf(tp, "m/s^2,");							//	169		Linear Acceleration Y (World)
	fprintf(tp, "m/s^2,");							//	170		Linear Acceleration Z (World)
	fprintf(tp, "m/s,");							//	171		Linear Velocity X (Sim Vehicle)
	fprintf(tp, "m/s,");							//	172		Linear Velocity Y (Sim Vehicle)
	fprintf(tp, "m/s,");							//	173		Linear Velocity Z (Sim Vehicle)
	fprintf(tp, "m/s^2,");							//	174		Linear Acceleration X (Sim Vehicle)
	fprintf(tp, "m/s^2,");							//	175		Linear Acceleration Y (Sim Vehicle)
	fprintf(tp, "m/s^2\n");							//	176		Linear Acceleration Z (Sim Vehicle)

	// Packet receive loop.
	char recvbuf[BUFFER_SIZE];
	memset(recvbuf, 0, sizeof(recvbuf)); // Set recvbuf to zero.
	int bytes = 0;
	do
	{
		bytes = recv(sock, recvbuf, BUFFER_SIZE, 0);
		if (bytes > 0)
		{
			SessionTick++;
			fwrite(recvbuf, 1, 272, tp2);							// Write bin file record
			OutSimPacketReceived((OutSimPack2 &)recvbuf, tp);		// Function to write to console and CSV file
		}
		else if (bytes == 0)
			printf("Error: Lost connection with LFS\n");
		else
			printf("Error: %d\n", WSAGetLastError());
	} while (bytes > 0);

	// Cleanup and exit.
	closesocket(sock);
	WSACleanup();
	fclose(tp);
	fclose(tp2);
	return EXIT_SUCCESS;
}

void OutSimPacketReceived(const OutSimPack2 packet, FILE *tp)
// Function to process received OutSim packet.
//	Displays Time and SWA on console.
//	Writes CSV file record (comma delimited).
//
{
	// Variables for slip angle computation as per my iRacing/Atlas slip angle functions.
	float DistCGX;
	float DistCGY;
	float VelocityTireXFromYaw;
	float VelocityTireYFromYaw;
	float VelocityTireTotalX;
	float VelocityTireTotalY;
	float ComputedSlipAngle;
	float VelocityAxleYFromYaw;
	float VelocityAxleTotalX;
	float VelocityAxleTotalY;

	// Variables for linear velocity and acceleration in local coordinates.
	Vector LocalVel;
	Vector LocalAccel;

	// Display to console - Time (seconds, precision = min delay between packets = 0.01 seconds) and SWA (degrees = tyre steered angle * radians-to-degrees * MRT5 steering ratio).
	printf("Time: %7.2f SWA: %6.2f \r", packet.Time / 1000.0f, packet.OSInputs.InputSteer * 57.2958f * 6.75f);

	// Transform linear velocities and accelerations from world coordinate system to vehicle coordinate system.
	LocalVel = World2Local(packet.OSMain.Vel, packet.OSMain.Pitch, packet.OSMain.Roll, packet.OSMain.Heading);
	LocalAccel = World2Local(packet.OSMain.Accel, packet.OSMain.Pitch, packet.OSMain.Roll, packet.OSMain.Heading);

	// Write CSV data row...
	//	Axis conventions:
	//		LFS Y (forward)	= iRacing X				Rotation = Roll
	//		LFS X (right)	= iRacing Y inverted	Rotation = Pitch
	//		LFS Z (up)		= iRacing Z				Rotation = Yaw
	//		=> Translation required (both sims use right-handed coordinates systems, but opposite conventions for left/right)
	//	Angle/rotation conventions (viewed from positive axis toward origin):
	//		LFS		Positive = Counter-clockwise
	//		iRacing	Positive = Counter-clockwise
	//		=> Invert Pitch only (due to opposite conventions for left/right)
	//																													Col		Channel Name
	fprintf(tp, "%.2f,", packet.Time / 1000.0f);																	//	  1		Time
	fprintf(tp, "%.3f,", packet.IndexedDistance);																	//	  2		Lap Distance
	GroundSpeed = sqrtf(powf(packet.OSMain.Vel.X, 2) + powf(packet.OSMain.Vel.Y, 2));
	fprintf(tp, "%.3f,", GroundSpeed);																				//	  3		Ground Speed (computed)
	fprintf(tp, "%.3f,", (GroundSpeed - PrevGroundSpeed) * 100);													//	  4		Ground Acceleration (computed)
	fprintf(tp, "%.3f,", LocalVel.Y);																				//	  5		Linear Velocity X
	fprintf(tp, "%.3f,", -LocalVel.X);																				//	  6		Linear Velocity Y (invert because iRacing Y is left)
	fprintf(tp, "%.3f,", LocalVel.Z);																				//	  7		Linear Velocity Z
	fprintf(tp, "%.3f,", LocalAccel.Y);																			//	  8		Linear Acceleration X
	fprintf(tp, "%.3f,", -LocalAccel.X);																			//	  9		Linear Acceleration Y (invert because iRacing Y is left)
	fprintf(tp, "%.3f,", LocalAccel.Z);																			//	 10		Linear Acceleration Z
	fprintf(tp, "%.2f,", packet.OSMain.Roll);																		//	 11		Roll
	fprintf(tp, "%.2f,", -packet.OSMain.Pitch);																	//	 12		Pitch (invert because iRacing Y is left)
	fprintf(tp, "%.2f,", packet.OSMain.Heading + 3.14159f);														//	 13		Yaw (adjust to iRacing range 0-2pi)
	fprintf(tp, "%.2f,", packet.OSMain.AngVel.Y);																	//	 14		Angular Velocity X
	fprintf(tp, "%.2f,", -packet.OSMain.AngVel.X);																	//	 15		Angular Velocity Y (invert because iRacing Y is left)
	fprintf(tp, "%.2f,", packet.OSMain.AngVel.Z);																	//	 16		Angular Velocity Z
	fprintf(tp, "%.2f,", (packet.OSMain.AngVel.Y - PrevAngVelY) * 100.0f);											//	 17		Angular Acceleration X (computed)
	fprintf(tp, "%.2f,", -(packet.OSMain.AngVel.X - PrevAngVelX) * 100.0f);										//	 18		Angular Acceleration Y (computed, invert because iRacing Y is left)
	fprintf(tp, "%.2f,", (packet.OSMain.AngVel.Z - PrevAngVelZ) * 100.0f);											//	 19		Angular Acceleration Z (computed)
	fprintf(tp, "%d,", packet.Gear);																				//	 20		Gear
	fprintf(tp, "%5f,", (packet.EngineAngVel * 60) / (2 * 3.14159));												//	 21		Engine Speed
	fprintf(tp, "%.2f,", packet.OSWheels[2].YForce);																//	 22		Force Long LF
	fprintf(tp, "%.2f,", packet.OSWheels[3].YForce);																//	 23		Force Long RF
	fprintf(tp, "%.2f,", packet.OSWheels[0].YForce);																//	 24		Force Long LR
	fprintf(tp, "%.2f,", packet.OSWheels[1].YForce);																//	 25		Force Long RR
	fprintf(tp, "%.2f,", -packet.OSWheels[2].XForce);																//	 26		Force Lat LF (invert because iRacing Y is left)
	fprintf(tp, "%.2f,", -packet.OSWheels[3].XForce);																//	 27		Force Lat RF (invert because iRacing Y is left)
	fprintf(tp, "%.2f,", -packet.OSWheels[0].XForce);																//	 28		Force Lat LR (invert because iRacing Y is left)
	fprintf(tp, "%.2f,", -packet.OSWheels[1].XForce);																//	 29		Force Lat RR (invert because iRacing Y is left)
	fprintf(tp, "%.2f,", packet.OSWheels[2].VerticalLoad);															//	 30		Vertical Load LF
	fprintf(tp, "%.2f,", packet.OSWheels[3].VerticalLoad);															//	 31		Vertical Load RF
	fprintf(tp, "%.2f,", packet.OSWheels[0].VerticalLoad);															//	 32		Vertical Load LR
	fprintf(tp, "%.2f,", packet.OSWheels[1].VerticalLoad);															//	 33		Vertical Load RR
	fprintf(tp, "%.2f,", packet.OSWheels[2].Steer);																//	 34		Steered Angle LF
	fprintf(tp, "%.2f,", packet.OSWheels[3].Steer);																//	 35		Steered Angle RF
	fprintf(tp, "%.2f,", packet.OSWheels[0].Steer);																//	 36		Steered Angle LR
	fprintf(tp, "%.2f,", packet.OSWheels[1].Steer);																//	 37		Steered Angle RR
	fprintf(tp, ",");																								//	 38		Patch Velocity Lateral LF (N/A)
	fprintf(tp, ",");																								//	 39		Patch Velocity Lateral RF (N/A)
	fprintf(tp, ",");																								//	 40		Patch Velocity Lateral LR (N/A)
	fprintf(tp, ",");																								//	 41		Patch Velocity Lateral RR (N/A)
	fprintf(tp, ",");																								//	 42		Patch Velocity Long LF (N/A)
	fprintf(tp, ",");																								//	 43		Patch Velocity Long RF (N/A)
	fprintf(tp, ",");																								//	 44		Patch Velocity Long LR (N/A)
	fprintf(tp, ",");																								//	 45		Patch Velocity Long RR (N/A)
	fprintf(tp, "%.2f,", -atanf(packet.OSWheels[2].TanSlipAngle) * 57.2958f);										//	 46		Slip Angle Sim LF (invert because iRacing Y is left)
	fprintf(tp, "%.2f,", -atanf(packet.OSWheels[3].TanSlipAngle) * 57.2958f);										//	 47		Slip Angle Sim RF (invert because iRacing Y is left)
	fprintf(tp, "%.2f,", -atanf(packet.OSWheels[0].TanSlipAngle) * 57.2958f);										//	 48		Slip Angle Sim LR (invert because iRacing Y is left)
	fprintf(tp, "%.2f,", -atanf(packet.OSWheels[1].TanSlipAngle) * 57.2958f);										//	 49		Slip Angle Sim RR (invert because iRacing Y is left)
	fprintf(tp, "%.3f,", packet.OSInputs.Throttle);																//	 50		Throttle
	fprintf(tp, "%.3f,", packet.OSInputs.Brake);																	//	 51		Brake
	fprintf(tp, "%.2f,", packet.OSInputs.InputSteer * 6.75f);														//	 52		SWA (computed)
	fprintf(tp, "%.2f,", packet.OSInputs.InputSteer);																//	 53		Delta Neutral
	fprintf(tp, "%.3f,", packet.OSInputs.Clutch);																	//	 54		Clutch
	fprintf(tp, "%.3f,", packet.OSInputs.Handbrake);																//	 55		Handbrake
	fprintf(tp, ",");																								//	 56		Ambient Temperature (N/A)
	fprintf(tp, ",");																								//	 57		Track Temperature (N/A)
	fprintf(tp, ",");																								//	 58		Lap Number (N/A)
	fprintf(tp, ",");																								//	 59		Lap Elapsed Time (N/A)
	fprintf(tp, ",");																								//	 60		Path Lateral Distance (N/A)
	fprintf(tp, "%.3f,", packet.OSMain.Pos.Y / 65536.0f);															//	 61		World Position X
	fprintf(tp, "%.3f,", -packet.OSMain.Pos.X / 65536.0f);															//	 62		World Position Y (invert because iRacing Y is left)
	fprintf(tp, "%.3f,", packet.OSMain.Pos.Z / 65536.0f);															//	 63		World Position Z
	fprintf(tp, ",");																								//	 64		Engine Coolant Temperature (N/A)
	fprintf(tp, ",");																								//	 65		Engine Oil Temperature (N/A)
	fprintf(tp, ",");																								//	 66		Engine Power (N/A)
	fprintf(tp, ",");																								//	 67		Engine Torque (N/A)
	fprintf(tp, ",");																								//	 68		Fuel Level (N/A)
	fprintf(tp, ",");																								//	 69		Steering Shaft Torque (N/A)
	fprintf(tp, ",");																								//	 70		FFB Output (N/A)
	fprintf(tp, "%.3f,", packet.OSWheels[2].SuspDeflect);															//	 71		Suspension Deflection LF
	fprintf(tp, "%.3f,", packet.OSWheels[3].SuspDeflect);															//	 72		Suspension Deflection RF
	fprintf(tp, "%.3f,", packet.OSWheels[0].SuspDeflect);															//	 73		Suspension Deflection LR
	fprintf(tp, "%.3f,", packet.OSWheels[1].SuspDeflect);															//	 74		Suspension Deflection RR
	fprintf(tp, "%.3f,", (packet.OSWheels[2].SuspDeflect - PrevSuspDeflectLF) * 100.0f);							//	 75		Damper Velocity LF (computed)
	fprintf(tp, "%.3f,", (packet.OSWheels[3].SuspDeflect - PrevSuspDeflectRF) * 100.0f);							//	 76		Damper Velocity RF (computed)
	fprintf(tp, "%.3f,", (packet.OSWheels[0].SuspDeflect - PrevSuspDeflectLR) * 100.0f);							//	 77		Damper Velocity LR (computed)
	fprintf(tp, "%.3f,", (packet.OSWheels[1].SuspDeflect - PrevSuspDeflectRR) * 100.0f);							//	 78		Damper Velocity RR (computed)
	fprintf(tp, "%.2f,", packet.OSWheels[2].LeanRelToRoad);														//	 79		Camber LF
	fprintf(tp, "%.2f,", packet.OSWheels[3].LeanRelToRoad);														//	 80		Camber RF
	fprintf(tp, "%.2f,", packet.OSWheels[0].LeanRelToRoad);														//	 81		Camber LR
	fprintf(tp, "%.2f,", packet.OSWheels[1].LeanRelToRoad);														//	 82		Camber RR
	// Replication of iRacing/Atlas function Slip_Angle_LF -------------------------------------------------------------------------------------------------------------
	DistCGY = -1116.0f * 0.5f * 0.001f;
	DistCGX = 1769.0f * 0.001f * (1.0f - 0.375f);
	VelocityTireXFromYaw = packet.OSMain.AngVel.Z * DistCGY;
	VelocityTireYFromYaw = packet.OSMain.AngVel.Z * DistCGX;
	VelocityTireTotalX = LocalVel.Y + VelocityTireXFromYaw;
	VelocityTireTotalY = -LocalVel.X + VelocityTireYFromYaw;
	ComputedSlipAngle = (atanf((-VelocityTireTotalY / VelocityTireTotalX)) * 57.2958f) - 0.2f + (packet.OSInputs.InputSteer * 57.2958f);
	fprintf(tp, "%.2f,", ComputedSlipAngle);																		//	 83		Slip Angle Computed LF (computed)
	// Replication of iRacing/Atlas function Slip_Angle_RF -------------------------------------------------------------------------------------------------------------
	DistCGY = 1116.0f * 0.5f * 0.001f;
	DistCGX = 1769.0f * 0.001f * (1.0f - 0.375f);
	VelocityTireXFromYaw = packet.OSMain.AngVel.Z * DistCGY;
	VelocityTireYFromYaw = packet.OSMain.AngVel.Z * DistCGX;
	VelocityTireTotalX = LocalVel.Y + VelocityTireXFromYaw;
	VelocityTireTotalY = -LocalVel.X + VelocityTireYFromYaw;
	ComputedSlipAngle = (atanf((-VelocityTireTotalY / VelocityTireTotalX)) * 57.2958f) + 0.2f + (packet.OSInputs.InputSteer * 57.2958f);
	fprintf(tp, "%.2f,", ComputedSlipAngle);																		//	 84		Slip Angle Computed RF (computed)
	// Replication of iRacing/Atlas function Slip_Angle_LR -------------------------------------------------------------------------------------------------------------
	DistCGY = -1116.0f * 0.5f * 0.001f;
	DistCGX = -1769.0f * 0.001f * 0.375f;
	VelocityTireXFromYaw = packet.OSMain.AngVel.Z * DistCGY;
	VelocityTireYFromYaw = packet.OSMain.AngVel.Z * DistCGX;
	VelocityTireTotalX = LocalVel.Y + VelocityTireXFromYaw;
	VelocityTireTotalY = -LocalVel.X + VelocityTireYFromYaw;
	ComputedSlipAngle = (atanf((-VelocityTireTotalY / VelocityTireTotalX)) * 57.2958f) - 0.5f;
	fprintf(tp, "%.2f,", ComputedSlipAngle);																		//	 85		Slip Angle Computed LR (computed)
	// Replication of iRacing/Atlas function Slip_Angle_RR -------------------------------------------------------------------------------------------------------------
	DistCGY = 1116.0f * 0.5f * 0.001f;
	DistCGX = -1769.0f * 0.001f * 0.375f;
	VelocityTireXFromYaw = packet.OSMain.AngVel.Z * DistCGY;
	VelocityTireYFromYaw = packet.OSMain.AngVel.Z * DistCGX;
	VelocityTireTotalX = LocalVel.Y + VelocityTireXFromYaw;
	VelocityTireTotalY = -LocalVel.X + VelocityTireYFromYaw;
	ComputedSlipAngle = (atanf((-VelocityTireTotalY / VelocityTireTotalX)) * 57.2958f) + 0.5f;
	fprintf(tp, "%.2f,", ComputedSlipAngle);																		//	 86		Slip Angle Computed RR (computed)
	// Replication of iRacing/Atlas function Slip_Angle_Front ----------------------------------------------------------------------------------------------------------
	DistCGX = 1769.0f * 0.001f * (1.0f - 0.375f);
	VelocityAxleYFromYaw = packet.OSMain.AngVel.Z * DistCGX;
	VelocityAxleTotalX = LocalVel.Y;
	VelocityAxleTotalY = -LocalVel.X + VelocityAxleYFromYaw;
	ComputedSlipAngle = (atanf((-VelocityAxleTotalY / VelocityAxleTotalX)) + packet.OSInputs.InputSteer) * 57.2958f;
	fprintf(tp, "%.2f,", ComputedSlipAngle);																		//	 87		Slip Angle Computed Front Axle (computed)
	// Replication of iRacing/Atlas function Slip_Angle_Rear -----------------------------------------------------------------------------------------------------------
	DistCGX = -1769.0f * 0.001f * 0.375f;
	VelocityAxleYFromYaw = packet.OSMain.AngVel.Z * DistCGX;
	VelocityAxleTotalX = LocalVel.Y;
	VelocityAxleTotalY = -LocalVel.X + VelocityAxleYFromYaw;
	ComputedSlipAngle = (atanf((-VelocityAxleTotalY / VelocityAxleTotalX))) * 57.2958f;
	fprintf(tp, "%.2f,", ComputedSlipAngle);																		//	 88		Slip Angle Computed Rear Axle (computed)
	fprintf(tp, "%.2f,", packet.OSWheels[2].AngVel);																//	 89		Angular Velocity LF
	fprintf(tp, "%.2f,", packet.OSWheels[3].AngVel);																//	 90		Angular Velocity RF
	fprintf(tp, "%.2f,", packet.OSWheels[0].AngVel);																//	 91		Angular Velocity LR
	fprintf(tp, "%.2f,", packet.OSWheels[1].AngVel);																//	 92		Angular Velocity RR
	fprintf(tp, "%.3f,", (float)packet.OSWheels[2].SlipFraction / 255.0f);											//	 93		Slip/Grip Fraction LF
	fprintf(tp, "%.3f,", (float)packet.OSWheels[3].SlipFraction / 255.0f);											//	 94		Slip/Grip Fraction RF
	fprintf(tp, "%.3f,", (float)packet.OSWheels[0].SlipFraction / 255.0f);											//	 95		Slip/Grip Fraction LR
	fprintf(tp, "%.3f,", (float)packet.OSWheels[1].SlipFraction / 255.0f);											//	 96		Slip/Grip Fraction RR
	fprintf(tp, "%d,", packet.OSWheels[2].Touching);																//	 97		Touching LF
	fprintf(tp, "%d,", packet.OSWheels[3].Touching);																//	 98		Touching RF
	fprintf(tp, "%d,", packet.OSWheels[0].Touching);																//	 99		Touching LR
	fprintf(tp, "%d,", packet.OSWheels[1].Touching);																//	100		Touching RR
	fprintf(tp, "%.2f,", packet.OSWheels[2].SlipRatio);															//	101		Slip Ratio LF
	fprintf(tp, "%.2f,", packet.OSWheels[3].SlipRatio);															//	102		Slip Ratio RF
	fprintf(tp, "%.2f,", packet.OSWheels[0].SlipRatio);															//	103		Slip Ratio LR
	fprintf(tp, "%.2f,", packet.OSWheels[1].SlipRatio);															//	104		Slip Ratio RR
	fprintf(tp, ",");																								//	105		Tyre Pressure LF (N/A)
	fprintf(tp, ",");																								//	106		Tyre Pressure RF (N/A)
	fprintf(tp, ",");																								//	107		Tyre Pressure LR (N/A)
	fprintf(tp, ",");																								//	108		Tyre Pressure RR (N/A)
	fprintf(tp, ",");																								//	109		Tyre Wear Outer LF (N/A)
	fprintf(tp, ",");																								//	110		Tyre Wear Outer RF (N/A)
	fprintf(tp, ",");																								//	111		Tyre Wear Outer LR (N/A)
	fprintf(tp, ",");																								//	112		Tyre Wear Outer RR (N/A)
	fprintf(tp, ",");																								//	113		Tyre Wear Middle LF (N/A)
	fprintf(tp, ",");																								//	114		Tyre Wear Middle RF (N/A)
	fprintf(tp, ",");																								//	115		Tyre Wear Middle LR (N/A)
	fprintf(tp, ",");																								//	116		Tyre Wear Middle RR (N/A)
	fprintf(tp, ",");																								//	117		Tyre Wear Inner LF (N/A)
	fprintf(tp, ",");																								//	118		Tyre Wear Inner RF (N/A)
	fprintf(tp, ",");																								//	119		Tyre Wear Inner LR (N/A)
	fprintf(tp, ",");																								//	120		Tyre Wear Inner RR (N/A)
	fprintf(tp, ",");																								//	121		Tyre Surface Temp Outer LF (N/A)
	fprintf(tp, ",");																								//	122		Tyre Surface Temp Outer RF (N/A)
	fprintf(tp, ",");																								//	123		Tyre Surface Temp Outer LR (N/A)
	fprintf(tp, ",");																								//	124		Tyre Surface Temp Outer RR (N/A)
	fprintf(tp, ",");																								//	125		Tyre Surface Temp Centre LF (N/A)
	fprintf(tp, ",");																								//	126		Tyre Surface Temp Centre RF (N/A)
	fprintf(tp, ",");																								//	127		Tyre Surface Temp Centre LR (N/A)
	fprintf(tp, ",");																								//	128		Tyre Surface Temp Centre RR (N/A)
	fprintf(tp, ",");																								//	129		Tyre Surface Temp Inner LF (N/A)
	fprintf(tp, ",");																								//	130		Tyre Surface Temp Inner RF (N/A)
	fprintf(tp, ",");																								//	131		Tyre Surface Temp Inner LR (N/A)
	fprintf(tp, ",");																								//	132		Tyre Surface Temp Inner RR (N/A)
	fprintf(tp, ",");																								//	133		Tyre Core Temp Outer LF (N/A)
	fprintf(tp, ",");																								//	134		Tyre Core Temp Outer RF (N/A)
	fprintf(tp, ",");																								//	135		Tyre Core Temp Outer LR (N/A)
	fprintf(tp, ",");																								//	136		Tyre Core Temp Outer RR (N/A)
	fprintf(tp, ",");																								//	137		Tyre Core Temp Centre LF (N/A)
	fprintf(tp, ",");																								//	138		Tyre Core Temp Centre RF (N/A)
	fprintf(tp, ",");																								//	139		Tyre Core Temp Centre LR (N/A)
	fprintf(tp, ",");																								//	140		Tyre Core Temp Centre RR (N/A)
	fprintf(tp, ",");																								//	141		Tyre Core Temp Inner LF (N/A)
	fprintf(tp, ",");																								//	142		Tyre Core Temp Inner RF (N/A)
	fprintf(tp, ",");																								//	143		Tyre Core Temp Inner LR (N/A)
	fprintf(tp, ",");																								//	144		Tyre Core Temp Inner RR (N/A)
	fprintf(tp, "%d,", packet.OSWheels[2].AirTemp);																	//	145		Air Temperature LF
	fprintf(tp, "%d,", packet.OSWheels[3].AirTemp);																	//	146		Air Temperature RF
	fprintf(tp, "%d,", packet.OSWheels[0].AirTemp);																	//	147		Air Temperature LR
	fprintf(tp, "%d,", packet.OSWheels[1].AirTemp);																	//	148		Air Temperature RR
	fprintf(tp, ",");																								//	149		Brake Temperature LF (N/A)
	fprintf(tp, ",");																								//	150		Brake Temperature RF (N/A)
	fprintf(tp, ",");																								//	151		Brake Temperature LR (N/A)
	fprintf(tp, ",");																								//	152		Brake Temperature RR (N/A)
	fprintf(tp, ",");																								//	153		Brake Pressure LF (N/A)
	fprintf(tp, ",");																								//	154		Brake Pressure RF (N/A)
	fprintf(tp, ",");																								//	155		Brake Pressure LR (N/A)
	fprintf(tp, ",");																								//	156		Brake Pressure RR (N/A)
	fprintf(tp, ",");																								//	157		Ride Height LF (N/A)
	fprintf(tp, ",");																								//	158		Ride Height RF (N/A)
	fprintf(tp, ",");																								//	159		Ride Height LR (N/A)
	fprintf(tp, ",");																								//	160		Ride Height RR (N/A)
	fprintf(tp, ",");																								//	161		Downforce Front (N/A)
	fprintf(tp, ",");																								//	162		Downforce Rear (N/A)
	fprintf(tp, ",");																								//	163		Drag (N/A)
	fprintf(tp, "%d,", SessionTick);																				//	164		SessionTick (computed)
	fprintf(tp, "%.3f,", packet.OSMain.Vel.X);																		//	165		Linear Velocity X (World)
	fprintf(tp, "%.3f,", packet.OSMain.Vel.Y);																		//	166		Linear Velocity Y (World)
	fprintf(tp, "%.3f,", packet.OSMain.Vel.Z);																		//	167		Linear Velocity Z (World)
	fprintf(tp, "%.3f,", packet.OSMain.Accel.X);																	//	168		Linear Acceleration X (World)
	fprintf(tp, "%.3f,", packet.OSMain.Accel.Y);																	//	169		Linear Acceleration Y (World)
	fprintf(tp, "%.3f,", packet.OSMain.Accel.Z);																	//	170		Linear Acceleration Z (World)
	fprintf(tp, "%.3f,", LocalVel.X);																				//	171		Linear Velocity X (Sim Vehicle)
	fprintf(tp, "%.3f,", LocalVel.Y);																				//	172		Linear Velocity Y (Sim Vehicle)
	fprintf(tp, "%.3f,", LocalVel.Z);																				//	173		Linear Velocity Z (Sim Vehicle)
	fprintf(tp, "%.3f,", LocalAccel.X);																			//	174		Linear Acceleration X (Sim Vehicle)
	fprintf(tp, "%.3f,", LocalAccel.Y);																			//	175		Linear Acceleration Y (Sim Vehicle)
	fprintf(tp, "%.3f\n", LocalAccel.Z);																			//	176		Linear Acceleration Z (Sim Vehicle)

	// Set previous sample variables
	PrevAngVelX = packet.OSMain.AngVel.X;
	PrevAngVelY = packet.OSMain.AngVel.Y;
	PrevAngVelZ = packet.OSMain.AngVel.Z;
	PrevSuspDeflectLR = packet.OSWheels[0].SuspDeflect;
	PrevSuspDeflectRR = packet.OSWheels[1].SuspDeflect;
	PrevSuspDeflectLF = packet.OSWheels[2].SuspDeflect;
	PrevSuspDeflectRF = packet.OSWheels[3].SuspDeflect;
	PrevGroundSpeed = GroundSpeed;
}

Vector World2Local(Vector WorldVector, float Pitch, float Roll, float Yaw)
// Function to transform WorldVector to LocalVector using the Euler angles
// and 3D affine transformation. Uses the 3x3 matrix form because only rotation is
// required. See section 11.9 in Computer Graphics by F.S. Hill Jr and
// formula 1.2 (total rotation matrix) in 3D Affine Coordinate Transformations
// by Constantin-Octavian Andrei (thesis, Royal Institute of Technology (KTH), Stockholm,
// Sweden).
//
// NOTE:
//	Input vector, output vector and Euler angles all use the sim-native axis system.
//	Mapping to iRacing axis system is done ONLY during CSV rendering.
//
{
	Vector LocalVector;

	// Total rotation matrix row 1 * WorldVector
	LocalVector.X = cosf(Roll)*cosf(Yaw)*WorldVector.X \
					+ (cosf(Pitch)*sinf(Yaw) + sinf(Pitch)*sinf(Roll)*cosf(Yaw))*WorldVector.Y \
					+ (sinf(Pitch)*sinf(Yaw) - cosf(Pitch)*sinf(Roll)*cosf(Yaw))*WorldVector.Z;

	// Total rotation matrix row 2 * WorldVector
	LocalVector.Y = -cosf(Roll)*sinf(Yaw)*WorldVector.X \
					+ (cosf(Pitch)*cosf(Yaw) - sinf(Pitch)*sinf(Roll)*sinf(Yaw))*WorldVector.Y \
					+ (sinf(Pitch)*cosf(Yaw) + cosf(Pitch)*sinf(Roll)*sinf(Yaw))*WorldVector.Z;

	// Total rotation matrix row 3 * WorldVector
	LocalVector.Z = sinf(Roll)*WorldVector.X \
					- sinf(Pitch)*cosf(Roll)*WorldVector.Y \
					+ cosf(Pitch)*cosf(Roll)*WorldVector.Z;

	return LocalVector;
}