<?php
/**
 * @file stringproc.php
 * @author Michael Simons <ixulai@gmail.com>
 * @license http://creativecommons.org/licenses/by/2.5/ Creative Commons 
 */

/**
 * Provides efficient & optimized codepage and colour switch parsing.
 * 
 * @class LFSStringProc
 * @version 2.0.00
 */
class LFSStringProc
{
    /**
     * (String) This is the preg_split pattern used to scan for colours.
     * @protected
     */
    var $m_sColourPattern = '/(?:\^)([0-8])/';
    
    /**
     * (fnCallback) This is the current function used for processing colours after they have been scanned.
     * @protected
     */
    var $m_sColourProc = '__sProcColourCSSStyle';

    /**
     * (Array) This is the default colour map to use when one is not specified at runtime. If this too  is not specified, the processor will use it's default.
     * @protected
     */
    var $m_aColourMap = null;
    
    /**
     * Ctor. Do not use to instantiate LFSStringProc directly. Use LFSStringProc::singleton instead.
     *
     * @public
     * @deprecated Do not use to instantiate LFSStringProc directly. Use LFSStringProc::singleton instead.
     * @param singletonCreator (Boolean) Set to true if you really need to use 'new' to create an instance.
     * @return (Void)
     */
    function __construct($singletonCreator = false)
    {
        if(!$singletonCreator)
            trigger_error('Do not use \'new\' to instantiate LFSStringProc. Use LFSStringProc::singleton instead)', E_USER_ERROR);
    }

    /**
     * Singleton distributor. Use this method to get an instance instead of new.
     * 
     * @public
     * @return (Object) Singleton LFSStingProc instance.
     */
    function &singleton()
    {
        static $singleton;
        if($singleton == null)
            $singleton =& new LFSStringProc(true);

        return $singleton;
    }

    /**
     * This method splits a string in to chunks based on the pattern provided.
     * 
     * @protected
     * @param string (String) The string to 'chunk'.
     * @param pattern (String) The pattern to use for 'chunking'.
     * @param current (String) The default group to assign to non-assigned chunks.
     * @return (Array) An array of chunks with their switch identifier.
     */  
    function _chunkString($string, $pattern, $current)
    {
        $output = array(array('',''));
        $last = $current;
        
        //The output will be an alternating array of index, string, with indexes on the odd array indexes.
        $splitString = preg_split($pattern, $string, -1, PREG_SPLIT_DELIM_CAPTURE);

        //$i is used for input, $j is used for output
        for($i=0, $j=0; $i < count($splitString); $i+=2)
        {
            if($splitString[$i] !== '')
            {
                //[0] = index, [1] = string... Note the concat on the string assignment...
                $output[$j][0] = $current;
                $output[$j][1] .= $splitString[$i];
                
                if(isset($splitString[$i+1]) && isset($splitString[$i+2]))
                {
                    $last = $current;
                    $current = $splitString[$i+1];

                    //Key of ISR. Output init is necessary because of the concat above
                    if($last !== $current)
                    {
                        $j++;
                        $output[$j] = array('', '');
                    }
                }
            }   
        }
        return $output;
    }

    /**
     * Applies both codepage and colour processing to a string using the given parameters.
     * 
     * @public
     * @param string (String) The string to be processed.
     * @param cProc (fnCallback) The colour processor to be used.
     * @param cMap (Array) The colour map to be used.
     * @return (Mixed) The output of the combined processors used.
     */
    function process($string, $cProc=null, $cMap=null)
    {
        $string = $this->processCodePages($string);
        return $this->processColours($string, $cProc, $cMap);
    }

    /**
     * Processes codepage switches in a string. Original credit goes to Victor Van Vlaardingen because this is a derivative of his codepage_convert function.
     * 
     * @public
     * @param string (String) The string to be processed.
     * @param cpFile (String) The codepage file to be used.
     * @return (String) A string with codepage switches removed & entities appropriately translated.
     */
    function processCodePages($string, $cpFile='xhtml_table.php')
    {
        static $cp;
        if($cp[$cpFile] == null)
        {
            $cp[$cpFile] = include(dirname(__FILE__).'/codepage-tables/'.$cpFile);
        }

        $codePage =& $cp[$cpFile]['L'];
        $output = '';
        $len = strlen($string);
        
        for($i=0; $i < $len; $i++)
        {
            $cChar = $string[$i];
            if($cChar === '^' && isset($cp[$cpFile][$string[$i+1]]))
            {
                $i++;
                $codePage =& $cp[$cpFile][$string[$i]];
                continue;
            }

            $output .= (isset($codePage[$cChar])) ? ($codePage[$cChar]) : ($cChar);
        }
        
        return $output;
    }

    /**
     * Processes colours in the given string.
     * 
     * @public
     * @param string (String) The string to process.
     * @param proc (fnCallback) The colour processor to use.
     * @param map (Array) The colour map to use.
     * @return (Mixed) The output of the colour processor used.
     */
    function processColours($string, $proc=null, $map=null)
    {
        if($proc == null)
            $proc = $this->m_sColourProc;
        if($map == null)
            $map = $this->m_aColourMap;

        $string = $this->_chunkString($string, $this->m_sColourPattern, '8');
        return call_user_func($proc, $string, $map);  
    }

    /**
     * Strips codepage switches from a string.
     * This should be used in preference to a blank code page because it is much faster.
     * 
     * @public
     * @param string (String) The string to strip.
     * @return (String) A copy of the input string with codepage switches removed.
     */
    function stripCodePages($string)
    {
        return str_replace(array('^E', '^C', '^L', '^:', '^G', '^T', '^B', '^J'), '', $string);
    }

    /**
     * Strips colour switches from a string.
     * This should be used in preference to a 'strip' processor because it is much faster.
     * 
     * @public
     * @param string (String) The string to strip.
     * @return (String) A copy of the input string with colour switches removed.
     */
    function stripColours($string)
    {
        return str_replace(array('^0', '^1', '^2', '^3', '^4', '^5', '^6', '^7', '^8'), '', $string);
    }

    /**
     * Sets the default colour processor to use if one is not given at runtume.
     * 
     * @public
     * @param proc (fnCallback) Colour processor to use.
     * @return (Void)
     */
    function setColourProcessor($proc)
    {
        if(function_exists($proc))
            $this->m_sColourProc = $proc;
    }

    /**
     * Sets the default colour map to be used if one is not given at runtime.
     * Some processors supply their own default.
     * 
     * @public
     * @param map (Array) The default colour map to use.
     * @return (Void)
     */
    function setColourMap($map)
    {
        $this->m_aColourMap = $map;
    }
}

/**
 * Callback processor function that processes a chunked string and reconstructs it. 
 * 
 * @param input (Array) The chunked input array.
 * @param inMap (Array) A map specified at runtime to use in preference of the default one.
 * @return (String) The reconstructed string.
 */
function __sProcColourCSSClass($input, $inMap=null)
{
    static $defMap;
    if($inMap == null)
    {
        if($defMap == null)
            $inMap = $defMap = array('lfsColZero', 'lfsColOne', 'lfsColTwo', 'lfsColThree', 'lfsColFour', 'lfsColFive', 'lfsColSix', 'lfsColSeven', 'lfsColEight', 'lfsColEight');
        else
            $inMap = $defMap;
    }
    $output = '';
    for($i=0; $i < count($input); $i++)
        $output .= "<span class=\"{$inMap[$input[$i][0]]}\">{$input[$i][1]}</span>";

    return $output;
}

/**
 * Callback processor function that processes a chunked string and reconstructs it. 
 * 
 * @param input (Array) The chunked input array.
 * @param inMap (Array) A map specified at runtime to use in preference of the default one.
 * @return (String) The reconstructed string.
 */
function __sProcColourCSSStyle($input, $inMap=null)
{
    static $defMap;
    if($inMap == null)
    {
        if($defMap == null)
            $inMap = $defMap = array('000', 'F00', '3F0', 'FF0', '30F', 'F0F', '0FF', 'FFF', '999', '999');
        else
            $inMap = $defMap;
    }
    $output = '';
    for($i=0; $i < count($input); $i++)
    	if($input[$i][0] != '')
	        $output .= "<span style=\"color: #{$inMap[$input[$i][0]]}\">{$input[$i][1]}</span>";

    return $output;
}
?>
