<?php
/**
 * Provides a class for parsing personal stats from LFSWorld.
 *
 * @file pst.php
 * @author Michael Simons <ixulai@gmail.com>
 * @license http://creativecommons.org/licenses/by/2.5/ Creative Commons
 */

/**
 * Include base provider class. 
 */
require_once(dirname(__FILE__).'/base.php');    
 
/**
 * Class for parsing LFSWorld personal stats.
 * 
 * @class LFSWorldPersonalStats
 * @version 1.1.00
 * @author Michael Simons <ixulai@gmail.com>
 * @license http://creativecommons.org/licenses/by/2.5/ Creative Commons
 */
class LFSWorldPersonalStats extends LFSWorldDataProvider
{
    /**
	 * The base url of this data source.
	 * @protected
	 */
	var $m_sURL = 'http://www.lfsworld.net/pubstat/get_stat2.php?action=pst';
	
	/**
	 * An array of meta data possibly useful to other parts of a program. (Such as tarpit duration)
	 * @protected
	 */
    var $m_aSrcMetaData = array('lifeTime' => 60, 'actualTarpit' => 5);

    /**
     * Parses the raw data from LFSWorld in to an array.
     *
     * Raw List Format
     *
     * <pre>
     * requires: &racer=RACER
     * optional: -
     * returns: distance in metres
     *          fuel burnt in dl          (NEW S2 stat)
     *          laps
     *          hosts joined
     *          wins
     *          second
     *          third
     *          finished
     *          quals
     *          pole
     *          online credits
     *          drags
     *          drag wins
     * (1.1)	online
     * (1.1)	hostname
     * </pre>
	 *
	 * @public
	 * @param rawList (String) The raw list retrieved from LFSWorld that needs to be parsed.
	 * @param[out] outputList (Array) A reference to an array that will store the parsed output.
	 * @return (Boolean) True on success, false otherwise.
     */
    function parse($rawList, &$outputList)
    {
        $tmp = array();
        $map = array(   'distance', 'fuel', 'laps', 'hosts_joined',
                        'wins', 'second', 'third', 'finished', 'quals',
                        'pole', 'credits', 'drags', 'drag_wins');
        if($this->m_aParams['version'] == 1.1)
        {
            $map[] = 'online';
            $map[] = 'hostname';
        }
        
        $tmp = explode("\n", $rawList);
        if(count($tmp) < count($map))
            return false;

        for($i=0; $i < count($map); $i++)
            $outputList[$map[$i]] = $tmp[$i];
        
        return true;
    }
    
    /**
     * Checks and adjusts input parameters where required & possible.
	 *
	 * @public
	 * @param params (&Array) An array of the params that are set.
	 * @return (Boolean) True if the parameters are ok, false otherwise.
     */
	function initParams(&$params)
	{
        if(!isset($params['version']) || ($params['version'] != 1.1 && $params['version'] != 1.0))
            $params['version'] = 1.1;  
		return (parent::initParams($params) && isset($params['racer']));
	}
}

?>
