<?php
/**
 * Provides a class for parsing the hosts list from LFSWorld.
 *
 * @file hosts.php
 * @author Michael Simons <ixulai@gmail.com>
 * @license http://creativecommons.org/licenses/by/2.5/ Creative Commons
 */

/**
 * Include base provider class. 
 */
require_once(dirname(__FILE__).'/base.php'); 
 
/**
 * Class for parsing LFSWorld hosts list.
 * 
 * @class LFSWorldHostsList
 * @version 2.5.00
 */
class LFSWorldHostsList extends LFSWorldDataProvider
{
    /**
	 * (String) The base url of this data source.
	 * @protected
	 */
	var $m_sURL = 'http://www.lfsworld.net/pubstat/get_stat2.php?action=hosts';
	
	/**
	 * (Array) An array of meta data possibly useful to other parts of a program. (Such as tarpit duration)
	 * @protected
	 */
    var $m_aSrcMetaData = array('lifeTime' => 90, 'actualTarpit' => 60);
    
    /**
     * Parses the raw data from LFSWorld in to an array.
     *
     * Raw List Format
     *
     * <pre>
     * requires:-
     * optional:-
     * returns: HOST STRUCT:
     * 
     * HOSTNAME        char[32]
     * TMLT            byte[4] //Type, MajorVersion*10, MinorVersion, TestVersion
     * TCR             byte[4] // Track, Config, Reversed
     * MAXRACERS       byte[1]
     * CARS            ulong (4 bytes)
     * RULES           ulong (4 bytes)
     * LAPS            byte [1]
     * QUALMINS        byte [1]
     * spare           byte [1]
     * spare           byte [1]
     * NROFRACERS      byte[1]
     * RACERNAMES      char[24 * NROFRACERS]
     *
     * TMLT means:
     * Type0 : old S1 / 1 : new S1 / 2 : S2
     * Mainversion times 10 (e.g. 3 means version 0.3)
     * Letter e.g. 'H' in 0.3H
     * TestId zero : official / non-zero : test patch id
     * (*) TRC is in numbers:
     * 000 = bl1
     * 001 = bl1r
     * 010 = bl2
     * ..
     * 220 = fe3
     * 221 = fe3r
     * etc.
     * </pre>
	 *
	 * @public
	 * @param rawList (String) The raw list retrieved from LFSWorld that needs to be parsed.
	 * @param[out] outputList (Array) A reference to an array that will store the parsed output.
	 * @return (Boolean) True on success, false otherwise.
     */
    function parse($rawList, &$outputList)
    {
        //Offset var for position
        $offset = 0; $t=0;

        while(true)
        {
            $blob = substr($rawList, $offset, 53);
            //If chunk len is less than 53 then it can't be valid
            if(strlen($blob) == 53)
            {
                $data = $racers = array();

                //We could use L1cars & L1rules but that limits cars and rules to 31 bits(4 byte int, 1 sign bit)
                $data = unpack('a32host_name/C1server_type/C1major_version/C1minor_version/C1test_version/C1track/C1config/C1reversed/C1max_players/C4cars/C4rules/C1laps/C1qualify_mins/x2/C1players', $blob);
                $offset+= 53;

                //Get racers
                for($i=0; $i < $data['players']; $i++)
                {
                    $racers[] = trim(substr($rawList, $offset, 24));
                    $offset += 24;
                }

                //Laps parsing.
                $data['hours'] = 0;
                if($data['laps'] > 100)
                {
                    if($data['laps'] < 191)
                    {
                        //Laps are in 10 step increment range (101-191)
                        $data['laps'] = (($data['laps'] - 100) * 10) + 100;
                    }
                    else
                    {
                        //Laps are hours
                        $data['hours'] = $data['laps'] - 190;
                        $data['laps'] = 0;
                    }
                }

                //Formatting
                //About as elegant as a wet fart but also very fast
                //Turn car and rule bit encoding in to bit strings
                $data['cars'] = strrev( str_pad(decbin($data['cars4']), 8, '0', STR_PAD_LEFT).
                                        str_pad(decbin($data['cars3']), 8, '0', STR_PAD_LEFT).
                                        str_pad(decbin($data['cars2']), 8, '0', STR_PAD_LEFT).
                                        str_pad(decbin($data['cars1']), 8, '0', STR_PAD_LEFT));
                                        
                $data['rules'] = strrev(str_pad(decbin($data['rules4']), 8, '0', STR_PAD_LEFT).
                                        str_pad(decbin($data['rules3']), 8, '0', STR_PAD_LEFT).
                                        str_pad(decbin($data['rules2']), 8, '0', STR_PAD_LEFT).
                                        str_pad(decbin($data['rules1']), 8, '0', STR_PAD_LEFT));
                                        
                //Compound the 3 integers that make up the track in to 1 var
                $data['track'] = $data['track'].$data['config'].$data['reversed'];
                
                //Unset transformed data
                unset($data['config'], $data['reversed'],
                        $data['cars1'], $data['cars2'], $data['cars3'], $data['cars4'],
                        $data['rules1'], $data['rules2'], $data['rules3'], $data['rules4']);

                //Put the racers in the output
                $data['racers'] = $racers;

                //Version formatting
                $data['major_version'] /= 10;
                $data['minor_version'] = chr($data['minor_version']);

                //Append to output list
                $outputList[] = $data;
            }
            else
            {
                break;
            }
        }
        return true;
    }
}

?>
