<?php
/**
 * Provides a class for parsing the highlights from LFSWorld.
 *
 * @file highlights.php
 * @author Michael Simons <ixulai@gmail.com>
 * @license http://creativecommons.org/licenses/by/2.5/ Creative Commons
 */

/**
 * Include base provider class. 
 */
require_once(dirname(__FILE__).'/base.php');    
 
/**
 * Class for parsing LFSWorld highlights.
 * 
 * @class LFSWorldHighlights
 * @version 2.1.00
 */
class LFSWorldHighlights extends LFSWorldDataProvider
{
    /**
	 * (String) The base url of this data source.
	 * @protected
	 */
	var $m_sURL = 'http://www.lfsworld.net/highlight_countries/highlights___COUNTRY__.txt';
	
	/**
	 * (Array) An array of data possibly useful to other parts of the program.
	 * @protected
	 */
    var $m_aSrcMetaData = array('lifeTime' => 60, 'actualTarpit'=>0);
    
	/**
	 * (String) Specific country to get highlights for.
	 * @protected
	 */
    var $m_sCountry = null;
    
	/**
	 * (Boolean) Whether or not to parse the strings in to arrays or leave them as raw strings.
	 * @protected
	 */
    var $m_bNoParse = false;
    
	/**
	 * (Array) Conversion array for HTML => LFS Colour code.
	 * @protected
	 */
    var $m_aColourReplace = array('<FONT COLOR="#000000"' => '^0',
                                  '<FONT COLOR="#FF0000"' => '^1',
                                  '<FONT COLOR="#00FF00"' => '^2',
                                  '<FONT COLOR="#FFFF00"' => '^3',
                                  '<FONT COLOR="#0000FF"' => '^4',
                                  '<FONT COLOR="#FF00FF"' => '^5',
                                  '<FONT COLOR="#00FFFF"' => '^6',
                                  '<FONT COLOR="#FFFFFF"' => '^7');
    
    /**
     * Parses the raw data from LFSWorld in to an array.
	 *
	 * @TODO add hlights specific car codes to car code lookup
	 * @public
	 * @param rawList (String) The raw list retrieved from LFSWorld that needs to be parsed.
	 * @param[out] outputList (Array) A reference to an array that will store the parsed output.
	 * @return (Boolean) True on success, false otherwise.
     */
    function parse($rawList, &$outputList)
    {
        $rawList = explode("\n", $rawList);
        
        if($this->m_bNoParse == true)
        {
            if(is_array($outputList))
                $outputList = array_merge($outputList, $rawList);
            else
                $outputList = $rawList;

            return true;
        }
        
        $pbHighs = array();
        $wonHighs = array();
        $unkHighs = array();

        $colourSrc = array_keys($this->m_aColourReplace);
        $colourDst = array_values($this->m_aColourReplace);
        foreach($rawList as $line)
        {
            $idSect = substr($line, 0, 50);
            if(stristr($idSect, ' drove a PB: '))
            {
                //%name drove a PB: %time on %track with %car
                preg_match("/(.*) drove a PB: ([0-9]{1,}):([0-9]{2})\.([0-9]{3}) on (.*) with (.*)/i", $line, $tmp);                
                list($line, $pb['racer'], $mins, $secs, $msecs, $pb['track'], $pb['car']) = $tmp;
                $pb['time'] = ($msecs/10) + ($secs*100) + (($mins*100)*60);
                $pbHighs[] = array($pb, $line);
            }
            else if(stristr($idSect, ' won the race held on '))
            {
                //%name won the race held on %server                
                $line = str_replace('</FONT>', '', $line);
                $line = str_replace($colourSrc, $colourDst, $line); 
                $won = array();
                $won['name'] = substr($line, 0, strpos($line, ' won the race held on '));
                $offset = strlen($won['name']) + 22;
                $won['server'] = substr($line, $offset);
                $wonHighs[] = $won;
            }
            else if(trim($line) != '')
            {
                $unkHighs[] = $line;
            }
        }

        $outputList['pbs'] = $pbHighs;
        $outputList['won'] = $wonHighs;
        $outputList['unknown'] = $unkHighs;
        
        return true;
    }
    
    /**
     * Checks and adjusts input parameters where required & possible.
	 *
	 * @public
	 * @param params (&Array) An array of the params that are set.
	 * @return (Boolean) True if the parameters are ok, false otherwise.
     */
    function initParams(&$params)
    {       
        if($params['noParse'] == true)
            $this->m_bNoParse = true;

        if(isset($params['country']))
        {
            $this->m_sCountry = str_replace(' ', '_', $params['country']);
            //reinit params since this provider doesn't take any and make sure base::initParams knows we don't want compression
            $params = array();
            $params['c'] = 0;
        }
        else
        {
            return false;
        }

        if(!parent::initParams($params))
            return false;
            
        return true;
    }

    /**
     * LFSWorldDataProvider::getURL
	 * 
     * Accessor method for retrieving the base url of the data source.
	 *
	 * @public
	 * @TODO test
	 * @return The base url of the data source as a string.
     */
    function getURL()
    {
        if($this->m_sCountry != null)
            return str_replace('__COUNTRY__', $this->m_sCountry, $this->m_oRequest->_url->getURL());
        else
            return $this->m_oRequest->_url->getURL();
    }
}

?>
