<?php
/**
 * Provides a base class for parsing basic LFSWorld data sources.
 *
 * @file base.php
 * @author Michael Simons <ixulai@gmail.com>
 * @license http://creativecommons.org/licenses/by/2.5/ Creative Commons
 */

/**
 * Include required base files
 */
require_once('HTTP/Request.php');
require_once(PPF_ROOT_DIR.'/utilities/decompress.php');
require_once(PPF_ROOT_DIR.'/utilities/errorcheck.php');

/**
 * Base class for parsing basic LFSWorld data sources.
 * 
 * @class LFSWorldDataProvider
 * @version 2.1.00
 */
class LFSWorldDataProvider
{
    /**
     * (Object) The PEAR HTTP_Request object.
     * @protected
     */
    var $m_oRequest = null;
    
	/**
	 * (Array) An array of meta data possibly useful to other parts of a program. (Such as tarpit duration)
	 * @protected
	 */
    var $m_aSrcMetaData = array();
    
    /**
	 * (Array) An array of parameters to be merged with the runtime and static parameters in order to specify how the parser should execute.
	 * @protected
	 */
    var $m_aParams = array();
    
    /**
     * LFSWorldDataProvider base class Ctor.
	 *
	 * @public
	 * @param srcMetaData (Array) An array of data to be passed to a cache.
	 * @param params (Array) An array of params to be merged with the runtime and static params in order to specify how the parser should execute.
	 * @return (Void)
     */
    function LFSWorldDataProvider($srcMetaData=array(), $params=array())
    {
        $this->m_aSrcMetaData = array_merge($this->m_aSrcMetaData, $srcMetaData);
        $this->m_aParams = array_merge($this->m_aParams, $params);
        $this->m_oRequest = new HTTP_Request(null, array('method'=>HTTP_REQUEST_METHOD_GET, 'timeout'=>2));
        $this->_init();
    }

    /**
     * Initialisation function called at the end of the ctor.
     * This func allows for per-provider init without having to recreate a base class compatible ctor.
     *
     * @protected
     */ 
    function _init()
    {
    }
      
    /**
     * Parses the raw data from LFSWorld in to an array.
	 *
	 * @public
	 * @param rawList (String) The raw list retrieved from LFSWorld that needs to be parsed.
	 * @param[out] outputList (Array) A reference to an array that will store the parsed output.
	 * @return (Boolean) True on success, false otherwise.
     */
    function parse($rawList, &$outputList)
    {
        $rawList = explode("\n", trim($rawList));

        if(sizeof($rawList) == 0)
        {
            $GLOBALS['PPFLog'][] = 'Empty input to LFSWorld Parser!';
            return false;
        }
        
        foreach($rawList as $entry)
        {
            $output = $this->_parseLine($entry);
            if($output != false)
                $outputList[] = $output;
        }

        return true;
    }
    
    /**
     * Makes the HTTP request and performs decompression / error checking etc.
	 *
	 * @public
	 * @return (Mixed) Parsed data on success (usually array), false otherwise.
     */   
    function getData()
    {
        //Set URL
        $this->m_oRequest->HTTP_Request($this->getURL());
        //Make request
        $this->m_oRequest->sendRequest(true);
        $data = ($this->m_oRequest->getResponseCode() != 200) ? (false) : ($this->m_oRequest->getResponseBody());

        //Did request go OK?
        if($data == false)
        {
            $GLOBALS['PPFLog'][] = 'Unable to connect to LFSWorld';
            return false;
        }
        
        //Errors are not compressed, so was there an LFSWorld error?
        if(isLFSWorldError($data))
        {
            $GLOBALS['PPFLog'][] = 'LFSWorld error was encountered! ('.$data.')';
            return false;
        }
        
        //Decompress data if it was requested in a compressed form
        if(isset($this->m_aParams['c']))
            $data = PPFDecompress($data, $this->m_aParams['c']);
        
        //Did decompression go ok?
        if($data == false)
        {
            $GLOBALS['PPFLog'][] = 'LFSWorld raw data decompression failed!';
            return false;
        }
        
        $this->saveRequestTime();
        
        $parsed = null;
        $this->parse($data, $parsed);
        return $parsed;
    }
    
    /**
     * Accessor method for retrieving the params stored in this instance.
	 *
	 * @public
	 * @return (Array) An array of params stored in this instance.
     */
    function getParams()
    {
        return $this->m_aParams;
    }
    
    /**
     * Accessor method for retrieving the meta data stored in this instance.
	 *
	 * @public
	 * @return (Array) An array of meta data stored in this instance.
     */
    function getSrcMetaData()
    {
        return $this->m_aSrcMetaData;
    }
    
    /**
     * Accessor method for retrieving the base url of the data source.
	 *
	 * @public
	 * @return (String) The base url of the data source as a string.
     */
    function getURL()
    {
        return $this->m_oRequest->_url->getURL();
    }
    
    /**
     * Checks and adjusts input parameters where required & possible.
	 *
	 * @public
	 * @param params (&Array) An array of parameters for the provider.
	 * @return (Boolean) True if the parameters provided are ok, false otherwise.
     */  
    function initParams(&$params)
    {
        //Reset request object
        $this->m_oRequest->HTTP_Request($this->m_sURL, array());
        
        //IDK && Username/password handling
        if(isset($params['IDK']) && strlen($params['IDK']) == 32)
        {
            if(isset($params['user']))
                unset($params['user']);
            if(isset($params['pass']))
                unset($params['pass']);
        }
        else if(isset($params['user']) && isset($params['pass']))
        {
            if(isset($params['IDK']))
                unset($params['IDK']);
            
            $params['user'] = urlencode($params['user']);
            if(strlen($params['pass']) != 32)
                $params['pass'] = md5($params['pass']);
        }
        
        //Compression is good so if it isn't enabled yet, enable it
        if(!isset($params['c']))
            $params['c'] = 2;
        else
            $params['c'] = (int)$params['c'];

        //But compression is no good if the server doesn't have gzip
        //Test to make sure server can decompress, and if it can't then disable compression
        if(PPFCanDecompress($params['c']) == false)
            unset($params['c']);
        
        //Premium Stats "smart" handling
        //It will only use premium if ps is set && the tarpit for a provider is still active
        //@TODO Check accuracy of actualTarpit times (especially teams)
        if(isset($params['ps']))
        {
            $lastRequest = $this->getLastRequestTime();
            if(time() - ($lastRequest+$this->m_aSrcMetaData['actualTarpit']) < 0)
                $params['ps'] = 1;
            else
                unset($params['ps']);            
        }
        
        //Put params to the URL
        foreach($params as $param=>$value)
            $this->m_oRequest->addQueryString($param, $value);
            
        $this->m_aParams = $params;

        return true;
    }
    
    /**
     * Retrieves a UNIX timestamp representing the last time this provider made a request.
	 *
	 * @public
	 * @return (Integer) UNIX timestamp representing last time the provider made a request, or 0 if a request has never been made or can not be determined.
     */
    function getLastRequestTime()
    {
        if(file_exists(PPF_DATA_DIR.'/'.get_class($this).'.lrt'))
            return filemtime(PPF_DATA_DIR.'/'.get_class($this).'.lrt');
            
        return 0;
    }
    
    /**
     * Saves the current time as the last time of a request for this provider.
	 *
	 * @public
	 * @return (Void)
     */
    function saveRequestTime()
    {
        $s = fopen(PPF_DATA_DIR.'/'.get_class($this).'.lrt', 'w+');
        fclose($s);
    }
}
?>
