<?php
/**
 * @file lfswinterface.php
 * @author Michael Simons <ixulai@gmail.com> 
 * @license http://creativecommons.org/licenses/by/2.5/ Creative Commons
 */

/**
 * Include generic ppf interface
 */
require_once(dirname(__FILE__).'/genericppfinterface.php');

/**
 * @class LFSWInterface
 * @version 1.0.00
 */
class LFSWInterface extends GenericPPFInterface
{  
    /**
     * Gets a host list from LFSWorld.
     *
     * @public
     * @see LFSWorldHostsList
     * @return (Mixed) A host listing in array format on success, false on failure.
     */
    function getHostList()
    {
        if(!class_exists('LFSWorldHostsList'))
            include PPF_ROOT_DIR.'/providers/LFSW/hosts.php';
        $provider = new LFSWorldHostsList();
        return $this->getData($provider, array());
    }

   /**
     * Gets a chart listing from LFSWorld.
     *
	 * @public
     * @param track (String) Track in numeric format.
     * @param car (String) Car in three letter format.
     * @param control (String) Optional. Control method as a string.
     * @todo Document control methods.
     * @see LFSWorldChart
     * @return (Mixed) The chart listing in an array on success, false on failure.
     */
    function getChart($track, $car, $control=null)
    {
        if(!class_exists('LFSWorldChart'))
            include PPF_ROOT_DIR.'/providers/LFSW/ch.php';
            
        $params = array('track'=>$track, 'car'=>$car);
        if($control != null)
            $params['control'] = $control;

        $provider = new LFSWorldChart();            
        return $this->getData($provider, array('providerParams'=>$params));
    }

    /**
     * Get a list of hotlaps from LFS-World.
     *
     * @public
     * @param racer (String) The name of the racer to fetch hotlaps for.
     * @see LFSWorldHotlaps
     * @return (Mixed) An array of hotlaps on success, false on failure.
     */
    function getHotlaps($racer)
    {
        if(!class_exists('LFSWorldHotlaps'))
            include PPF_ROOT_DIR.'/providers/LFSW/hl.php';
            
        $provider = new LFSWorldHotlaps();
        return $this->getData($provider, array('providerParams' => array('racer' => $racer)));
    }

    /**
     * Gets a list of PBs from LFS-World.
     * 
     * @public
     * @param racer (String) The name of the racer to get a list of PBs for. 
     * @see LFSWorldPersonalBests
     * @return (Mixed) An array of PBs on success, false on failure.
     */
    function getPBs($racer)
    {
        if(!class_exists('LFSWorldPersonalBests'))
            include PPF_ROOT_DIR.'/providers/LFSW/pb.php';
            
        $provider = new LFSWorldPersonalBests();
        return $this->getData($provider, array('providerParams' => array('racer' => $racer)));
    }

    /**
     * Gets PST data from LFS-World. 
     *
     * @public
     * @param racer (String) The racer name to fetch PST data for.
     * @see LFSWorldPersonalStats
     * @return (Mixed) An array of PST data on success, false on failure.
     */
    function getPST($racer)
    {
        if(!class_exists('LFSWorldPersonalStats'))
            include PPF_ROOT_DIR.'/providers/LFSW/pst.php';
            
        $provider = new LFSWorldPersonalStats();
        return $this->getData($provider, array('providerParams' => array('racer' => $racer)));
    }

    /**
     * Gets the full list of teams from LFS-World.
     *
     * @public
     * @see LFSWorldTeams
     * @return (Mixed) An array of teams on success, false on failure.
     */
    function getTeams()
    {
        if(!class_exists('LFSWorldTeamsList'))
            include PPF_ROOT_DIR.'/providers/LFSW/teams.php';
            
        $provider = new LFSWorldTeamsList();
        return $this->getData($provider, array());
    }

    /**
     * Gets a list of WRs from LFS-World.
     *
     * @public
     * @param control (String) An optional 'control method filter'.  
     * @see LFSWorldWorldRecords
     * @return (Mixed) An array of WRs on success, false on failure.
     */
    function getWRs($control=null)
    {
        if(!class_exists('LFSWorldWorldRecords'))
            include PPF_ROOT_DIR.'/providers/LFSW/wr.php';
            
        $params = array();
        if($control != null)
            $params['control'] = $control;
        
        $provider = new LFSWorldWorldRecords();
        return $this->getData($provider, array('providerParams' => $params));
    }

    /**
     * Gets a list of highlights from LFS-World.
     *
     * @public
     * @param country (String) The (case-sensitive) country name to get highlights for.
     * @param noParse (Boolean) Set to true if you want the LFS-World formatting to remain in tact. 
     * @see LFSWorldHighlights
     * @return (Mixed) An array of highlights on success, false on failure or when the country does not have any highlights.
     */
    function getHighlights($country, $noParse=false)
    {
        if(!class_exists('LFSWorldHighlights'))
            include PPF_ROOT_DIR.'/providers/LFSW/highlights.php';
            
        $noParse = (bool)$noParse;
        $provider = new LFSWorldHighlights();
        return $this->getData($provider, array('providerParams' => array('country' => $country, 'noParse' => $noParse)));
    }

    /**
     * Gets a list of hotlap log entries from LFS-World.
     *
     * @public
     * @param params (Array) An array of filter options for this provider. 
     * @see LFSWorldHotlapLog
     * @return (Mixed) An array of hotlap log entries on success, false on failure.
     */
    function getHotlapLog($params=array())
    {
        if(!class_exists('LFSWorldHotlapLog'))
            include PPF_ROOT_DIR.'/providers/LFSW/hllog.php';
            
        $provider = new LFSWorldHotlapLog();
        return $this->getData($provider, array('providerParams' => $params));
    }
}
?>
