<?php
/**
 * Provides a generic SQL cache provider.
 *
 * @file lfsworldfilecache.php
 * @author Mike Simons
 * @license http://creativecommons.org/licenses/by/2.5/
 */

/**
 * Provides a generic SQL cache provider.
 * 
 * @class LFSWorldSQLCache
 * @version 1.0.00
 */
class LFSWorldSQLCache
{
	/**
	 * (Array) An array of options to use as default configuration values.
	 * @protected
	 */
    var $m_aDefaults = array(   'lifeTime' => 60,
                                'noValidityCheck' => false,
                                'queries' => array( 'count' => 'SELECT COUNT(lastUpdate) AS rowCount FROM {{table}} WHERE lastUpdate >= {{threshold}}',
                                                    'get' => 'SELECT * FROM {{table}} WHERE lastUpdate >= {{threshold}} {{where}} {{limit}} {{order}}',
                                                    'save' => 'REPLACE INTO {{table}} {{fields}} VALUES {{values}}',
                                                    'oldThreshold' => 'SELECT DISTINCT lastUpdate FROM {{table}} ORDER BY lastUpdate DESC LIMIT 1, 1'),
                                'resultsPerPage' => 0,
                                'page' => 1,
                                'getTransformer' => '',
                                'saveTransformer' => '',
                                'where' => '',
                                'order' => '');

    /**
     * LFSWorldSQLCache Ctor.
	 *
	 * @public
	 * @param params (Array) An array of options to configure the cache.
	 * @return (Void)
     */
    function LFSWorldSQLCache($params)
    {
        $this->m_aDefaults = array_merge_replace($this->m_aDefaults, $params);
    }
    
    /**
     * Retrieve data from the cache.
	 *
	 * @public
	 * @param params (Array) An array of params to be used in querying the cache. (Cache group, entry key etc)
	 * @return (Array) An array including meta data, and retrieved data on success, false on failure.
     */
    function get($params)
    {
        $params = array_merge_replace($this->m_aDefaults, $params);
        $threshold = ($params['noValidityCheck'] == true) ? ($this->_getLastThreshold($params)) : (time() - $params['lifeTime']);
        
        $totalRows = $this->_countRows($params, $threshold); 
        if($totalRows == 0)
        {
            $GLOBALS['PPFLog'][] = 'Zero rows available from cache!';
            return false;
        }
        
        $pageData = $this->_createPaginationData($params, $totalRows);

        if($params['where'] != '')
            $params['where'] = "AND ({$params['where']})";
            
        $q = str_replace(   array('{{table}}', '{{threshold}}', '{{where}}', '{{limit}}', '{{order}}'),
                            array($params['table'], $threshold, $params['where'], $pageData['limitString'], $params['order']),
                            $params['queries']['get']); 
        
        $return = array('totalCount' => $totalRows,
                        'page' => $pageData['page'],
                        'resultsPerPage' => $pageData['resultsPerPage'],
                        'pages' => $pageData['pages']);
                        
        $result = mysql_db_query($params['dbName'], $q, $params['dbResource']);

        if(!$result || mysql_num_rows($result) == 0)
        {
            $GLOBALS['PPFLog'][] = 'Cache query has zero rows or is invalid!';
            return false;
        }
        else
        {
            $data = array();
            while(($row = mysql_fetch_assoc($result)) != false)
                $data[] = $row;

            if(function_exists($params['getTransformer']))
                $data = call_user_func($params['getTransformer'], &$this, $data, $params);
        }
        
        $return['data'] = $data;
        $return['lastUpdate'] = $data[0]['lastUpdate'];
        
        return $return;        
    }

    /**
     * Save data to the cache.
	 *
	 * @public
	 * @param data (Mixed) The data to cache.
	 * @param params (Array) An array of params to be used in saving the data. (Cache group, entry key etc)
	 * @return (Void)
     */
    function save($data, $params)
    {
        $params = array_merge_recursive($this->m_aDefaults, $params);

        if(function_exists($params['saveTransformer']))
            $data = call_user_func($params['saveTransformer'], &$this, $data, $params);
        
        reset($data);
        list($k, $v) = each($data);
        $fields = '('.implode(', ', array_keys($data[$k])).')';
        $values = array_map(array($this, '_stringify'), $data);
        $values = implode(', ', $values); 
        
        $q = str_replace(   array('{{table}}', '{{fields}}', '{{values}}'),
                            array($params['table'], $fields, $values),
                            $params['queries']['save']);

        $result = mysql_db_query($params['dbName'], $q, $params['dbResource']);
 
        if(!$result || mysql_affected_rows($params['dbResource']) == 0)
        {
            $GLOBALS['PPFLog'][] = 'Failed to save cache for reasons unknown!';
            return false;
        }

        return true; 
    }
            
    /**
     * Returns a boolean indicating whether LFSWorldFileCache::get should be called immediately after a save.
	 *
	 * @public
	 * @return (Boolean) False because the file cache does not require this behaviour.
     */ 
    function requiresDataReload()
    {
        return true;
    }

    /**
     * Purges entries older than the threshold from the cache.
     * Be warned that setting the purge threshold too low will render the fallback feature of interfaces useless.
     * 
     * @public
     * @param params (Array) An array of params to pull the purge query data from.
     * @return (Void)
     */
    function purgeOld($params)
    {
        $threshold = $this->getLastUpdate($params);
          
        //delete all data older than threshold
    }

    /**
     * Creates a unique key for the data provided.
	 *
	 * @public
	 * @param uniqData (String) A unique string to create a further unique identifier.
	 * @return (String) A unique identifier for the data given.
     */
    function createKey()
    {
        return ' ';
    }
    
    /**
     * Returns the last update of the last cache fetched.
	 *
	 * @public
	 * @param params (Array) Array of params to use in getting last update. (Cache key, group etc)
	 * @return (Integer) 0 if the update time could not be retrieved, unix timestamp of last update otherwise.
     */
    function getLastUpdate($params)
    {
        $params = array_merge_replace($this->m_aDefaults, $params);
        
        $q = str_replace(   array('{{table}}'),
                            array($params['table']),
                            $params['queries']['oldThreshold']);

        $results = mysql_db_query($params['dbName'], $q, $params['dbResource']);
        if($results && mysql_num_rows($results) == 1)
        {
            $row = mysql_fetch_assoc($results);
            return $row['lastUpdate'];
        }

        return 0;
    }
    
    /**
     * Gets the total entry count within a specified threshold.
     *
     * @param params (Array) Parameters to be used in getting the total rows count.
     * @param threshold (Integer) Threshold in seconds.
     * @return (Integer) Total entry count.
     */
    function _countRows($params, $threshold)
    {
        $q = str_replace(   array('{{table}}', '{{threshold}}'),
                            array($params['table'], $threshold),
                            $params['queries']['count']);

        $result = mysql_db_query($params['dbName'], $q, $params['dbResource']);
        if(!$result || !mysql_num_rows($result))
            return 0;

        $result = mysql_fetch_assoc($result);
        return $result['rowCount'];
    }

    /**
     * Creates an array of meta data that makes it easier to paginate results.
     * 
     * @protected
     * @param params (Array)
     * @param totalRows (Integer) Total rows available.
     * @return (Array) Meta data for pagination such as current page, results per page, num of pages etc.
     */
    function _createPaginationData($params, $totalRows)
    {
        //Validate page num
        $page = (int)abs($params['page']);
        $page = ($page > 0) ? ($page) : (1);
        
        //Validate Results per page
        $rpp = ($params['resultsPerPage'] <= 0) ? ($totalRows) : ((int)abs($params['resultsPerPage']));
        $rpp = ($rpp > 0) ? ($rpp) : (1);
        
        //Calc page count
        $pages = ceil($totalRows / $rpp);
        
        //Calc start row based on page count and results per page
        $start = ($page-1) * $rpp;
        $limit = "LIMIT $start, $rpp";

        return array(   'page' => $page,
                        'resultsPerPage' => $rpp,
                        'pages' => $pages,
                        'start' => $start,
                        'limitString' => $limit);
    }

    /**
     * Takes an array of values, escapes them for SQL insertion and converts them to a string.
     * 
     * @protected
     * @param array (Array) Array of data to be 'stringified'.
     * @return (String) String form of escaped data that 'should' be ready for use in an SQL insert/update/replace statement.
     */
    function _stringify($array)
    {
        $array = array_map(array($this, '_escape'), $array);
        return '(' . implode(', ', $array) . ')';
    }

    /**
     * Escape a astring to be safely included in a query.
     * 
     * @protected
     * @param string (String) String to be escaped.
     * @return (String) The escaped string.
     */
    function _escape($string)
    {
        return '\'' . str_replace('\'', '\'\'', $string) . '\'';
    }
}

function hostSaveTransformer(&$cache, $data, $params)
{
    $lastUpdate = time();
    while(list($key, $val) = each($data))
    {
        $data[$key]['strippedHostName'] = $data[$key]['hostName'];
        $data[$key]['racers'] = implode(', ', $data[$key]['racers']);
        $data[$key]['lastUpdate'] = $lastUpdate;
    }

    return $data;
}
