<?php
/**
 * Provides an interface bridge between LFSWorld and Cache_lite.
 *
 * @file lfsworldfilecache.php
 * @author Mike Simons
 * @license http://creativecommons.org/licenses/by/2.5/
 */

/**
 * Include required files
 */
require_once('Cache/Lite.php');
require_once(dirname(__FILE__).'/../utilities/array_merge_replace.php');

/**
 * Provides an interface bridge between LFSWorld and Cache_lite.
 * 
 * @class LFSWorldFileCache
 * @version 1.2.00
 */
class LFSWorldFileCache
{
    /**
	 * (Object) An instance of a Cache_lite object, or at least an object with a compatibile interface.
	 * @protected
	 */
    var $m_oCache = null;
    
    /**
     * (Integer) Last update time of the cache file as a UNIX timestamp.
     * @protected
     */
     var $m_nLastUpdate = 0;
     
     /**
      * (Array) An array of options to use as default configuration values.
      * @protected
      */
     var $m_aDefaults = array(  'group' => 'default',
                                'noValidityCheck' => false);
      
    /**
     * LFSWorldFileCache Ctor.
	 *
	 * @public
	 * @param cache (&Object) A reference to a Cache_lite object, or at least an object with a compatible interface.
	 * @param params (Array) An array of options to configure the cache.
	 * @return (Void)
     */
    function LFSWorldFileCache(&$cache, $params=array())
    {
        $this->m_oCache =& $cache;
        $this->m_aDefaults = array_merge_recursive($this->m_aDefaults, $params);
    }
    
    /**
     * Retrieve data from the cache.
	 *
	 * @public
	 * @param params (Array) An array of params to be used in querying the cache. (Cache group, entry key etc)
	 * @return (Array) An array including meta data, and retrieved data on success, false on failure.
     */
    function get($params)
    {
        $params = array_merge_replace($this->m_aDefaults, $params);
        if(!isset($params['key']))
            trigger_error('can\'t make a cache request without a key', E_USER_ERROR);
    
        //re-init cache with params
        $ctor = get_class($this->m_oCache); 
        $this->m_oCache->$ctor($params);

        $data = $this->m_oCache->get($params['key'], $params['group'], $params['noValidityCheck']);
        $lastUpdate = $this->_getLastUpdate();
        
        return array('lastUpdate' => $lastUpdate,
                     'data' => $data);
    }
    
    /**
     * Save data to the cache.
	 *
	 * @public
	 * @param data (Mixed) The data to cache.
	 * @param params (Array) An array of params to be used in saving the data. (Cache group, entry key etc)
	 * @return (Void)
     */
    function save($data, $params)
    {
        $params = array_merge_replace($this->m_aDefaults, $params);
        if(count($params) != 0)
        {
        	//re-init cache with params
            $ctor = get_class($this->m_oCache); 
            $this->m_oCache->$ctor($params);
        }

        $this->m_oCache->save($data, $params['key'], $params['group']);
        $this->m_nLastUpdate = time();
    }

    /**
     * Creates a unique key for the data provided.
	 *
	 * @public
	 * @param uniqData (String) A unique string to create a further unique identifier.
	 * @return (String) A unique identifier for the data given.
     */
    function createKey($uniqData)
    {
        return md5($uniqData);
    }
    
    /**
     * Returns the last update timestamp of the last cache fetched.
	 *
	 * @protected
	 * @return (Integer) 0 if the update time could not be retrieved, unix timestamp of last update otherwise.
     */
    function _getLastUpdate()
    {
        $this->m_nLastUpdate = (isset($this->m_oCache->_file) && file_exists($this->m_oCache->_file)) ? (filemtime($this->m_oCache->_file)) : (0);
        return $this->m_nLastUpdate;
    }
    
    /**
     * Returns a boolean indicating whether LFSWorldFileCache::get should be called immediately after a save.
	 *
	 * @public
	 * @return (Boolean) False because the file cache does not require this behaviour.
     */    
    function requiresDataReload()
    {
        return false;
    }
}
?>
