<?php
/**
 * Provides a class for parsing the teams list from LFSWorld.
 *
 * @file teams.php
 * @author Michael Simons <ixulai@gmail.com>
 * @license http://creativecommons.org/licenses/by/2.5/ Creative Commons
 */

/**
 * Include base provider class. 
 */
require_once(dirname(__FILE__).'/base.php');    

/**
 * Class for parsing LFSWorld teams list.
 * 
 * @class LFSWorldTeamsList
 * @version 1.0.00
 * @author Michael Simons <ixulai@gmail.com>
 * @license http://creativecommons.org/licenses/by/2.5/ Creative Commons
 */
class LFSWorldTeamsList extends LFSWorldDataProvider
{
    /**
	 * The base url of this data source.
	 * @protected
	 */
    var $m_sURL = 'http://www.lfsworld.net/pubstat/get_stat2.php?action=teams';
    
	/**
	 * An array of meta data possibly useful to other parts of a program. (Such as tarpit duration)
	 * @protected
	 */
    var $m_aSrcMetaData = array('lifeTime' => 180, 'actualTarpit' => 180);
    
    /**
     * Get the entire teams database stored on lfs world
     *
     * Raw list format
     *
     * <pre>
     * requires:-
     * optional:-
     * returns: TEAM STRUCT:
     *
     * TEAM          [64]
     * TAG           [6]
     * COUNTRY       [32]
     * URL           [64]
     * BITS          [4]
     * INFO_LEN      [2]
     * INFO          [INFO_LEN]  (Note : urlencoded)
     * NR_MEMBERS    [2]
     * MEMBERS       [24] * NR_MEMBERS
     * 
     * BITS :
     * 1 : race
     * 2 : drift
     * 4 : drag
     * 8 : can apply
     * 16 : has host
     * 32 : Demo
     * 64 : S1
     * 128 : S2
     * 256 : S3
     * </pre>
     * 
	 * @public
	 * @param rawList (String) The raw list retrieved from LFSWorld that needs to be parsed.
	 * @param[out] outputList (Array) A reference to an array that will store the parsed output.
	 * @return (Boolean) True on success, false otherwise.
     */
    function parse($rawList, &$outputList)
    {
        //Offset var for position
        $offset = 0;
        
        while(true)
        {
            //grab the next chunk of static team data
            $blob = substr($rawList, $offset, 172);

            //Must be 172 chars else it can't be valid
            if(strlen($blob) == 172)
            {
                //(Re-)Init tmp vars to parse into
                $data = $racers = array();
                $memberCount = 0;
                
                //Min len 172. Parse and advance the offset to the end of the static chunk
                $data = unpack('a64name/a6tag/a32country/a64url/V1bits/v1infoLen', $blob);
                $offset += 172;

                //Get dynamic length info chunk and advance the offset
                $data['info'] = urldecode(substr($rawList, $offset, $data['infoLen']));
                $offset += $data['infoLen'];

                //Get member count and advance offset
                $memberCount = (ord($rawList{$offset})) | (ord($rawList{$offset+1}));
                $offset += 2;
                
                //Extract the entire member chunk, split it in to smaller chunks of 24, and explode
                $members = explode('&', chunk_split(substr($rawList, $offset, $memberCount*24), 24, '&'));
                //This method always adds an empty entry so just pop it off
                array_pop($members);

                //Trim the guff and advance the offset to the next chunk
                $data['members'] = array_map('rtrim', $members);
                $offset += $memberCount*24;

                //Assign member count and clean up (now) useless data
                $data['memberCount'] = $memberCount;
                unset($data['infoLen']);

                $outputList[] = $data;
            }
            else
            {
                break;
            }
        }

        return true;
    }
}

?>
