<?php
/**
 * @file genericppfinterface.php
 * @author Michael Simons <ixulai@gmail.com> 
 * @license http://creativecommons.org/licenses/by/2.5/ Creative Commons
 */

/**
 * Include required base files & setup base stuffs
 * @TODO Move defines & config stuffs to its own file
 */
require_once(dirname(__FILE__).'/utilities/array_merge_replace.php');

define('PPF_ROOT_DIR', dirname(__FILE__));
define('PPF_DATA_DIR', dirname(__FILE__).'/datacache/');

if(file_exists(dirname(__FILE__).'/PEAR/'))
	ini_set('include_path', dirname(__FILE__).'/PEAR/');

if(!is_writable(PPF_DATA_DIR))
	trigger_error('PPF data directory ('.PPF_DATA_DIR.') is not writable! Please grant write priveleges.', E_USER_ERROR);

$GLOBALS['PPFLog'] = array();

/**
 * @class GenericPPFInterface
 * @version 1.0.01
 */
class GenericPPFInterface
{    
    /**
     * (Array) An array of default execution parameters.
     * @protected
     */
    var $m_aDefaults = array(   'cacheParams' => array(),
                                'providerParams' => array(),
                                'options' => array('useFallback' => true, 'cacheUpdateOnly' => false),
                                'cache' => null);
                                
    /**
     * Ctor
     * 
     * @todo Document list of options
     * @public
     * @param params (Array) An array of options.
     * @return (Void)
     */
    function GenericPPFInterface($params=array())
    {
        //Setup default file cache
        if(!isset($params['cache']))
        {
            if(!class_exists('LFSWorldFileCache'))
                include PPF_ROOT_DIR.'/caches/lfsworldfilecache.php';
            $dir = (isset($params['cacheDir'])) ? ($params['cacheDir']) : (PPF_DATA_DIR);
            $cache = new Cache_lite(array('cacheDir' => $dir, 'lifeTime' => 60, 'automaticSerialization' => true));
            $this->m_aDefaults['cache'] = new LFSWorldFileCache($cache);
        }
        
        //Update default params
        $this->m_aDefaults = array_merge_replace($this->m_aDefaults, $params);
    }

    /**
     * Base cache and request mechanism for data.
     *
     * @public
     * @param provider (&Object) A reference to the provider object to use for this request.
     * @param params (Array) An array of parameters used to configure the instance at runtime. 
     * @return (Array) An array that includes useful meta info about the request as well as any data retrieved.
     */
    function getData(&$provider, $params)
    {
        //Init vars to be used later
        $cacheParams = $providerParams = $runtimeOptions = array();
        $data = $isFreshData = false;
        $logOffset = count($GLOBALS['PPFLog']);

        //Init params, merging defaults with RT to create final exec params
        $params = array_merge_replace($this->m_aDefaults, $params);
        $cacheParams = array_merge_replace($provider->getSrcMetaData(), $params['cacheParams']); 
        $providerParams = array_merge_replace($provider->getParams(), $params['providerParams']);
        $runtimeOptions = $params['options'];
        $cacheHandler =& $params['cache'];
        $cacheParams['group'] = 'LFSWPPF';
        
        //Make sure the provider is OK with params provided
        if($provider->initParams($providerParams))
        {
            //Check cache. In a cacheUpdateOnly situation we want to force the data to be false so it will be updated.
            if($runtimeOptions['cacheUpdateOnly'] == false)
            {
                $cacheParams['key'] = $cacheHandler->createKey($provider->getURL());
                $data = $cacheHandler->get($cacheParams);
            }
            
            if($data['data'] == false)
            {
                //Request fresh data from provider
                $data['data'] = $provider->getData();

                //Did provider exec OK?
                if($data['data'] != false)
                {
                    //Cache data
                    $isFreshData = true;
                    $cacheHandler->save($data['data'], $cacheParams);

                    //Some caches (i.e. SQL) will require a data reload after cache update
                    //If we only want to update the cache, not actually get the data then this isn't required
                    if($runtimeOptions['cacheUpdateOnly'] == false && $cacheHandler->requiresDataReload())
                        $data = $cacheHandler->get($cacheParams);
                }
                else if($runtimeOptions['useFallback'] == true) //Something went wrong with provider so use an old cache if we can
                {
                    //Disable lifetime check on cache & try cache again. Last resort to failure
                    $cacheParams['noValidityCheck'] = true;
                    $data = $cacheHandler->get($cacheParams);

                    //Provider failed and no cache exists that we can use. We have to fail at this point.
                    if($data == false)
                        $GLOBALS['PPFLog'][] = 'Cache fallback attempted but no valid data exists!';
                }
            }
        }
        else
        {
            //Bad params to the provider
            trigger_error("Bad or incomplete parameters specified for provider", E_USER_ERROR);
        }

        //Populate meta data (which will be in addition to the cache meta if the cache was hit)
		$data['isFreshData'] = $isFreshData;
		$data['errors'] = array_slice($GLOBALS['PPFLog'], $logOffset);
		$data['cacheKey'] = $cacheParams['key'];
		$data['success'] = (count($data['errors'])==0);     
       
        return $data;      
    }

    /**
     * Sets the current cache handler.
     *
     * @public
     * @param handler (&Object) A reference to a cache handler object. 
     * @return (Void)
     */
    function setCacheHandler(&$handler)
    {
        $this->m_aDefaults['cache'] =& $handler;
    }

    /**
     * Returns a reference to the current cache handler.
     * 
     * @public
     * @return (&Object) A reference to the current cache handler object.
     */
    function &getCacheHandler()
    {
        return $this->m_aDefaults['cache'];
    }
}
?>
