// LFS Vertex Shader : Car2

/*/
*	Car GGX paint shader 1.0 by DarkDrifter Nacim
/*/

#define USE_FAKE_TWO_COATS 1
#define BASE_COAT_ROUGHNESS 0.55


#define PI 3.14159265359

// Vertex shader input structure

struct VS_INPUT_PRELIT // VERT_PDST1
{
	float4	v0		: POSITION;
	float4	LitCol	: COLOR0;
	float4	AmbCol	: COLOR1;
	float2	v8		: TEXCOORD0;
};

struct VS_INPUT
{
	float4	v0		: POSITION;
	float3	Normal	: NORMAL;
	float4	Col		: COLOR;
	float2	v8		: TEXCOORD0;
};


// Vertex shader output structures

struct VS_OUTPUT_PRELIT
{
    float4	oPos		: POSITION;
	float	oFog		: FOG;
	float4	VCol		: COLOR0;
	float2	oT0			: TEXCOORD0;
};

struct VS_OUTPUT
{
    float4	oPos		: POSITION;
	float	oFog		: FOG;

	float2	oT0			: TEXCOORD0;
	float2	oT1			: TEXCOORD1;

	float4	VCol		: COLOR0; // vertex colour
	float	DirectL		: TEXCOORD2;
	float3	AmbientL	: TEXCOORD3;
	float3	Normal		: TEXCOORD4;
	float3	LightDir	: TEXCOORD5;

#ifdef SHINY
	float3	NormalEnv	: TEXCOORD6;
	float3	EyeToVEnv	: TEXCOORD7;
#endif
};

struct VS_OUTPUT_TINT
{
    float4	oPos		: POSITION;
	float	oFog		: FOG;

	float2	oT0			: TEXCOORD0;

	float4	VCol		: COLOR0;		// vertex colour
};


// Global variables

float4x4	lightinfo_mat	: register(c0);

float4		light_dir_env		: register(c4);
//
float4		light_dir		: register(c8);
float4		up_dir			: register(c9);
float4		basecolamb		: register(c10);
float4		diffcolamb		: register(c11);
float4x4	local_to_env	: register(c12);
float4		eye_local		: register(c16);
//
float4x4	local_to_texel	: register(c20);
float4		clampmin		: register(c24);
float4		clampmax		: register(c25);

float4		fog_info		: register(c90);

float RoughnessToSpecPower (float roughness)
{
#if UNITY_GLOSS_MATCHES_MARMOSET_TOOLBAG2
	// from https://s3.amazonaws.com/docs.knaldtech.com/knald/1.0.0/lys_power_drops.html
	float n = 10.0 / log2((1-roughness)*0.968 + 0.03);
	return n * n;

	// NOTE: another approximate approach to match Marmoset gloss curve is to
	// multiply roughness by 0.7599 in the code below (makes SpecPower range 4..N instead of 1..N)
#else
	float m = roughness * roughness * roughness + 1e-4f;	// follow the same curve as unity_SpecCube
	float n = (2.0 / m) - 2.0;							// http://jbit.net/%7Esparky/academic/mm_brdf.pdf
	n = max(n, 1e-4f);									// prevent possible cases of pow(0,0), which could happen when roughness is 1.0 and NdotH is zero
	return n;
#endif
}


// VSHADER_PRELIT_CAR
VS_OUTPUT_PRELIT vs_prelit_car( in VS_INPUT_PRELIT In ) // for prelit matt vertices in car space, e.g. lights / dashboards / mirrors
{
    VS_OUTPUT_PRELIT Out;
	Out.oPos = mul(In.v0, lightinfo_mat);
	Out.oFog = Out.oPos.z * fog_info.x + fog_info.y;
	Out.VCol = In.LitCol;
	Out.oT0 = In.v8;
    return Out;
}

// VSHADER_CAR_SOLID_SHINY
// VSHADER_CAR_SOLID_MATT
// VSHADER_CAR_ALPHA_SHINY
// VSHADER_CAR_ALPHA_MATT
VS_OUTPUT vs_main( in VS_INPUT In )
{
    VS_OUTPUT Out;
	Out.oPos = mul(In.v0, lightinfo_mat);
	Out.oFog = Out.oPos.z * fog_info.x + fog_info.y;
	Out.oT0 = In.v8;
	Out.oT1 = clamp(mul(In.v0, local_to_texel), clampmin, clampmax).xy; // lightmap texture

	Out.VCol = In.Col; // vertex colour
	Out.VCol.a = 0.0; // vertex colour alpha is used for base color spec highlight, so by default we force it to 0

	Out.Normal = In.Normal;
	Out.LightDir = light_dir.xyz;
	
	Out.DirectL = max(0.0, dot(In.Normal, light_dir.xyz));
	
	// ambient lighting
	float up_dot_n = dot(In.Normal, up_dir.xyz);
	Out.AmbientL = basecolamb.rgb + up_dot_n * diffcolamb.rgb;
	
#ifdef SHINY
	// for environment map, send the eye to vertex vector and the vertex normal, in environment map space
	Out.EyeToVEnv = mul(float4(In.v0.xyz - eye_local.xyz, 1.0f), local_to_env).xyz;
	Out.NormalEnv = mul(float4(In.Normal, 1.0f), local_to_env).xyz;

// If we aren't rendering glass, add a fake base coat specular highlight (blinn phong normalized)
#ifndef ALPHA
#if USE_FAKE_TWO_COATS
	float3 v = normalize(Out.EyeToVEnv);
	float3 l = -normalize(mul(float4(light_dir.xyz, 1.0f), local_to_env).xyz);
	float3 n = normalize(Out.NormalEnv.xyz);
	float3 h = normalize(v + l);
	
	float NoH = clamp(dot(-n, h), 0.0, 1.0);
	
	// reduce diffuse lighting when lit at glancing angles because more of the sun's light is reflected
	float baseCoatRoughness = BASE_COAT_ROUGHNESS;
	float specPower = RoughnessToSpecPower(baseCoatRoughness);

	float alpha = baseCoatRoughness * baseCoatRoughness;
	float specNormalization = (alpha + 2.0) / (8.0 * PI);
	
	Out.VCol.a = pow(NoH, specPower) * specNormalization;	
#endif // USE_FAKE_TWO_COATS
#endif // !ALPHA

#endif // SHINY
    return Out;
}
