#include <string>
#include <iostream>

#include <cstdlib>

#include <cinsim.h>

#include "MKeyFile.hpp"

#include <string>
#include <algorithm>
#include <sstream>

namespace StringUtils {
    inline std::string
    trim (const std::string &pString,
          const std::string &pWhitespace = " \t") {
        const size_t beginStr = pString.find_first_not_of(pWhitespace);

        if (beginStr == std::string::npos) {
            // no content
            return "";
        }

        const size_t endStr = pString.find_last_not_of(pWhitespace);
        const size_t range = endStr - beginStr + 1;
        return pString.substr(beginStr, range);
    }

    inline std::string
    reduce (const std::string &pString,
            const std::string &pFill = " ",
            const std::string &pWhitespace = " \t") {
        // trim first
        std::string result(trim(pString, pWhitespace));
        // replace sub ranges
        size_t beginSpace = result.find_first_of(pWhitespace);

        while (beginSpace != std::string::npos) {
            const size_t endSpace =
                result.find_first_not_of(pWhitespace, beginSpace);
            const size_t range = endSpace - beginSpace;
            result.replace(beginSpace, range, pFill);
            const size_t newStart = beginSpace + pFill.length();
            beginSpace = result.find_first_of(pWhitespace, newStart);
        }

        return result;
    }


    inline std::string
    capitalized (const std::string &str) {
        std::string ret_str = str;

        if (ret_str.length() > 0) {
            ret_str[0] = toupper(ret_str[0]);
        }
        return ret_str;
    }

    inline void
    capitalize (std::string &str) {
        str[0] = toupper(str[0]);
    }


    template <typename T>
    inline T
    fromString (const std::string &str) {
        std::stringstream ss_val(str);
        T ret_val;
        ss_val >> ret_val;
        return ret_val;
    }

    template <>
    inline std::string
    fromString (const std::string &str) {
        return str;
    }

    template <typename T>
    inline std::string
    toString (const T &val) {
        std::stringstream ss_val;
        ss_val << val;
        return ss_val.str();
    }
}



using std::string;
using std::cerr;
using std::endl;

///
#ifdef WIN32
#include <windows.h>
#endif
#ifdef __unix__
#include <unistd.h>
#endif


inline void sleep_millisecs (const unsigned millisecs) {
#ifdef WIN32
    ::Sleep(millisecs);
#endif
#ifdef __unix__
    usleep(millisecs * 1000);
#endif
}

///

#include "boost/date_time/posix_time/posix_time.hpp"
using namespace boost;
struct AppConfig {
    string server;
    int port;
    string pass;

    int pos_h;
    int pos_v;
    int width;
    int height;

    int l_pos_h;
    int l_pos_v;
    int l_width;
    int l_height;


    string timer_string;

    posix_time::ptime event_time;

    int blink_shown_time;
    int blink_hidden_time;

    int clickid;
};


typedef std::map<string, string> SettingsMap;

template <typename T>
T
get_config_value (SettingsMap& settings,
                  const string& option_name,
                  const T& default_value) {
    if (settings.count(option_name)) {
        return StringUtils::fromString<T>(settings[option_name]);
    }
    return default_value;
}

template <typename T>
T
get_required_config_value (SettingsMap& settings,
                           const string& option_name) {
    if (settings.count(option_name)) {
        return StringUtils::fromString<T>(settings[option_name]);
    }
    else {
        cerr << "\'" << option_name << "\' " << "was not specified" << endl;
        exit(-1);
    }
}


const AppConfig
load_config () {
    AppConfig c;
    MKeyFile f("config.ini");
    SettingsMap settings;

    MSectionsIterator sec_iter = f.getSectionsIterator();
    while (sec_iter.isElement()) {
        MSettingsIterator set_iter = sec_iter.getSettingsIterator();
        const std::string sec_name = sec_iter.getName();
        const std::string prefix =
            sec_name.empty() ? "" : sec_name + ".";
        while (set_iter.isElement()) {
            settings[prefix + set_iter.getName()] =
                set_iter.getValue();
            set_iter.peekNext();
        }
        sec_iter.peekNext();
    }

    c.server = get_required_config_value<string>(settings, "connection.server");
    c.port = get_required_config_value<int>(settings, "connection.port");
    c.pass = get_config_value<string>(settings, "connection.pass", "");

    c.pos_h = get_config_value(settings, "appearance.pos_hor", 60);
    c.pos_v = get_config_value(settings, "appearance.pos_vert", 20);
    c.width = get_config_value(settings, "appearance.width", 80);
    c.height = get_config_value(settings, "appearance.height", 80);

    c.l_pos_h = get_config_value(settings, "appearance.lobby_pos_hor", c.pos_h);
    c.l_pos_v = get_config_value(settings, "appearance.lobby_pos_vert", c.pos_v);
    c.l_width = get_config_value(settings, "appearance.lobby_width", c.width);
    c.l_height = get_config_value(settings, "appearance.lobby_height", c.height);


    c.blink_shown_time = get_config_value(settings, "timer.blink_shown_time", 1000);
    c.blink_hidden_time = get_config_value(settings, "timer.blink_hidden_time", 400);

    c.clickid = get_config_value(settings, "insim.clickid", 239);

    c.timer_string = get_config_value<string>(settings, "timer.timer_string", "");

    string t = get_required_config_value<string>(settings, "timer.event_time");
    if (settings.count("timer.event_date")) {
        string s = settings["timer.event_date"] + " " + t;
        c.event_time = posix_time::time_from_string(s);
    }
    else {
        posix_time::ptime n = posix_time::microsec_clock::local_time();
        string d = gregorian::to_iso_extended_string(n.date());

        c.event_time = posix_time::time_from_string(d + " " + t);
    }

    return c;
}


template <typename T>
inline T
abs (T value) {
    return value < 0 ? -value : value;
}


#include "date_time/time_facet.hpp"
string
pretty_print_period (posix_time::time_period p) {
    string s = "";
    posix_time::time_duration d = p.length();
    if (d.seconds() < 0) {
        s += "-";
    }
    if (d.hours() != 0) {
        s += StringUtils::toString(abs(d.hours())) + ":";
    }

    if (d.hours() != 0 && abs(d.minutes()) < 10) {
        s += "0";
    }
    s += StringUtils::toString(abs(d.minutes())) + ":";
    if (d.seconds() < 10) {
        s += "0";
    }

    s += StringUtils::toString(abs(d.seconds()));
    return s;
}


class SimpleStopwatch {
public:
    SimpleStopwatch ()
        : start (posix_time::microsec_clock::local_time()){
    }

    posix_time::time_duration
    elapsed () {
        return posix_time::time_period(
            this->start,
            posix_time::microsec_clock::local_time()).length();
    }

    unsigned
    elapsed_millisecs () {
        return this->elapsed().total_milliseconds();
    }

    void
    reset () {
        this->start = posix_time::microsec_clock::local_time();
    }



private:
    posix_time::ptime start;
};


struct QuitpacketGrabberArgs {
    bool *quit;
    posix_time::ptime *event_time;
    CInsim *insim;
    bool *server_in_lobby;
};


#include <bitset>
void*
quitpacket_grabber (void *args) {
    QuitpacketGrabberArgs* a = (QuitpacketGrabberArgs*) args;

    IS_TINY sr;
    sr.Size = 4;
    sr.Type = ISP_TINY;
    sr.ReqI = 10;
    sr.SubT = TINY_SST;
    a->insim->send_packet(&sr);

    bool vote_in_action = false;
    bool run = true;
    while (run) {
        int p = a->insim->next_packet();
        char pt = a->insim->peek_packet();
        void* pp = a->insim->get_packet();
        char tptype = -1;
        if (pt == ISP_TINY) {
            tptype = ((IS_TINY*) a->insim->get_packet())->SubT;
        }
        if (tptype == TINY_REN) {
            // show timer again
            *a->server_in_lobby = true;
        }
        if (pt == ISP_STA) {
            IS_STA* packet = (IS_STA*) pp;
            *a->server_in_lobby = !packet->RaceInProg;
        }

        if (pt == ISP_RST || tptype == TINY_MPE || tptype == TINY_REN) {
            posix_time::ptime now = posix_time::microsec_clock::local_time();
            if (now > *a->event_time) {
                *a->quit = true;
                return NULL;
            }
        }
        sleep_millisecs(100);
    }
    return NULL;
}


int
main () {
    AppConfig cfg = load_config();

    CInsim ins;
    bool conn_fail = ins.init(cfg.server.c_str(),
                              cfg.port,
                              "Timer",
                              cfg.pass.c_str());

    if (conn_fail) {
        cerr << "ERROR: can't connect" << endl;
        exit(-1);
    }

    bool quit = false;

    bool server_in_lobby = false;

    struct QuitpacketGrabberArgs args;
    args.quit = &quit;
    args.event_time = &cfg.event_time;
    args.insim = &ins;
    args.server_in_lobby = &server_in_lobby;

    pthread_t thread;
    int th = pthread_create (&thread, NULL, quitpacket_grabber, (void*) &args);
    if (th) {
        cerr << "ERROR: can't create thread for receiving packets" << endl;
        exit(-1);
    }


    struct IS_BTN btnd;
    btnd.Size = 12 + 240;
    btnd.Type = ISP_BTN;
    btnd.ReqI = 1;
    btnd.UCID = 255;

    btnd.ClickID = cfg.clickid;
    btnd.Inst = 0;
    btnd.BStyle = 0 | 3;
    btnd.TypeIn = 0;

    btnd.L = cfg.pos_h;
    btnd.T = cfg.pos_v;
    btnd.W = cfg.width;
    btnd.H = cfg.height;

    struct IS_BTN btnl = btnd;
    btnl.L = cfg.l_pos_h;
    btnl.T = cfg.l_pos_v;
    btnl.W = cfg.l_width;
    btnl.H = cfg.l_height;

    struct IS_BTN bprh = btnd;
    bprh.L = 0;
    bprh.T = 0;
    bprh.W = 1;
    bprh.H = 1;
    memcpy(bprh.Text, " ", 2);

    struct IS_BFN bclr;
    bclr.Size = 8;
    bclr.Type = ISP_BFN;
    bclr.ReqI = 0;
    bclr.SubT = BFN_DEL_BTN;

    bclr.UCID = 255;
    bclr.ClickID = cfg.clickid; // TODO: make configurable in case this
                                // ID is used by some other insim

    SimpleStopwatch blink_timer = SimpleStopwatch();
    bool blink_state = false; // false is hidden
    posix_time::ptime last_update = posix_time::second_clock::local_time();

    bool moving_to_lobby = false;
    bool prev_server_in_lobby = server_in_lobby;
    while (!quit) {
        struct IS_BTN *btn = server_in_lobby ? &btnl : &btnd;

        posix_time::ptime now = posix_time::microsec_clock::local_time();
        posix_time::ptime now_second_precision =
            posix_time::second_clock::local_time();
        if (now >= cfg.event_time) {
            int blink_time_to_wait = blink_state
                ? cfg.blink_shown_time
                : cfg.blink_hidden_time;
            if (blink_timer.elapsed_millisecs() > blink_time_to_wait) {
                blink_timer.reset();
                if (blink_state) {
                    ins.send_packet(&bclr);
                }
                else {
                    string s = cfg.timer_string + " 0:00";
                    memcpy(btn->Text, s.c_str(), s.length() + 1);
                    ins.send_packet(btn);
                }

                blink_state = !blink_state;
            }
            // TODO: maybe put 'else' here and sleep the time left
            // before next blink?
        }
        else if ((now_second_precision - last_update).total_seconds() >= 1) {
            posix_time::time_period p (posix_time::time_period(now, cfg.event_time));

            string s = cfg.timer_string + " " + pretty_print_period(p);
            memcpy(btn->Text, s.c_str(), s.length() + 1);

            ins.send_packet(btn);
        }
        sleep_millisecs(30); // TODO: balance sleeping to keep packet
                             // sendings closer to system clock?
    }
    ins.isclose();
}
