# provides parsing raw insim packets

from struct import unpack
STRIP = '\x00\x20'

OUTGAUGE_ATTRIBUTE_NAMES = ['turbo',
                            'shiftlight',
                            'gear',
                            'display2',
                            'display1',
                            'rpm',
                            'headlights',
                            'pitspeed',
                            'speed',
                            'throttle',
                            'clutch',
                            'signal_r',
                            'handbrake',
                            'signal_l',
                            'car',
                            'fuel',
                            'brake',
                            'tc']

def _pedalvalue(val, res):
    return int(val * 100 / res) * res

def _outGaugeFlagBits(inp):
    inp = inp - 192
    mask = [(1, 'shiftlight'),    #OK
            (2, False),
            (4, False),
            (8, False),
            (16, False),
            (32, False),
            (64, False),
            (128, False),
            (256, False),
            (512, 'headlights'), #OK
            (1024, 'handbrake'), #OK
            (2048, 'pitspeed'),  #OK
            (4096, 'tc'), #OK
            (8192, False),
            (16384, 'signal_l'), #OK
            (32768, 'signal_r'),] #OK
    mask.reverse()
    ret = {}
    for bit, name in mask:
        if not name:
            continue
        active = not not inp // bit
        ret[name] = active
        if active:
            inp = inp - bit
    return ret



class OutGaugePacket:
    """an outgauge packet, pass raw data to constructor"""
    def __init__(self, data, rpm_res=10, pedals_res=1, speed_res=1, 
                 turbo_res=0.1):
        """convert the packet to attributes"""
        format = '<I4s2sccffffffffffff16x16x'
        format = '<I4s2sccffffffffffff16s16s'
        descriptors = ('time', 'car', 'flags', 'gear', '_spare', 'speed', 'rpm',
                       'turbo', 'engtemp', 'fuel', 'oilpres', 'spare1',
                       'spare2',' spare3', 'throttle', 'brake', 'clutch',
                       'display1', 'display2')
        data = unpack(format, data)
        data = dict(zip(descriptors, data))

        #convert speed
        data['speed'] = int(data['speed'] * 3.6 / speed_res) * speed_res

        #convert gear
        gear = int(data['gear'].encode('hex'))
        if gear == 0: gear = 'R'
        elif gear == 1: gear = 'N'
        else: gear -= 1
        data['gear'] = str(gear)

        #convert rpm
        data['rpm'] = int(data['rpm']/rpm_res) * rpm_res
        #convert throttle, brake, clutch
        data['throttle'] = _pedalvalue(data['throttle'], pedals_res)
        data['brake'] = _pedalvalue(data['brake'], pedals_res)
        data['clutch'] = _pedalvalue(data['clutch'], pedals_res)

        #convert fuel
        data['fuel'] = float('%0.2f' % (data['fuel'] * 100))

        #convert turbo pressure
        data['turbo'] = int(data['turbo']/turbo_res)*turbo_res
        #car name
        data['car'] = data['car'].strip(STRIP)

        #renice display fields
        data['display1'] = data['display1'].strip(STRIP)
        data['display2'] = data['display2'].strip(STRIP)
        
        #parse flags
        flags = _outGaugeFlagBits(int(data['flags'].encode('hex'),16))
        data.update(flags)


        self.__dict__.update(data)