// LFS Vertex Shader : World

 
/*/
 *	Crytek's Volumetric Fog implementation by DarkDrifter Nacim
/*/

// CONFIGURATION
#define USE_CUSTOM_FOG // Comment this to restore LFS's default fog
#define DENSITY_MULTIPLIER 1.0 // Fog density multipler (< 1 = less dense, > 1 = more dense)
#define USE_FORWARD_SCATTERING // Comment this to disable the forward scattering effect (where the fog is more dense in the sun's direction)
#define FORWARD_SCATTERING_POWER 16.0 // Forward scattering power multiplier, lower this for a more diffuse effect (but also less pronounced)


// Vertex shader input structures

struct VS_INPUT // VERT_PDST1
{
	float4	v0		: POSITION;
	float4	LitCol	: COLOR0;
	float4	AmbCol	: COLOR1;
	float2	v8		: TEXCOORD0;
};

struct VS_INPUT2 // VERT_PNDT1
{
	float4	v0		: POSITION;
	float3	Normal	: NORMAL;
	float4	LitCol	: COLOR0;
	float4	AmbCol	: COLOR1;
	float2	v8		: TEXCOORD0;
};


// Vertex shader output structures

struct VS_OUTPUT
{
	float4	oPos		: POSITION;
	float	oFog		: FOG;
	float4	VCol		: COLOR0;
	float2	oT0			: TEXCOORD0;
};

struct VS_OUTPUT_ENV
{
    float4	oPos		: POSITION;
	float	oFog		: FOG;
	float4	VCol		: COLOR0;
	float3	NormalEnv	: TEXCOORD0;
	float3	EyeToVEnv	: TEXCOORD1;
};

struct VS_OUTPUT_PAINT
{
    float4	oPos		: POSITION;
	float	oFog		: FOG;
	float4	VCol		: COLOR0;
	float2	oT0			: TEXCOORD0;
	float3	NormalEnv	: TEXCOORD1;
	float3	EyeToVEnv	: TEXCOORD2;
};


// Global variables

float4x4	lightinfo_mat	: register(c0); // for vertices in world space
float4x4	proj_mat		: register(c4); // for vertices in view space

float4		light_dir		: register(c8);

/*
float4		light_dir		: register(c8);
float4		up_dir			: register(c9);
float4		basecolamb		: register(c10);
float4		diffcolamb		: register(c11);
*/

// For envmap
float4x4	local_to_env	: register(c12);
float4		eye_local		: register(c16);

float4		fog_info		: register(c90);


float4 GammaToLinear(float4 col)
{
	return float4(pow(abs(col.rgb), 2.2), col.a);
}

float4 LinearToGamma(float4 col)
{
	return float4(pow(abs(col.rgb), 1/2.2), col.a);
}

float ComputeVolumetricFog(in float3 cameraToWorldPos, in float cameraHeight, in float density = 0.0003f)
{
	float cHeightFalloff = 0.001f;
	float cGlobalDensity = density;
	float cVolFogHeightDensityAtViewer = exp(-cHeightFalloff * cameraHeight);
	float fogInt = length(cameraToWorldPos) * cVolFogHeightDensityAtViewer;
	const float cSlopeThreshold = 0.01;
	if (abs(cameraToWorldPos.z) > cSlopeThreshold)
	{
		float t = cHeightFalloff * cameraToWorldPos.z;
		fogInt *= (1.0 - exp(-t)) / t;
	}
	return exp(-cGlobalDensity * fogInt);
}


float ComputeWorldSpaceFog(float3 vertPos) 
{
#ifdef USE_CUSTOM_FOG
	// Crytek Volumetric Fog
	float3 viewtoVertex = mul(float4(vertPos - eye_local.xyz, 1.0f), local_to_env).xyz;

#ifdef USE_FORWARD_SCATTERING
	float3 V = normalize(viewtoVertex);
	float3 L = normalize(light_dir.xyz);

	float VdotL = max(dot(V, L), 0.0f);
	float intensity = max(pow(VdotL, FORWARD_SCATTERING_POWER), 0.1f);
#else
	float intensity = 0.1f;
#endif

	return ComputeVolumetricFog(viewtoVertex, eye_local.z, (fog_info.y / 200) * intensity * DENSITY_MULTIPLIER);
#else
	// Base LFS Fog
	float4 pos = mul(float4(vertPos.xyz, 1.0f), lightinfo_mat);
	return pos.z * fog_info.x + fog_info.y; // Base LFS fog
#endif
}


// VSHADER_PRELIT_VIEW
VS_OUTPUT vs_prelit_view( in VS_INPUT In ) // for prelit vertices in view space
{
    VS_OUTPUT Out;
	Out.oPos = mul(In.v0, proj_mat);
	Out.oFog = Out.oPos.z * fog_info.x + fog_info.y;
	Out.VCol = In.LitCol;
	Out.oT0 = In.v8;
    return Out;
}


// VSHADER_PRELIT_WORLD
VS_OUTPUT vs_prelit_world( in VS_INPUT In ) // for prelit matt vertices in world space
{
    VS_OUTPUT Out;
	Out.oPos = mul(In.v0, lightinfo_mat);
	Out.oFog = ComputeWorldSpaceFog(In.v0.xyz);
	Out.VCol = In.LitCol;
	Out.oT0 = In.v8;
    return Out;
}

// VSHADER_LIGHT_ONLY
VS_OUTPUT vs_light_only( in VS_INPUT In ) // for the first pass on a shadowed surface - shiny or matt - direct light only
{
    VS_OUTPUT Out;
	Out.oPos = mul(In.v0, lightinfo_mat);
	Out.oFog = ComputeWorldSpaceFog(In.v0.xyz);
	Out.VCol = In.LitCol - In.AmbCol;
	Out.VCol.a = 1.0f; // keep the alpha value or ATEST surfaces disappear when their object is in shadow
	Out.oT0 = In.v8; // diffuse texture coordinates
    return Out;
}


// VSHADER_MATT_ADD_AMBIENT
VS_OUTPUT vs_matt_add_ambient( in VS_INPUT In ) // the final pass on a matt shadowed surface - adding the ambient light
{
    VS_OUTPUT Out;
	Out.oPos = mul(In.v0, lightinfo_mat);
	Out.oFog = ComputeWorldSpaceFog(In.v0.xyz);
	Out.VCol = In.AmbCol;
	Out.oT0 = In.v8;
    return Out;
}


// VSHADER_SKY_ENVMAP
VS_OUTPUT_ENV vs_sky_envmap( in VS_INPUT2 In ) // for untextured reflections on glass or metal
{
    VS_OUTPUT_ENV Out;
	Out.oPos = mul(In.v0, lightinfo_mat);
	Out.oFog = ComputeWorldSpaceFog(In.v0.xyz);
	Out.VCol = In.LitCol;

	// for environment map, send the eye to vertex vector and the vertex normal, in environment map space
	Out.EyeToVEnv = mul(float4(In.v0.xyz - eye_local.xyz, 1.0f), local_to_env).xyz;
	Out.NormalEnv = mul(float4(In.Normal, 1.0f), local_to_env).xyz;
    return Out;
}


// VSHADER_WORLD_SHINY
VS_OUTPUT_PAINT vs_world_shiny( in VS_INPUT2 In ) // for prelit shiny vertices in world space - single pass
{
    VS_OUTPUT_PAINT Out;
	Out.oPos = mul(In.v0, lightinfo_mat);
	Out.oFog = ComputeWorldSpaceFog(In.v0.xyz);
	Out.VCol = In.LitCol;
	Out.oT0 = In.v8;

	// for environment map, send the eye to vertex vector and the vertex normal, in environment map space
	Out.EyeToVEnv = mul(float4(In.v0.xyz - eye_local.xyz, 1.0f), local_to_env).xyz;
	Out.NormalEnv = mul(float4(In.Normal, 1.0f), local_to_env).xyz;
    return Out;
}


// VSHADER_SHINY_ADD_AMBIENT
VS_OUTPUT_PAINT vs_shiny_add_ambient( in VS_INPUT2 In ) // the final pass on a shiny shadowed surface - adding the ambient light
{
    VS_OUTPUT_PAINT Out;
	Out.oPos = mul(In.v0, lightinfo_mat);
	Out.oFog = ComputeWorldSpaceFog(In.v0.xyz);
	Out.VCol = In.AmbCol;
	Out.oT0 = In.v8;

	// for environment map, send the eye to vertex vector and the vertex normal, in environment map space
	Out.EyeToVEnv = mul(float4(In.v0.xyz - eye_local.xyz, 1.0f), local_to_env).xyz;
	Out.NormalEnv = mul(float4(In.Normal, 1.0f), local_to_env).xyz;
    return Out;
}