#include "stdafx.h"
#include "LFSInterface.h"
#include "WinMutex.h"
#include <tchar.h>

static LPCSTR szAppName = "G27 LEDS";
static LPCWSTR szInitError = L"Initialization error";
static LPCWSTR szNetworkError = L"Network error.";

static HANDLE hISThr = 0;
static HANDLE hMSGThr = 0;
static HANDLE hOGThr = 0;
static HANDLE hSHM;
static HANDLE hSHMMutex;
static HANDLE hMsgSHM;
static HANDLE hMsgEvent;
static HANDLE hMsgMutex;

static SOCKET isSock = 0;
static DWORD isThrID;
static SOCKET ogSock = 0;
static DWORD ogThrID;
static DWORD msgThrID;
static volatile BOOL receiveInSim;
static volatile BOOL receiveMessages;
static volatile BOOL receiveOutGauge;
static WSADATA wsaData;
static BOOL wsaInited = FALSE;

static const INT OGPACK_SIZE = 92;
static const INT OGPACK_ID_SIZE = 96;

#pragma pack(1)
static struct {
	float rpm;
	float firstLeds;
	float redline;
	UINT debug;
} sharedData;
LPVOID pSharedData;
#pragma pack(1)
typedef struct {
	WCHAR msgTxt[128];
	BOOL fPending;
	BOOL fClosed;
} SharedMsg;
SharedMsg sharedMsg;
LPVOID pSharedMsg;

BOOL InitDLLIFace()
{
	/* Create shared memory for OutGauge data */
	ZeroMemory(&sharedData, sizeof(sharedData));
	hSHMMutex = CreateMutexA(NULL, FALSE, "G27LEDS_SHMMUTEX");
	if (!hSHMMutex) {
		MessageBoxW(NULL, L"Unable to create SHM mutex.", szInitError, MB_OK | MB_ICONERROR);
		return FALSE;
	}

	hSHM = CreateFileMappingA(INVALID_HANDLE_VALUE, NULL, PAGE_READWRITE, 0, sizeof(sharedData), "G27LEDS_SHM");
	if (!hSHM) {
		MessageBoxW(NULL, L"Unable to create file mapping.", szInitError, MB_OK | MB_ICONERROR);
		return FALSE;
	}

	pSharedData = MapViewOfFile(hSHM, FILE_MAP_ALL_ACCESS, 0, 0, sizeof(sharedData));
	if (!pSharedData) {
		MessageBoxW(NULL, L"Unable to map shared data.", szInitError, MB_OK | MB_ICONERROR);
		return FALSE;
	}

	/* Create shared memory for DLL messages */
	ZeroMemory(&sharedMsg, sizeof(sharedMsg));
	hMsgMutex = CreateMutexA(NULL, FALSE, "G27LEDS_SHMMSGMUTEX");
	if (!hMsgMutex) {
		MessageBoxW(NULL, L"Unable to create Msg mutex.", szInitError, MB_OK | MB_ICONERROR);
		return FALSE;
	}
	hMsgEvent = CreateEventA(NULL, TRUE, FALSE, "G27LEDS_MSGEVENT");
	if (!hMsgEvent) {
		MessageBoxW(NULL, L"Unable to create Msg event.", szInitError, MB_OK | MB_ICONERROR);
		return FALSE;
	}
	hMsgSHM = CreateFileMappingA(INVALID_HANDLE_VALUE, NULL, PAGE_READWRITE, 0, sizeof(sharedMsg), "G27LEDS_MSGSHM");
	if (!hMsgSHM) {
		MessageBoxW(NULL, L"Unable to create Msg SHM.", szInitError, MB_OK | MB_ICONERROR);
		return FALSE;
	}
	pSharedMsg = MapViewOfFile(hMsgSHM, FILE_MAP_ALL_ACCESS, 0, 0, sizeof(sharedMsg));
	if (!pSharedMsg) {
		MessageBoxW(NULL, L"Unable to map view of Msg SHM.", szInitError, MB_OK | MB_ICONERROR);
		return FALSE;
	}

	{
		WinMutex lk(hMsgMutex, INFINITE);
		if (lk.status == WAIT_OBJECT_0)
			CopyMemory(pSharedMsg, &sharedMsg, sizeof(SharedMsg));
		else {
			MessageBoxW(NULL, L"Unable to write default so MSg SHM.", szInitError, MB_OK | MB_ICONERROR);
			return FALSE;
		}
	}

	sharedData.debug = debugLevel;
	{
		WinMutex lk(hSHMMutex, INFINITE);
		if (lk.status == WAIT_OBJECT_0)
			CopyMemory(pSharedData, &sharedData, sizeof(sharedData));
		else {
			MessageBoxW(NULL, L"Unable to write defaults to OutGauge SHM.", szInitError, MB_OK | MB_ICONERROR);
			return FALSE;
		}
	}

	return TRUE;
}

BOOL InitMsgIFace()
{
	hMSGThr = CreateThread(NULL, 0, &MessageThrFunc, NULL, 0, &msgThrID);
	if (hMSGThr == NULL) {
		MessageBoxW(NULL, L"Unable to start messaging thread.", szInitError, MB_OK | MB_ICONERROR);
		return FALSE;
	}

	return TRUE;
}


BOOL InitLFSIFace(word isPort, word ogPort, LPCWSTR adminPass)
{
	/* Close handles to any previously run threads */
	if (hISThr) {
		receiveInSim = FALSE;
		WaitForSingleObject(hISThr, INFINITE);
		CloseHandle(hISThr);
	}
	if (hOGThr) {
		receiveOutGauge = FALSE;
		WaitForSingleObject(hOGThr, INFINITE);
		CloseHandle(hOGThr);
	}

	if (wsaInited == FALSE) {
		if (WSAStartup(MAKEWORD(2, 2), &wsaData) == SOCKET_ERROR) {
			LogMessage(L"(ERR)InitLFSIFace> Unable to initialize WinSock.", MSG_SEVERITY_INFO);
			return FALSE;
		}
		wsaInited = TRUE;
	}

	/* Init and start OutGauge */
	if (!InitOutGauge(ogPort))
		return FALSE;
	hOGThr = CreateThread(NULL, 0, &OutGaugeThrFunc, NULL, 0, &ogThrID);
	if (hOGThr == NULL) {
		LogMessage(L"(ERR)InitLFSIFace> Unable to start OutGauge thread.", MSG_SEVERITY_INFO);
		return FALSE;
	}

	if (!InitInSim(isPort)) {
		receiveOutGauge = FALSE;
		WaitForSingleObject(hOGThr, INFINITE);
		return FALSE;
	}
	hISThr = CreateThread(NULL, 0, &InSimThrFunc, (LPVOID)adminPass, 0, &isThrID);
	if (hISThr == NULL) {
		LogMessage(L"(ERR)InitLFSIFace> Unable to start InSim thread.", MSG_SEVERITY_INFO);
		receiveOutGauge = FALSE;
		WaitForSingleObject(hOGThr, INFINITE);
		return FALSE;
	}

	return TRUE;
}

VOID ShutdownLFSIFace()
{
	if (receiveInSim == TRUE) {
		receiveInSim = FALSE;
		WaitForSingleObject(hISThr, INFINITE);
	}
	if (receiveOutGauge == TRUE) {
		receiveOutGauge = FALSE;
		WaitForSingleObject(hOGThr, INFINITE);
	}
	if (receiveMessages == TRUE) {
		receiveMessages = FALSE;
		WaitForSingleObject(hMSGThr, INFINITE);
	}
	CloseHandle(hISThr); 
	CloseHandle(hOGThr);
	CloseHandle(hMSGThr);

	WSACleanup();

	LogMessage(L"LFS interface shut down.", MSG_SEVERITY_INFO);
}

static BOOL InitInSim(word isPort)
{
	isSock = socket(AF_INET, SOCK_STREAM, 0);
	if (isSock == SOCKET_ERROR) {
		LogMessage(L"(ERR)InitInSim> Unable to create InSim socket.", MSG_SEVERITY_INFO);
		return FALSE;
	}

	struct sockaddr_in saddr;
	saddr.sin_family = AF_INET;
	saddr.sin_port = htons(isPort);

	saddr.sin_addr.s_addr = inet_addr("127.0.0.1");

	int attempts = 5;
	while (attempts > 0) {
		if (connect(isSock, (const struct sockaddr*)&saddr, sizeof(sockaddr_in)) >= 0)
			break;
		else {
			Sleep(2500);
			attempts--;
		}
	}
	if (attempts == 0) {
		LogMessage(L"(ERR)InitInSim> Unable to connect to InSim.", MSG_SEVERITY_INFO);
		return FALSE;
	}

	LogMessage(L"InSim socket opened.", MSG_SEVERITY_INFO);

	return TRUE;
}

static DWORD WINAPI InSimThrFunc(LPVOID lpArg)
{
	if (lpArg == NULL) {
		MessageBoxW(NULL, L"(ERR)ISThr> Null pointer to admin password!", NULL, MB_OK);
		goto out;
	}

	struct IS_ISI isIsi;
	ZeroMemory(&isIsi, sizeof(struct IS_ISI));
	isIsi.Size = sizeof(struct IS_ISI);
	isIsi.Type = ISP_ISI;
	isIsi.ReqI = 1;	/* We want IS_VER */
	isIsi.Flags = ISF_LOCAL;
	strcpy_s(isIsi.IName, 15, szAppName);
	wcstombs_s(NULL, isIsi.Admin, (LPCWSTR)lpArg, 16);

	int ret = send(isSock, (const char*)&isIsi, sizeof(struct IS_ISI), 0);
	if (ret < sizeof(struct IS_ISI)) {
		LogMessage(L"(ERR)ISThr> Network error while sending IS_ISI packet.", MSG_SEVERITY_INFO);
		goto out;
	}

	/* Wait for IS_VER */
	struct IS_VER isVer;
	ZeroMemory(&isVer, sizeof(struct IS_VER));
	int recvd = 0;
	recvd = recv(isSock, (char*)&isVer + recvd, sizeof(struct IS_VER) - recvd, 0);
	if (recvd == 0) {
		LogMessage(L"(ERR)ISThr> LFS closed connection while waiting for IS_VER.", MSG_SEVERITY_INFO);
		goto out;
	}
	else if (recvd == SOCKET_ERROR) {
		LogSocketError(WSAGetLastError(), _T("(ERR)ISThr> Error while waiting for IS_VER - "));
		goto out;
	}

	/* Read until we have a full packet */
	while (recvd < sizeof(struct IS_VER)) {
		ret = recv(isSock, (char*)&isVer + recvd, sizeof(struct IS_VER) - recvd, 0);
		if (ret == 0) {
			LogMessage(L"(ERR)ISThr> LFS closed connection while reading IS_VER.", MSG_SEVERITY_INFO);
			goto out;
		}
		else if (ret == SOCKET_ERROR) {
			LogSocketError(WSAGetLastError(), _T("(ERR)ISThr> Error while reading IS_VER - "));
			goto out;
		}

		recvd += ret;
	}

	WCHAR lfsVer[8];
	WCHAR lfsVerMsg[32];
	WCHAR insimVerMsg[32];
	mbstowcs_s(NULL, lfsVer, isVer.Version, 7);
	wsprintfW(lfsVerMsg, L"LFS Version: %s", lfsVer );
	wsprintfW(insimVerMsg, L"InSim version: %d", isVer.InSimVer);
	LogMessage(lfsVerMsg, MSG_SEVERITY_INFO);
	LogMessage(insimVerMsg, MSG_SEVERITY_INFO);

	/* Receiving loop */
	timeval timeout;
	timeout.tv_sec = 1;
	timeout.tv_usec = 0;
	recvd = 0;
	fd_set readfd;
	fd_set exceptfd;

	receiveInSim = TRUE;
	while (receiveInSim == TRUE) {
		char packBuffer[512];
		ZeroMemory(packBuffer, 512);

		FD_ZERO(&readfd);
		FD_ZERO(&exceptfd);
		FD_SET(isSock, &readfd);
		FD_SET(isSock, &exceptfd);

		int sel = select(0, &readfd, NULL, &exceptfd, &timeout);
		if (sel < 0) {
			LogMessage(L"(ERR)IsThr> Communication error on InSim socket.", MSG_SEVERITY_INFO);
			goto out;
		} else if (sel == 0)
			continue;
		else {
			if (FD_ISSET(isSock, &exceptfd)) {
				LogMessage(L"(WARN)IsThr> Exception on InSim socket.", MSG_SEVERITY_INFO);
				goto out;
			}

			/* Read until we have a full packet */
			recvd += recv(isSock, packBuffer + recvd, 512 - recvd, 0);

			if (recvd == SOCKET_ERROR) {
				LogSocketError(WSAGetLastError(), _T("(ERR)IsThr> Error reading InSim packet - "));
				goto out;
			} else if (recvd == 0) { /* Connection closed on remote end */
				LogMessage(L"LFS has closed InSim connection.", MSG_SEVERITY_INFO);
				
				/* Either we lost connection to LFS or LFS has quit. Shut the interface down */
				goto out;
			}

			/* We have at least one full packet */
			while (recvd >= (byte)packBuffer[0]) {
				WCHAR dbgLog[64];
				wsprintfW(dbgLog, L"Packet type: %d, Size: %d, Recvd: %d", (byte)packBuffer[1], (byte)packBuffer[0], recvd);
				LogMessage(dbgLog, MSG_SEVERITY_DEBUG);

				ProcessISPacket(packBuffer);
				
				byte firstPackLen = (byte)packBuffer[0];	
				/* Copy the other (possibly incomplete) packet to the beggining of the buffer */
				if (recvd > (byte)packBuffer[0])
					memmove(packBuffer, packBuffer + firstPackLen, recvd - firstPackLen);
				recvd -= firstPackLen;
			}
		}
	}

out:
	closesocket(isSock);

	LogMessage(L"InSim thread finished", MSG_SEVERITY_DEBUG);
	receiveOutGauge = FALSE;
	return 0;
}

static BOOL InitOutGauge(word ogPort)
{
	ogSock = socket(AF_INET, SOCK_DGRAM, 0);

	if (ogSock == SOCKET_ERROR) {
		LogMessage(L"(ERR)InitOutGauge> Unable to create OutGauge socket.", MSG_SEVERITY_INFO);
		return FALSE;
	}

	struct sockaddr_in saddr;
	saddr.sin_family = AF_INET;
	saddr.sin_port = htons(ogPort);
	saddr.sin_addr.s_addr = inet_addr("127.0.0.1");

	if (bind(ogSock, (const struct sockaddr*)&saddr, sizeof(sockaddr_in)) < 0) {
		LogMessage(L"(ERR)InitOutGauge> Unable to bind OutGauge socket.", MSG_SEVERITY_INFO);
		return FALSE;
	}

	LogMessage(L"OutGauge socket opened.", MSG_SEVERITY_INFO);

	return TRUE;
}

static VOID LogSocketError(const DWORD errorId, const LPCTSTR prefix)
{
	LPTSTR errMsg;
	DWORD fmRet = FormatMessage(FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_FROM_SYSTEM, NULL, errorId, LANG_USER_DEFAULT, (LPTSTR)&errMsg, 0, NULL);
	if (fmRet > 0) {
		size_t len = (_tcslen(prefix) + fmRet + 1) * sizeof(TCHAR);
		LPTSTR outbuf = (LPTSTR)malloc(len);
		ZeroMemory(outbuf, len);
		_stprintf_s(outbuf, len - 1, _T("%s%s"), prefix, errMsg);
		LogMessage(outbuf, MSG_SEVERITY_INFO);
		free(outbuf);
		LocalFree(errMsg);
	}
}

static VOID ProcessISPacket(const char* packBuffer)
{
	/* Handle keepalive request */
	if (packBuffer[1] == ISP_TINY && packBuffer[3] == ISP_NONE) {
		struct IS_TINY isTiny;
		ZeroMemory(&isTiny, sizeof(struct IS_TINY));
		isTiny.Size = sizeof(struct IS_TINY);
		isTiny.Type = ISP_TINY;
		isTiny.ReqI = 0;
		isTiny.SubT = ISP_NONE;

		send(isSock, (const char*)&isTiny, sizeof(struct IS_TINY), 0);
		LogMessage(L"Keepalive.", MSG_SEVERITY_DEBUG);
	}

	/* Handle other packet types */
	switch ((byte)packBuffer[1]) {
	case ISP_STA:
		IS_STAHandler((struct IS_STA*)packBuffer);
		break;
	default:
		break;
	}
}

static DWORD WINAPI OutGaugeThrFunc(LPVOID lpArg)
{
	UNREFERENCED_PARAMETER(lpArg);

	fd_set readfd;
	fd_set exceptfd;
	timeval timeout;
	timeout.tv_sec = 1;
	timeout.tv_usec = 0;

	receiveOutGauge = TRUE;

	BOOL pitLimiterLEDsOn = FALSE;
	DWORD pitLimiterLastSwitch = 0;
	DWORD timeStamp = GetTickCount();
	while (receiveOutGauge == TRUE) {
		FD_ZERO(&readfd);
		FD_ZERO(&exceptfd);
		FD_SET(ogSock, &readfd);
		FD_SET(ogSock, &exceptfd);

		int ret = select(0, &readfd, NULL, &exceptfd, &timeout);
		if (FD_ISSET(ogSock, &exceptfd)) {
			LogMessage(L"(ERR)OgThr> Exception on OutGauge socket.", MSG_SEVERITY_INFO);
			break;
		}
		if (ret < 0) {
			LogMessage(L"(ERR)OgThr> Error reading OutGauge socket.", MSG_SEVERITY_INFO);
			break;
		} else if (ret == 0)
			continue; /* Timeout */
		else { /* We got data */
			char packBuffer[OGPACK_ID_SIZE];
			int recvd = recv(ogSock, packBuffer, sizeof(packBuffer), 0);

			if (recvd == SOCKET_ERROR) {
				LogSocketError(WSAGetLastError(), _T("(ERR)OgThr> Error receiving OutGauge packet - "));
				goto out;
			} else if (recvd == 0) {
				LogMessage(L"(WARN)OgThr> OutGauge closed on remote end.", MSG_SEVERITY_INFO); /* Does this happen? */
				break;
			} else if (recvd == OGPACK_SIZE || recvd == OGPACK_ID_SIZE) {
				/* We have what appears to be a valid packet, process it */
				OutGaugePack* ogp = reinterpret_cast<OutGaugePack*>(packBuffer);
				std::string carType(ogp->Car);

				DWORD newTimeStamp = GetTickCount();
				/* Pit limiter indication enabled, it is time to switch the display and pit limiter in the car is on */
				if ((pitLimiterDelay > 0) && (newTimeStamp - pitLimiterLastSwitch >= pitLimiterDelay) && (ogp->ShowLights & 0x8)) {
					if (pitLimiterLEDsOn) {
						sharedData.rpm = 0.0f;
						sharedData.firstLeds = 1.0f;
						sharedData.redline = 2.0f;
					} else {
						sharedData.rpm = 999.0f;
						sharedData.firstLeds = 1.0f;
						sharedData.redline = 1000.0f;	
					}
					pitLimiterLEDsOn = !pitLimiterLEDsOn;
					pitLimiterLastSwitch = newTimeStamp;
				} else { /* Display RPM */
					std::map<std::string, rpmData>::const_iterator cit = rpms.find(carType);
					if (cit == rpms.end()) {
						LogMessage(L"(WARN)OgThr> Invalid car type", MSG_SEVERITY_INFO);
						continue;
					}

					sharedData.rpm = ogp->RPM;
					sharedData.firstLeds = cit->second.firstLeds;
					sharedData.redline = cit->second.redline;
				}

				/* Copy data to shared memory */
				{
					WinMutex lk(hSHMMutex, 0);
					if (lk.status == WAIT_OBJECT_0)
						CopyMemory(pSharedData, &sharedData, sizeof(sharedData));
					else
						LogMessage(L"Lock would block", MSG_SEVERITY_DEBUG);
				}
				timeStamp = newTimeStamp;
			} else {
				LogMessage(L"(WARN)OgThr> Received malformed OutGauge packet", MSG_SEVERITY_INFO);
				continue;
			}
		}
	}
out:

	closesocket(ogSock);
	LogMessage(L"OutGauge thread finished.", MSG_SEVERITY_DEBUG);

	return 0;
}

static DWORD WINAPI MessageThrFunc(LPVOID lpArg)
{
	UNREFERENCED_PARAMETER(lpArg);

	receiveMessages = TRUE;
	while (receiveMessages) {
		DWORD dwWait = WaitForSingleObject(hMsgEvent, 1000);
		
		switch (dwWait) {
		case WAIT_TIMEOUT:
			continue;
		case WAIT_OBJECT_0:
			{
				WinMutex lk(hMsgMutex, INFINITE);
				if (lk.status == WAIT_OBJECT_0) {
					CopyMemory(&sharedMsg, pSharedMsg, sizeof(sharedMsg));
					SharedMsg* pSM = (SharedMsg*)pSharedMsg;
					pSM->fPending = FALSE; /* Mark the message as read */
					ResetEvent(hMsgEvent);
					lk.Unlock();
					LogMessage(sharedMsg.msgTxt, MSG_SEVERITY_DEBUG);
				} else {
					LogMessage(L"(ERR)MsgThr> Unable to claim mutex.", MSG_SEVERITY_INFO);
					ResetEvent(hMsgEvent);
					receiveMessages = FALSE;
					continue;
				}
			}
			continue;
		default:
			LogMessage(L"(ERR)MsgThr> Error while waiting for event.", MSG_SEVERITY_INFO);
			receiveMessages = FALSE;
			break;
		}
	}

	/* Tell the DLL that we are done receving */
	{
		WinMutex lk(hMsgMutex, INFINITE);
		if (lk.status == WAIT_OBJECT_0) {
			SharedMsg* pSM = (SharedMsg*)pSharedMsg;
			pSM->fClosed = TRUE;
		}
	}

	UnmapViewOfFile(pSharedMsg);
	CloseHandle(hMsgSHM);
	CloseHandle(hMsgEvent);
	CloseHandle(hMsgMutex);

	return 0;
}

static VOID IS_STAHandler(struct IS_STA* packet)
{
	if (packet->Flags & ISS_GAME)
		LogMessage(L"In game", MSG_SEVERITY_DEBUG);
}