#include <windows.h>
#include <stdio.h>
#include <math.h>
#include "LFSORDLL.h"
#include "..\LibOVR\Src\OVR_CAPI.h"


// UTILS

typedef unsigned char		byte;
typedef unsigned short		word;
typedef const char			*ccs;

inline int max_int(int i, int j)	{ return i < j ? j : i; }
inline int min_int(int i, int j)	{ return i > j ? j : i; }

void copy_str_zz(char *dst, ccs s, int max_len)
{
	int len = 0;
	ccs n = s;
	char *d = dst;
	while (*n && len < max_len-1)
	{
		*d = *n;
		d++;
		n++;
		len++;
	}

	while (len < max_len)
	{
		*d = 0;
		d++;
		len++;
	}
}

const float OVR_SINGULARITYRADIUS = 0.0000001f; // Use for Gimbal lock numerical problems
const float OVR_PI = 3.14159265358979f;
const float OVR_PIOVER2 = 0.5f * OVR_PI;

void GetYPRFromQuat(const ovrQuatf *q, float *yaw, float *pitch, float *roll) // yaw, pitch, roll
{
	float xx = q->x * q->x;
	float yy = q->y * q->y;
	float zz = q->z * q->z;
    float ww = q->w * q->w;

    float s2 = -2.0f * (-q->w * q->x + q->y * q->z);

    if (s2 < -1.0f + OVR_SINGULARITYRADIUS) // South pole singularity
    {
        *yaw	=  0.0f;
        *pitch	= -OVR_PIOVER2;
        *roll	=  atan2f( 2.0f * (-q->y * q->x + q->w * q->z), ww + xx - yy - zz);
    }
    else if (s2 > 1.0f - OVR_SINGULARITYRADIUS) // North pole singularity
    {
        *yaw	=  0.0f;
        *pitch	=  OVR_PIOVER2;
        *roll	=  atan2f( 2.0f * (-q->y * q->x + q->w * q->z), ww + xx - yy - zz);
    }
    else
    {
        *yaw	= -atan2f(-2.0f * ( q->w * q->y + q->x * q->z), ww + zz - yy - xx);
        *pitch	=  asinf(s2);
        *roll	=  atan2f( 2.0f * ( q->w * q->z + q->y * q->x), ww + yy - xx - zz);
    }      
}


// STATIC

int					ovr_inited;
int					ovr_opened;
ovrHmd				hmd;
ovrEyeRenderDesc	eyeRenderDesc[2];
ovrTrackingState	tracking_state;
OVR_HMDInfo			hmd_info;


// FUNCTIONS

LFSOR_API int LFSOR_GetVersion()
{
	return LFSOR_VERSION;
}

LFSOR_API int LFSOR_Open(OVR_HMDStart *start_info, int *error)
{
	*error = OR_FAIL_NONE;

	if (ovr_opened) return LFSOR_OPEN_FAIL;

	if (sizeof(OVR_Vector2f)			!= sizeof(ovrVector2f)				||
		sizeof(OVR_DistortionVertex)	!= sizeof(ovrDistortionVertex)		||
		sizeof(OVR_DistortionMesh)		!= sizeof(ovrDistortionMesh)		||
		sizeof(OVR_UVScaleOffset)		!= 2 * sizeof(ovrVector2f)			||
		sizeof(OVR_FrameTiming)			!= sizeof(ovrFrameTiming)			||
		sizeof(OVR_Matrix4f)			!= sizeof(ovrMatrix4f))
	{
		return 0; // check for structure change if compiling with a new SDK
	}

	if (ovr_inited==0) // only try it once
	{
		if (ovr_Initialize()==0)
		{
			*error = OR_FAIL_INITIALIZE;
			return 0;
		}
	}

	ovr_inited = 1; // indicate that ovr_Initialize has been called - no longer calling ovr_Shutdown

	hmd = ovrHmd_Create(0);

	if (hmd==NULL)
	{
//		ovr_Shutdown(); never call this - causes D3D crashes
		*error = OR_FAIL_HMD_CREATE;
		return 0;
	}

	// check caps

	if ((hmd->HmdCaps & ovrHmdCap_Present)==0)
	{
		ovrHmd_Destroy(hmd);
//		ovr_Shutdown(); never call this - causes D3D crashes
		*error = OR_FAIL_NOT_PRESENT;
		return 0;
	}

	if ((hmd->HmdCaps & ovrHmdCap_ExtendDesktop)==0)
	{
		ovrHmd_Destroy(hmd);
//		ovr_Shutdown(); never call this - causes D3D crashes
		*error = OR_FAIL_NOT_EXTENDED;
		return 0;
	}

	copy_str_zz(start_info->ProductName,			hmd->ProductName,			32);
	copy_str_zz(start_info->DisplayDeviceName,		hmd->DisplayDeviceName,		32);
	start_info->HResolution = hmd->Resolution.w;
	start_info->VResolution = hmd->Resolution.h;

	ovr_opened = 1;
	return 1; // ok
}

LFSOR_API void LFSOR_Close()
{
	if (ovr_opened==0) return; // nothing to do

	memset(&hmd_info, 0, sizeof(OVR_HMDInfo));

	ovrHmd_Destroy(hmd);
//	ovr_Shutdown(); never call this - causes D3D crashes

	ovr_opened = 0;
}

LFSOR_API int LFSOR_QueryHMD(OVR_HMDInfo *info, int want_tracking)
{
	if (ovr_opened==0) return 0; // can't do anything

	// see similar code in GetCaps
	if (hmd->HmdCaps & ovrHmdCap_LowPersistence)			hmd_info.Caps |= OR_CAPS_LOWPERSISTENCE;
	if (hmd->HmdCaps & ovrHmdCap_DynamicPrediction)			hmd_info.Caps |= OR_CAPS_DYNAMICPREDICTION;
	if (hmd->HmdCaps & ovrHmdCap_NoVSync)					hmd_info.Caps |= OR_CAPS_NOVSYNC;

	if (hmd->TrackingCaps & ovrTrackingCap_Orientation)		hmd_info.Caps |= OR_TRACK_ORIENTATION;
	if (hmd->TrackingCaps & ovrTrackingCap_Position)		hmd_info.Caps |= OR_TRACK_POSITION;

	if (hmd->DistortionCaps & ovrDistortionCap_FlipInput)	hmd_info.Caps |= OR_DIST_FLIPINPUT;
	if (hmd->DistortionCaps & ovrDistortionCap_Vignette)	hmd_info.Caps |= OR_DIST_VIGNETTE;

	// Configure Stereo settings.
	ovrSizei recommendedTex0Size = ovrHmd_GetFovTextureSize(hmd, ovrEye_Left,	hmd->DefaultEyeFov[0], 1.0f);
	ovrSizei recommendedTex1Size = ovrHmd_GetFovTextureSize(hmd, ovrEye_Right,	hmd->DefaultEyeFov[1], 1.0f);

	hmd_info.RTRecommendedW = recommendedTex0Size.w + recommendedTex1Size.w;
	hmd_info.RTRecommendedH = max_int(recommendedTex0Size.h, recommendedTex1Size.h);

	// Initialize ovrEyeRenderDesc struct.
	eyeRenderDesc[0] = ovrHmd_GetRenderDesc(hmd, ovrEye_Left,  hmd->DefaultEyeFov[0]);
	eyeRenderDesc[1] = ovrHmd_GetRenderDesc(hmd, ovrEye_Right, hmd->DefaultEyeFov[1]);

	ovrEyeRenderDesc *src = eyeRenderDesc;
	OVR_EyeRenderDesc *dst = hmd_info.EyeRenderDesc;
	for (int i=0; i<2; i++)
	{
		dst->Fov.UpTan = src->Fov.UpTan;
		dst->Fov.DownTan = src->Fov.DownTan;
		dst->Fov.LeftTan = src->Fov.LeftTan;
		dst->Fov.RightTan = src->Fov.RightTan;

		dst->DistortedViewport.x = src->DistortedViewport.Pos.x;
		dst->DistortedViewport.y = src->DistortedViewport.Pos.y;
		dst->DistortedViewport.w = src->DistortedViewport.Size.w;
		dst->DistortedViewport.h = src->DistortedViewport.Size.h;

		src++;
		dst++;
	}

	hmd_info.IPD = eyeRenderDesc[0].ViewAdjust.x - eyeRenderDesc[1].ViewAdjust.x;

	memcpy(info, &hmd_info, sizeof(OVR_HMDInfo));

	if (want_tracking)
	{
		unsigned int supportedTrackingCaps = ovrTrackingCap_Orientation | ovrTrackingCap_MagYawCorrection | ovrTrackingCap_Position;
		unsigned int requiredTrackingCaps = 0;
	
		if (ovrHmd_ConfigureTracking(hmd, supportedTrackingCaps, requiredTrackingCaps))
		{
			return LFSOR_OPEN_SENSOR;
		}
	}

	return LFSOR_OPEN_NO_SENSOR; // ok
}

LFSOR_API void LFSOR_CreateMeshes(OVR_MeshInfo *info, int tex_width, int tex_height)
{
	ovrSizei textureSize;
	textureSize.w = tex_width;
	textureSize.h = tex_height;

	ovrRecti EyeRenderViewport[2];
	int midx = tex_width >> 1;

	EyeRenderViewport[0].Pos.x = 0;
	EyeRenderViewport[0].Pos.y = 0;
	EyeRenderViewport[0].Size.w = midx;
	EyeRenderViewport[0].Size.h = tex_height;

	EyeRenderViewport[1].Pos.x = midx;
	EyeRenderViewport[1].Pos.y = 0;
	EyeRenderViewport[1].Size.w = midx;
	EyeRenderViewport[1].Size.h = tex_height;

	for (int i=0; i<2; i++)
	{
		ovrDistortionMesh meshData;
		ovrHmd_CreateDistortionMesh(hmd, eyeRenderDesc[i].Eye, eyeRenderDesc[i].Fov, hmd->DistortionCaps, &meshData);
		memcpy(&info->DistortionMesh[i], &meshData, sizeof(ovrDistortionMesh));

		ovrVector2f uvScaleOffsetOut[2];
		ovrHmd_GetRenderScaleAndOffset(eyeRenderDesc[i].Fov, textureSize, EyeRenderViewport[i], uvScaleOffsetOut);
		memcpy(&info->UVScaleOffset[i], uvScaleOffsetOut, sizeof(OVR_UVScaleOffset));
	}
}

LFSOR_API void LFSOR_FreeMeshes(OVR_MeshInfo *info)
{
	for (int i=0; i<2; i++)
	{
		ovrDistortionMesh *mesh = (ovrDistortionMesh *)&info->DistortionMesh[i];
		ovrHmd_DestroyDistortionMesh(mesh);
	}
}

LFSOR_API double LFSOR_GetTimeInSeconds()
{
	return ovr_GetTimeInSeconds();
}

LFSOR_API int LFSOR_Peek(double abs_time, float *yaw, float *pitch, float *roll, float *x, float *y, float *z)
{
	tracking_state = ovrHmd_GetTrackingState(hmd, abs_time);

	const unsigned relevant_flags = ovrStatus_OrientationTracked | ovrStatus_PositionTracked;

	if ((tracking_state.StatusFlags & relevant_flags)==0) return 0;

	ovrPosef *pose = &tracking_state.HeadPose.ThePose;

	if (tracking_state.StatusFlags & ovrStatus_OrientationTracked)
	{
		GetYPRFromQuat(&pose->Orientation, yaw, pitch, roll); // yaw, pitch, roll
	}
	else
	{
		*yaw = *pitch = *roll = 0.0f;
	}

	if (tracking_state.StatusFlags & ovrStatus_PositionTracked)
	{
		*x = pose->Position.x;
		*y = pose->Position.y;
		*z = pose->Position.z;
		return 2;
	}

	*x = *y = *z = 0.0f;
	return 1;
}

LFSOR_API void LFSOR_Reset()
{
	ovrHmd_RecenterPose(hmd);
}

LFSOR_API unsigned LFSOR_GetCaps()
{
	unsigned ovr_caps = ovrHmd_GetEnabledCaps(hmd);

	unsigned lfs_caps = 0;

	// see similar code in QueryHMD
	if (ovr_caps & ovrHmdCap_LowPersistence)		lfs_caps |= OR_CAPS_LOWPERSISTENCE;
	if (ovr_caps & ovrHmdCap_DynamicPrediction)		lfs_caps |= OR_CAPS_DYNAMICPREDICTION;
	if (ovr_caps & ovrHmdCap_NoVSync)				lfs_caps |= OR_CAPS_NOVSYNC;

	return lfs_caps;
}

LFSOR_API void LFSOR_SetCaps(unsigned lfs_caps)
{
//	unsigned keep_caps_mask = ovrHmdCap_Writable_Mask ^ (ovrHmdCap_LowPersistence | ovrHmdCap_DynamicPrediction | ovrHmdCap_NoVSync);

//	unsigned ovr_caps = ovrHmd_GetEnabledCaps(hmd) & keep_caps_mask;

	unsigned ovr_caps = 0;

	if (lfs_caps & OR_CAPS_LOWPERSISTENCE)		ovr_caps |= ovrHmdCap_LowPersistence;
	if (lfs_caps & OR_CAPS_DYNAMICPREDICTION)	ovr_caps |= ovrHmdCap_DynamicPrediction;
	if (lfs_caps & OR_CAPS_NOVSYNC)				ovr_caps |= ovrHmdCap_NoVSync;

	ovrHmd_SetEnabledCaps(hmd, ovr_caps);
}

LFSOR_API void LFSOR_BeginFrameTiming(OVR_FrameTiming *timing)
{
	ovrFrameTiming ovrt = ovrHmd_BeginFrameTiming(hmd, 0);
	memcpy(timing, &ovrt, sizeof(ovrFrameTiming));
}

LFSOR_API void LFSOR_EndFrameTiming()
{
	ovrHmd_EndFrameTiming(hmd);
}

LFSOR_API void LFSOR_GetTimewarpMatrices(OVR_Matrix4f *four_mats)
{
	OVR_Matrix4f *eye_mats = four_mats;
	for (int i=0; i<2; i++)
	{
		ovrHmd_GetEyeTimewarpMatrices(hmd, (ovrEyeType)i, tracking_state.HeadPose.ThePose, (ovrMatrix4f *)eye_mats);
		eye_mats += 2;
	}
}

LFSOR_API void LFSOR_GetHSWInfo(OVR_HSWDisplayState *state)
{
	ovrHSWDisplayState ovr_hsw_state;

	ovrHmd_GetHSWDisplayState(hmd, &ovr_hsw_state);

	state->Displayed		= ovr_hsw_state.Displayed;
	state->Sp1				= 0;
	state->Sp2				= 0;
	state->Sp3				= 0;
	state->StartTime		= ovr_hsw_state.StartTime;
	state->DismissibleTime	= ovr_hsw_state.DismissibleTime;
}

LFSOR_API int LFSOR_DismissHSWDisplay()
{
	return ovrHmd_DismissHSWDisplay(hmd);
}