/*  This file is part of LFSBench for Linux/X11
 *
 *  LFSBench for Linux/X11 is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  LFSBench for Linux/X11 is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.

 *  You should have received a copy of the GNU General Public License
 *  along with LFSBench for Linux/X11. If not, see <http://www.gnu.org/licenses/>.
 * 
 */

#include "appwindow.h"
#include "fpscounter.h"
#include <sys/ptrace.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <ctype.h>
#include <dirent.h>
#include <math.h>
#include <pthread.h>

#define READ_BUF_SIZE 50
#define FPS_OFFSET_6E 0x00988B6C /* Uncertain */
#define FPS_OFFSET_6E12 0x00921DC4
#define FPS_OFFSET_6F 0x923644
#define LFS_EXEC_NAME "LFS.exe"
#define LFS_EXEC_NAME_6E12 "LFS_6E12.exe"


#define SEC_TO_USEC(n) (n * 1000000)
#define USEC_TO_NSEC(n) (n * 1000)

static pid_t lfs_pid = 0;
static volatile int run_loop = 0;
static pthread_t benchmarking_loop_thr;
static int benchmark_running = 0;

#ifdef LFSVER_6B
static const off_t OFFSET = FPS_OFFSET_6B;
static const char* EXEC_NAME = LFS_EXEC_NAME;
#elif defined LFSVER_6E12
static const off_t OFFSET = FPS_OFFSET_6E12;
static const char* EXEC_NAME = LFS_EXEC_NAME_6E12;
#else
static const off_t OFFSET = FPS_OFFSET_6F;
static const char* EXEC_NAME = LFS_EXEC_NAME;
#endif

static void* benchmarking_loop();
static void create_fraps_output(const float, const long long, const float, const float, const float);
static pid_t find_pid_by_name(const char*);
static void ptrace_error_str(const int, char** const);
static int start_process();
static int stop_process();

/** Checks if LFS is running and accesses it's memory
  * using /proc/$PID/mem to read the FPS count. */
int init_fps_counter(exit_thrdata* etd)
{
  char* noexe = strstr(EXEC_NAME, ".exe");
  size_t len = noexe - EXEC_NAME;
  noexe = malloc(len + 1);
  memcpy(noexe, EXEC_NAME, len);
  noexe[len] = 0;

  /* Get PID of the LFS process */
  append_window_text("Waiting for LFS to start...", 0);
  while(lfs_pid == 0) {
    lfs_pid = find_pid_by_name(EXEC_NAME);
    if (lfs_pid == 0)
      lfs_pid = find_pid_by_name(noexe);

    pthread_mutex_lock(etd->exit_mutex);
    if (*(etd->pexit_app) == 1) {
      pthread_mutex_unlock(etd->exit_mutex);
      goto out;
    }
    pthread_mutex_unlock(etd->exit_mutex);
    sleep(1);
  }
  append_window_text("LFS process found.", 0);

out:
  free(noexe);
  return 0;
}

/** Starts or stops the benchmarking loop */
int start_stop_benchmark() {
  if (benchmark_running == 0) {
    run_loop = 1;
    if (pthread_create(&benchmarking_loop_thr, NULL, &benchmarking_loop, NULL) != 0) {
      append_window_text("CRITICAL: Cannot start benchmark!", 0);
      run_loop = 0;
      return -1;
    }
    benchmark_running = 1;
    return 0;
  } else {
    run_loop = 0;
    pthread_join(benchmarking_loop_thr, NULL);
    benchmark_running = 0;
  }
  return 0;
}

/** Continouously reads the FPS count and calculates
  * minimum, maximum and average FPS in the end. */
static void* benchmarking_loop()
{
  const struct timespec RATE_NSEC = { 0, 50000000 };
  const int ONE_SECOND = 1000000000;
  double max_fps = 0.0;
  double min_fps = 1001.0; /* LFS won't render faster than 1000 FPS */
  double total_frames = 0.0;
  double avg_fps;
  struct timeval tp_start, tp_end;
  long long start_usec, end_usec, diff;
  long traceret;

  traceret = ptrace(PTRACE_SEIZE, lfs_pid, NULL, 0);
  if (traceret != 0) {
    char* s = NULL;
    char* ss = NULL;
    ptrace_error_str(errno, &ss);
    asprintf(&s, "Cannot ptrace LFS process, %s", ss);
    append_window_text(s, 0);
    free(s);
    free(ss);
    return NULL;
  }
  append_window_text("Benchmark started.", 0);

  gettimeofday(&tp_start, NULL);

  while(run_loop) {
    if (stop_process())
      goto out;
    errno = 0;
    traceret = ptrace(PTRACE_PEEKDATA, lfs_pid, OFFSET, NULL);
    if (errno == 0) {
      float _fps = *((float*)(&traceret));
      double fps = _fps;
      if (fps > max_fps)
	max_fps = fps;
      if (fps < min_fps)
	min_fps = fps;
      total_frames += (fps * RATE_NSEC.tv_nsec) / ONE_SECOND;
    } else {
      char* s = NULL;
      char* ss = NULL;
      ptrace_error_str(errno, &ss);
      asprintf(&s, "CRITICAL: Error reading FPS, %s", ss);
      append_window_text(s, 0);
      free(s);
      free(ss);
      goto out;
    }
    if (start_process())
      goto out;
    nanosleep(&RATE_NSEC, NULL);
  }
  gettimeofday(&tp_end, NULL);
  append_window_text("Benchmark completed.", 0);

  start_usec = SEC_TO_USEC(tp_start.tv_sec) + tp_start.tv_usec;
  end_usec = SEC_TO_USEC(tp_end.tv_sec) + tp_end.tv_usec;
  diff = end_usec - start_usec;
  avg_fps = (total_frames * ONE_SECOND) / USEC_TO_NSEC(diff);

  char* s = NULL;

  asprintf(&s, "TIME Start: %ld, %ld End: %ld, %ld", tp_start.tv_sec, tp_start.tv_usec, tp_end.tv_sec, tp_end.tv_usec);
  append_window_text(s, 1);
  free(s);

  asprintf(&s, "TIME Diff: %lld", diff);
  append_window_text(s, 1);
  free(s);

  asprintf(&s, "Total frames: %g", total_frames);
  append_window_text(s, 1);
  free(s);

  asprintf(&s, "Max: %g, Min: %g, Avg: %g", max_fps, min_fps, avg_fps);
  append_window_text(s, 1);
  free(s);
  manual_xflush();

  create_fraps_output(total_frames, diff, avg_fps, min_fps, max_fps);
  
out:
  ptrace(PTRACE_DETACH, lfs_pid, 0, 0);
  return NULL;
}

/** Stops benchmark if it's running */
void stop_benchmark()
{
  if (run_loop == 1) {
    run_loop = 0;
    pthread_join(benchmarking_loop_thr, NULL);
  }
}

/** Creates FRAPS-like output and writes in to a file */
static void create_fraps_output(const float total_frames, const long long diff, const float avg_fps, 
				const float min_fps, const float max_fps)
{
  const char* filename = "lfsbench_fps.txt";
  const int fraps_total_frames = roundf(total_frames);
  const int fraps_min_fps = roundf(min_fps);
  const int fraps_max_fps = roundf(max_fps);
  const long fraps_time_msec = diff / 1000;
  time_t curtime;
  struct tm* t;
  
  curtime = time(NULL);
  t = localtime(&curtime);
  
  FILE* fraps_output;
  fraps_output = fopen(filename, "a+");
  if (fraps_output == NULL) {
    append_window_text("WARNING: Could not write results to log", 0);
    return;
  }
  
  fprintf(fraps_output, "%d-%02d-%02d %02d:%02d:%02d - LFS\n", t->tm_year+1900, t->tm_mon+1, t->tm_mday, t->tm_hour, t->tm_min, t->tm_sec);
  fprintf(fraps_output, "Frames: %d - Time: %ldms - Avg: %f - Min: %d - Max: %d\n", fraps_total_frames, fraps_time_msec, avg_fps, fraps_min_fps, fraps_max_fps);
  fprintf(fraps_output, "\n");

  fclose(fraps_output);
}

/** Gets a PID from a process name.
  * If there is more than one process of that name, PID of the first matching process is returned. */
static pid_t find_pid_by_name(const char* pidName)
{
	DIR *dir;
	struct dirent *next;

	dir = opendir("/proc");
	if (!dir) {
		perror("Cannot open /proc");
		return 0;
	}
	
	while ((next = readdir(dir)) != NULL) {
		FILE* status;
		char filename[READ_BUF_SIZE];
		char buffer[READ_BUF_SIZE];
		char name[READ_BUF_SIZE];

		/* Must skip ".." since that is outside /proc */
		if (strcmp(next->d_name, "..") == 0)
			continue;

		/* If it isn't a number, we don't want it */
		if (!isdigit(*next->d_name))
			continue;

		sprintf(filename, "/proc/%s/status", next->d_name);
		if (! (status = fopen(filename, "r")) ) {
			continue;
		}
		if (fgets(buffer, READ_BUF_SIZE-1, status) == NULL) {
			fclose(status);
			continue;
		}
		fclose(status);

		/* Buffer should contain a string like "Name:   binary_name" */
		sscanf(buffer, "%*s %s", name);
		if (strcmp(name, pidName) == 0) {
			long pid = strtol(next->d_name, NULL, 0);
			closedir(dir);
			return pid;
		}
	}
	
	closedir(dir);
	return 0;
}

static void ptrace_error_str(const int err, char** const str)
{
  switch (err) {
    case EPERM:
      asprintf(str, "Bad permissions");
      break;
    case ESRCH:
      asprintf(str, "No such process");
      break;
    case EIO:
      asprintf(str, "Invalid memory address");
      break;
    case EFAULT:
      asprintf(str, "Access to unmapped memory");
      break;
    case EINVAL:
      asprintf(str, "Invalid operation");
      break;
    case EBUSY:
      asprintf(str, "Debug register operation failed");
      break;
    default:
      asprintf(str, "Unknown error");
  }
}

static int start_process()
{
  long ret;

  errno = 0;
  ret = ptrace(PTRACE_CONT, lfs_pid, NULL, 0);
  if (ret) {
    char* s, *ss;
    ptrace_error_str(errno, &ss);
    asprintf(&s, "Cannot restart LFS process, %s", ss);
    append_window_text(s, 0);
    free(s);
    free(ss);
    return 1;
  }
  return 0;
}

static int stop_process()
{
  long ret;

  errno = 0;
  ret = ptrace(PTRACE_INTERRUPT, lfs_pid, NULL, 0);
  if (ret) {
    char* s, *ss;
    ptrace_error_str(errno, &ss);
    asprintf(&s, "Cannot stop LFS process, %s", ss);
    append_window_text(s, 0);
    free(s);
    free(ss);
    return 1;
  }
  waitpid(lfs_pid, NULL, 0);
  return 0;
}