/*  This file is part of LFSBench for Linux/X11
 *
 *  LFSBench for Linux/X11 is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  LFSBench for Linux/X11 is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.

 *  You should have received a copy of the GNU General Public License
 *  along with LFSBench for Linux/X11. If not, see <http://www.gnu.org/licenses/>.
 * 
 */

#include "appwindow.h"
#include "sharedstructs.h"

#include <X11/Xutil.h>
#include <X11/keysym.h>

#define TEXT_TOP_OFFSET 15
#define TEXT_LEFT_OFFSET 10
#define TEXT_LINE_OFFSET 12

#ifdef LFSVER_6B
#define LFS_VERSTR "0.6B"
#elif defined LFSVER_6E12
#define LFS_VERSTR "0.6E12"
#else
#define LFS_VERSTR "0.6F"
#endif

static Display* Xdisp;
static GC Xgc;
static Window Xwin;
static KeyCode KC_Q;
static int XblackColor;
static int XwhiteColor;

static char** window_text = NULL;
static int last_text_y_pos;
static int run_loop;
static unsigned int window_text_count = 0;
static pthread_mutex_t draw_mutex;
static Atom wmDeleteWindow;

static const struct timespec wait_usecs = { 0, 20000000 };

static void redraw_window();

/** Draws a new line of text into the window. */
int append_window_text(char* text, int dont_flush)
{
  pthread_mutex_lock(&draw_mutex);
  /* Append the text to the list */
  window_text = (char**)realloc(window_text, ++window_text_count * sizeof(char**));
  if (window_text == NULL) {
    fprintf(stderr, "CRITICAL: Insufficient memory to reacollate text array!\n.");
    run_loop = 0;
    return ENOMEM;
  }
  
  window_text[window_text_count - 1] = (char*)malloc((strlen(text) + 1) * sizeof(char));
  if (window_text[window_text_count - 1] == NULL) {
    fprintf(stderr, "CRITICAL: Insufficient memory to append text!\n");
    run_loop = 0;
    return ENOMEM;
  }
  strcpy(window_text[window_text_count - 1], text);

  /* Draw the text to the window */
  XDrawString(Xdisp, Xwin, Xgc, TEXT_LEFT_OFFSET, last_text_y_pos, text, strlen(text));
  last_text_y_pos += TEXT_LINE_OFFSET;
  
  if (dont_flush != 1)
    XFlush(Xdisp);
  
  pthread_mutex_unlock(&draw_mutex);
  
  return 0;
}

/** Creates and displays the main window
  * of the application */
int create_window()
{
  /* Initialize the draw_mutex.
   * Multiple threads can draw to the window simultaneously,
   * this mutex makes sure the drawing functions are thread-safe. */
  if (pthread_mutex_init(&draw_mutex, NULL) != 0) {
    fprintf(stderr, "CRITICAL: Cannot init draw mutex.\n");
    return -1;
  }
  
  Xdisp = XOpenDisplay(NULL);
  if (Xdisp == NULL) {
    fprintf(stderr, "CRITICAL: Unable to open X display!\n");
    return -1;
  }
  
  /* Get black and white color */
  XblackColor = BlackPixel(Xdisp, DefaultScreen(Xdisp));
  XwhiteColor = WhitePixel(Xdisp, DefaultScreen(Xdisp));
  
  /* Quit key keycode */
  KC_Q = XKeysymToKeycode(Xdisp, XK_Q);
  
  /* Create the window */
  Xwin = XCreateSimpleWindow(Xdisp, DefaultRootWindow(Xdisp),
			     0, 0, 350, 250, 0, XblackColor, XwhiteColor);
  XSetStandardProperties(Xdisp, Xwin, "LFS Benchmark", "LFSBench", None, NULL, 0, NULL);
  wmDeleteWindow = XInternAtom(Xdisp, "WM_DELETE_WINDOW", False);
  XSetWMProtocols(Xdisp, Xwin, &wmDeleteWindow, 1);
  XMapWindow(Xdisp, Xwin);
  Xgc = XCreateGC(Xdisp, Xwin, 0, NULL);
  XSetForeground(Xdisp, Xgc, XblackColor);
  XSetBackground(Xdisp, Xgc, XwhiteColor);
  XClearWindow(Xdisp, Xwin);
  XMapRaised(Xdisp, Xwin);
  XFlush(Xdisp);
  nanosleep(&wait_usecs, NULL);

  /* Print welcome text */
  last_text_y_pos = TEXT_TOP_OFFSET;
  char* verinfo;
  asprintf(&verinfo, "LFS %s benchmarking tool for Linux/X11", LFS_VERSTR);
  append_window_text(verinfo, 1);
  append_window_text("========================================", 1);
  append_window_text("To start the benchmark, press F11 when", 1);
  append_window_text("instructed in the replay.", 1);
  append_window_text("", 1);
  manual_xflush();
  free(verinfo);
  
  return 0;
}

/** Destroys the window and cleans up */
void destroy_window()
{
  XFreeGC(Xdisp, Xgc);
  XDestroyWindow(Xdisp, Xwin);
  XCloseDisplay(Xdisp);
}

/** Manually flushes the drawing requests.
  * Must be called after calling "append_window_text()" with "dont_flush = 1" */
void manual_xflush()
{
  pthread_mutex_lock(&draw_mutex);
  XFlush(Xdisp);
  pthread_mutex_unlock(&draw_mutex);
}

/** Runs the main window's event loop */
void* window_event_loop(void* args)
{
  run_loop = 1;
  exit_thrdata* etd = (exit_thrdata*)args;
  
  XSelectInput(Xdisp, Xwin, KeyPressMask | ExposureMask | StructureNotifyMask);
  while(run_loop) {
    XEvent Xev;
    XNextEvent(Xdisp, &Xev);
    switch (Xev.type) {
      case KeyPress:
	if (Xev.xkey.keycode == KC_Q) {
	  run_loop = 0;
	  pthread_mutex_lock(etd->exit_mutex);
	  *(etd->pexit_app) = 1;
	  pthread_mutex_unlock(etd->exit_mutex);
	}
      case Expose:
	redraw_window();
	break;
      case ConfigureNotify:
	redraw_window();
	break;
      case ClientMessage:
	if (Xev.xclient.data.l[0] == wmDeleteWindow) {
	  run_loop = 0;
	  pthread_mutex_lock(etd->exit_mutex);
	  *(etd->pexit_app) = 1;
	  pthread_mutex_unlock(etd->exit_mutex);
	}
	break;
      default:
	break;
    }

    /* Check if we should exit the loop */
    pthread_mutex_lock(etd->exit_mutex);
    if (*(etd->pexit_app) == 1)
      run_loop = 0;
    pthread_mutex_unlock(etd->exit_mutex);
  }

  return NULL;
}

/** Thread-safe redrawing routine.
  * Called then the window is exposed, moved or resized. */
static void redraw_window()
{
  pthread_mutex_lock(&draw_mutex);
  int i;
  int text_y_pos = TEXT_TOP_OFFSET;
  
  for (i = 0; i < window_text_count; i++) {
    XDrawString(Xdisp, Xwin, Xgc, TEXT_LEFT_OFFSET, text_y_pos, window_text[i], strlen(window_text[i]));
    text_y_pos += TEXT_LINE_OFFSET;
  }
  XFlush(Xdisp);
  pthread_mutex_unlock(&draw_mutex);
}
