///
/// @file
/// @details Create a bezier curve from a container of points along the curve for more precision.
/// <!-- Copyright  Tim Beaudet 2014 - All Rights Reserved -->
///-----------------------------------------------------------------------------------------------------------------///

#ifndef _GameMath_BezierCurve_h_
#define _GameMath_BezierCurve_h_

#include "game_math.h"
#include <vector>

namespace GameMath
{

	class Curve
	{
	public:
		///
		/// @details Construct a curve object from a set of points that lay along the curve using a catmullrom to
		///   configure the control points, b and c.
		///
		/// @param points   Points that lay on the desired curve.  Must contain at least 2 points for a not looping
		///   curve, or at least 5 for a loop.
		/// @param isLoop   True if the curved path should use the last and first points to create a continuous loop.
		///
		/// @note Non-looping curves have not been tested at this point in development, but exists for generalizing, the
		///   minimum number of points in a curved loop can likely be lowered, this is currently set arbritrarily.
		///
		Curve(const std::vector<GameMath::Vector3>& points, bool isLoop);
		~Curve(void);

		///
		/// @details Takes a curve object and breaks it back into lines.  This variant will split each curve section into
		///   several small lines.  If the curve section is shorter, the amount of lines will remain the same as a longer
		///   curve section.  Obviously, the more steps per section, the more lines will be created, so carefully increase.
		///
		void MakeLines(std::vector<GameMath::Vector3>& points, const size_t& stepsPerSection = 20) const;

		///
		/// @details Takes a curve object and breaks it back into lines.  This variant will make each line approximately
		///   the length specified by maxLineLength independent of the actual curve sections.  Shorter line lenghts will
		///   increase accurracy whereas a longer line may skip over portions of the curve.  Do not, the lineLength is 
		///   not a guaranteed that the points given back will actually be that length!
		///
		/// @note Not yet implemented
		///
		void MakeLines(std::vector<GameMath::Vector3>& points, const float maxLineLength);

		///
		/// @details Estimates the length of the entire curve, the more steps the more accurate.  In the case of the FE1
		///   centerline 1,602 meters was found to be the length at 25 steps, but at 250 steps 90m was added to the length
		///   to bring the total to 1692 meters.
		///
		float GetEstimatedLength(const size_t& stepsPerSection = 25) const;

	private:
		///
		/// @details Set up each section of the curve object from the given set of points that lay along the curve using
		///   catmullrom to configure the control points of the section, b and c where a and d are the start and end points
		///   that exist on the curve.
		///
		/// @note: isLoop = false has not been tested.
		///
		void CreateCurve(const std::vector<GameMath::Vector3>& points, bool isLoop);

		struct Section
		{
			GameMath::Vector3 a, b, c, d;  //a = start, d = end, b and c are control points.
		};

		std::vector<Section> mSections;
		mutable float mEstimatedLength;
		bool mIsLoop;
	};

}; //namespace GameMath

#endif /* _GameMath_BezierCurve_h_ */
