///
/// @file
/// @details Create a bezier curve from a container of points along the curve for more precision.
/// <!-- Copyright  Tim Beaudet 2014 - All Rights Reserved -->
///-----------------------------------------------------------------------------------------------------------------///

#include "bezier_curve.h"
#include "tb_assert.h"

//-------------------------------------------------------------------------------------------------------------------//

GameMath::Curve::Curve(const std::vector<GameMath::Vector3>& points, bool isLoop) :
	mEstimatedLength(-1.0f),
	mIsLoop(isLoop)
{ 
	CreateCurve(points, mIsLoop);
}

//-------------------------------------------------------------------------------------------------------------------//

GameMath::Curve::~Curve(void)
{
}

//-------------------------------------------------------------------------------------------------------------------//

GameMath::Vector3 LinearInterp(const GameMath::Vector3& start, const GameMath::Vector3& end, const float percentage)
{
	const GameMath::Vector3 diff(end - start);
	return start + (diff * percentage);
}

void GameMath::Curve::MakeLines(std::vector<GameMath::Vector3>& points, const size_t& stepsPerSection) const
{
	points.clear();
	points.reserve(mSections.size() * stepsPerSection);

	GameMath::Vector3 tempA, tempB, tempC, tempAA, tempBB;

	for (size_t sectionIndex(0); sectionIndex < mSections.size(); ++sectionIndex)
	{
		const Section& section(mSections[sectionIndex]);

		for (size_t step(0); step < stepsPerSection; ++step)
		{
			const float t = static_cast<float>(step) / static_cast<float>(stepsPerSection);
			
			tempA = LinearInterp(section.a, section.c, t);
			tempB = LinearInterp(section.c, section.d, t);
			tempC = LinearInterp(section.d, section.b, t);

			tempAA = LinearInterp(tempA, tempB, t);
			tempBB = LinearInterp(tempB, tempC, t);

			points.push_back(LinearInterp(tempAA, tempBB, t));
		}
	}
}

//-------------------------------------------------------------------------------------------------------------------//

void GameMath::Curve::CreateCurve(const std::vector<GameMath::Vector3>& points, bool isLoop)
{
	tb_assert_if(points.size() < 2, "Invalid parameter points, expected at least 2 points for a nonlooping curve.");
	tb_assert_if(true == isLoop && points.size() < 5, "Invalid parameter points, expected at least 10 points for a looping curve.");

	mSections.clear();
	mSections.reserve(points.size() - 1);
	Section tempSection;

	GameMath::Vector3 prevPoint((true == isLoop) ? points.back() : points.front());
	GameMath::Vector3 currPoint(points.front());
	GameMath::Vector3 nextPoint(points[1]);
	GameMath::Vector3 afterPoint((points.size() > 2) ? points[2] : points[1]);

	for (size_t index(1); ((false == isLoop && index < points.size() - 1) || (true == isLoop && index < points.size())); ++index)
	{
		//C0 = P1 + (P2 - P0) / 6
		//D0 = P2 - (P3 - P1) / 6

		tempSection.a = currPoint;
		tempSection.b = nextPoint;
		tempSection.c = tempSection.a + (tempSection.b - prevPoint) / 6.0f;
		tempSection.d = tempSection.b - (afterPoint - tempSection.a) / 6.0f;

		mSections.push_back(tempSection);

		prevPoint = points[index - 1];
		currPoint = points[index];
		nextPoint = (true == isLoop) ? points[(index + 1) % points.size()] : points[index + 1];
		afterPoint = (true == isLoop) ? points[(index + 2) % points.size()] : (index + 2 >= points.size()) ? tempSection.b : points[index + 2];
	}
}

//-------------------------------------------------------------------------------------------------------------------//

void GameMath::Curve::MakeLines(std::vector<GameMath::Vector3>& points, const float maxLineLength)
{
}

//-------------------------------------------------------------------------------------------------------------------//

float GameMath::Curve::GetEstimatedLength(const size_t& stepsPerSection) const
{
	if (mEstimatedLength < 0.0f)
	{
		mEstimatedLength = 0.0f;
		
		bool hasFirstPoint = false;
		GameMath::Vector3 currentPoint, previousPoint;
		GameMath::Vector3 tempA, tempB, tempC, tempAA, tempBB;

		for (size_t sectionIndex(0); sectionIndex < mSections.size(); ++sectionIndex)
		{
			const Section& section(mSections[sectionIndex]);

			for (size_t step(0); step < stepsPerSection; ++step)
			{
				const float t = static_cast<float>(step) / static_cast<float>(stepsPerSection);
			
				tempA = LinearInterp(section.a, section.c, t);
				tempB = LinearInterp(section.c, section.d, t);
				tempC = LinearInterp(section.d, section.b, t);

				tempAA = LinearInterp(tempA, tempB, t);
				tempBB = LinearInterp(tempB, tempC, t);

				currentPoint = LinearInterp(tempAA, tempBB, t);

				if (true == hasFirstPoint)
				{
					const float stepLength = (previousPoint - currentPoint).Magnitude();
					mEstimatedLength += stepLength;
				}

				previousPoint = currentPoint;
				hasFirstPoint = true;
			}
		}
	}

	return mEstimatedLength;
}

//-------------------------------------------------------------------------------------------------------------------//
