#!/usr/bin/env python
# coding=utf-8
#
# Copyright Constantin Köpplinger 2009.
#
# This file is part of OutGaugeRelay.
#
# OutGaugeRelay is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# OutGaugeRelay is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with OutGaugeRelay. If not, see <http://www.gnu.org/licenses/>.

# Some filtering / shaping for CSR
def csr(pack):
	if pack.Car == 'VWS': pack.Car = 'XFG' # Use this to assign a different sound set to the VWS
	if pack.RPM < 10:
		pack.RPM = -20000.0
		pack.Throttle = 0.0
	if (pack.ShowLights & (256 | 512)) or pack.Fuel == 0.0: pack.Throttle = 0.0
	return True

import socket, struct

HOST = 'localhost'
OG_PORT = 6699 # OutGauge port as defined in LFS's cfg.txt

# Applications (ports, actually) to forward incoming OutGauge traffic to.
# A tuple containing tuples of port, OutGauge ID, the convert toggle and a custom function.
# The custom function's purpose is to evaluate whether the packet should be sent or not.
# It must take one argument and is expected to return a boolean. It can manipulate the packet contents.
# Can be either a string containing the function name or a function object. Set to None if all packets are to be sent.
#
# Example for CSR: (35555, 0, True, minRPM)
# Note: If you only need OGRelay for the conversion to old-style OutGauge 
# packets, you'll have to append a comma to the end of your tuple for python
# to actually interpret the FORWARD_TO value as a tuple.
FORWARD_TO = ((35555, 0, True, csr),)

# No need to change anything below
BUFFER_SIZE = 1024
OGP_IS4 = struct.Struct('I4sH2B12f16s16si')
OGP_Z25 = struct.Struct('I4sH2B7f2i3f16s16si')

class OGPACK_IS4:
	def __init__(self, data):
		if data:
			self.unpack(data)
		else:
			self.Time		= 0
			self.Car		= ''
			self.Flags		= 0
			self.Gear		= 0
			self.SpareB		= 0
			self.Speed		= 0.0
			self.RPM		= 0.0
			self.Turbo		= 0.0
			self.EngTemp	= 0.0
			self.Fuel		= 0.0
			self.OilPress	= 0.0
			self.OilTemp	= 0.0
			self.Spare2		= 0.0
			self.Spare3		= 0.0
			self.Throttle	= 0.0
			self.Brake		= 0.0
			self.Clutch		= 0.0
			self.Display1	= ''
			self.Display2	= ''
			self.ID			= 0
	def pack(self):
		return OGP_IS4.pack(self.Time, self.Car, self.Flags, self.Gear, self.SpareB, self.Speed, self.RPM, self.Turbo, self.EngTemp, self.Fuel, self.OilPress, self.OilTemp, self.Spare2, self.Spare3, self.Throttle, self.Brake, self.Clutch, self.Display1, self.Display2, self.ID)
	def unpack(self, data):
		self.Time, self.Car, self.Flags, self.Gear, self.SpareB, self.Speed, self.RPM, self.Turbo, self.EngTemp, self.Fuel, self.OilPress, self.OilTemp, self.Spare2, self.Spare3, self.Throttle, self.Brake, self.Clutch, self.Display1, self.Display2, self.ID = OGP_IS4.unpack(data)
		
class OGPACK_Z25:
	def __init__(self, data):
		if data:
			self.unpack(data)
		else:
			self.Time		= 0
			self.Car		= ''
			self.Flags		= 0
			self.Gear		= 0
			self.SpareB		= 0
			self.Speed		= 0.0
			self.RPM		= 0.0
			self.Turbo		= 0.0
			self.EngTemp	= 0.0
			self.Fuel		= 0.0
			self.OilPress	= 0.0
			self.OilTemp	= 0.0
			self.DashLights	= 0
			self.ShowLights	= 0
			self.Throttle	= 0.0
			self.Brake		= 0.0
			self.Clutch		= 0.0
			self.Display1	= ''
			self.Display2	= ''
			self.ID			= 0
	def pack(self):
		return OGP_Z25.pack(self.Time, self.Car, self.Flags, self.Gear, self.SpareB, self.Speed, self.RPM, self.Turbo, self.EngTemp, self.Fuel, self.OilPress, self.OilTemp, self.DashLights, self.ShowLights, self.Throttle, self.Brake, self.Clutch, self.Display1, self.Display2, self.ID)
	def unpack(self, data):
		self.Time, self.Car, self.Flags, self.Gear, self.SpareB, self.Speed, self.RPM, self.Turbo, self.EngTemp, self.Fuel, self.OilPress, self.OilTemp, self.DashLights, self.ShowLights, self.Throttle, self.Brake, self.Clutch, self.Display1, self.Display2, self.ID = OGP_Z25.unpack(data)
	def toIS4(self):
		dashflags = 31 & self.ShowLights # first five bits
		if self.ShowLights & 128: dashflags |= 192 # ANY convert to BOTH
		else:
			if self.ShowLights & 32: dashflags |= 64 #left
			elif self.ShowLights & 64: dashflags |= 128 #right
		dashflags |= ((self.ShowLights & 2147483393) << 1) # 
		return OGPACK_IS4(OGP_Z25.pack(self.Time, self.Car, self.Flags | dashflags, self.Gear, self.SpareB, self.Speed, self.RPM, self.Turbo, self.EngTemp, self.Fuel, self.OilPress, self.OilTemp, 0, 0, self.Throttle, self.Brake, self.Clutch, self.Display1, self.Display2, self.ID))

print 'OutGauge Relay started. Forwarding to the following %d port(s):' % len(FORWARD_TO)
for app in FORWARD_TO:
	print app[0]
print "--------------------------------"
s = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
try:
	s.bind((HOST, OG_PORT))
except (socket.error, socket.herror), err:
	print 'Could not bind to port %d: %s' % (OG_PORT, err)
try:
	for app in FORWARD_TO:
		if not callable(app[3]):
			if callable(eval(app[3])):
				app[3] = eval(app[3])

	pack = data = ''
	while True:
		try:
			data = s.recv(BUFFER_SIZE)
			if len(data) == 92: data += '\x00\x00\x00\x00' # no ID set but our precompiled struct needs it
			if len(data) != 96: continue # corrupted/invalid packet
			pack = OGPACK_Z25(data)
		except: pass
		for app in FORWARD_TO:
			try:
				if app[1] != 0:
					pack.ID = app[1]
				if callable(app[3]):
					if app[3](pack) == False: continue
				data = pack.toIS4().pack() if app[2] else pack.pack()
				if app[1] == 0:
					data = data[:-4]
				s.sendto(data, (HOST, app[0]))
			except: pass
finally:
	s.close()