#include "StdAfx.h"
#include "ConfigLoader.h"
#include <algorithm>
#include <fstream>
#include <iostream>
#include <sstream>

const std::string commentChar = "#";
const std::string rpmPrefix = "RPM_";
const std::string debugDirective = "DEBUG";

ConfigLoader::ConfigLoader(void)
{
	//Initialize config with default values
	rpmCfg["XFG"].firstLeds = 5000;
	rpmCfg["XFG"].redline = 8000;
	rpmCfg["UF1"].firstLeds = 4000;
	rpmCfg["UF1"].redline = 7000;
	rpmCfg["XRG"].firstLeds = 4000;
	rpmCfg["XRG"].redline = 7000;
	rpmCfg["LX4"].firstLeds = 6000;
	rpmCfg["LX4"].redline = 9000;
	rpmCfg["LX6"].firstLeds = 6000;
	rpmCfg["LX6"].redline = 9000;
	rpmCfg["RB4"].firstLeds = 4500;
	rpmCfg["RB4"].redline = 7500;
	rpmCfg["FXO"].firstLeds = 4500;
	rpmCfg["FXO"].redline = 7500;
	rpmCfg["XRT"].firstLeds = 4500;
	rpmCfg["XRT"].redline = 7500;
	rpmCfg["RAC"].firstLeds = 4000;
	rpmCfg["RAC"].redline = 7000;
	rpmCfg["FZ5"].firstLeds = 5000;
	rpmCfg["FZ5"].redline = 8000;
	rpmCfg["UFR"].firstLeds = 6000;
	rpmCfg["UFR"].redline = 9000;
	rpmCfg["XFR"].firstLeds = 5000;
	rpmCfg["XFR"].redline = 9000;
	rpmCfg["FXR"].firstLeds = 4500;
	rpmCfg["FXR"].redline = 7500;
	rpmCfg["XRR"].firstLeds = 4500;
	rpmCfg["XRR"].redline = 7500;
	rpmCfg["FZR"].firstLeds = 5500;
	rpmCfg["FZR"].redline = 8500;
	rpmCfg["MRT"].firstLeds = 7000;
	rpmCfg["MRT"].redline = 10000;
	rpmCfg["FBM"].firstLeds = 6000;
	rpmCfg["FBM"].redline = 9000;
	rpmCfg["FOX"].firstLeds = 4500;
	rpmCfg["FOX"].redline = 7500;
	rpmCfg["FO8"].firstLeds = 6500;
	rpmCfg["FO8"].redline = 9500;
	rpmCfg["BF1"].firstLeds = 14000;
	rpmCfg["BF1"].redline = 20500;

	debugCfg = false;
}

ConfigLoader::~ConfigLoader(void)
{
}

/** Opens "g27leds.cfg" file and
	tries to read config from it.
	If a valid config is found, default
	configuration is overwritten.
	*/
HRESULT ConfigLoader::ReadConfig(std::ofstream* dbg)
{
	//Open the file for reading
	std::ifstream in("./g27leds.cfg", std::ios_base::in);

	//Could we open the file?
	if(!in)
	{
		//We couldn't, tell the user and exit
		*dbg << "! ConfigLoader:: Cannot open config file, using defaults." << std::endl;
		return E_FAIL;
	}

	std::vector<std::string> rawCfgData;
	std::string cfgLine;

	while(!in.eof())
	{
		std::getline(in, cfgLine);
		rawCfgData.push_back(cfgLine);
		cfgLine = "";
	}

	return ParseConfig(rawCfgData);
}

/** Parses lines read from the config
	file and if a valid config entry is found
	alters default configuration accordingly.
	*/
HRESULT ConfigLoader::ParseConfig(const std::vector<std::string>& rawCfgData)
{
	std::string temp;
	int delimPos;

	//Iterate through all read lines
	for (unsigned int i = 0; i<rawCfgData.size(); i++)
	{
		//Eliminate leading white spaces if any
		delimPos = rawCfgData[i].find_first_not_of(" ");
		if(delimPos != std::string::npos)
			temp = rawCfgData[i].substr(delimPos);
		else
			//Line is empty, skip it 
			continue;
		
		//Convert line to upper case
		transform(temp.begin(), temp.end(), temp.begin(), std::ptr_fun(::toupper));

		//Is the line is a commentary
		if (temp.compare(0, 1, commentChar) == 0) {
			//Line is a commentary, skip it
			continue;
		}
		//Is the line a "debug" directive?
		if (temp.compare(0, 5, debugDirective) == 0) {
			delimPos = temp.find(" ");
			temp = temp.substr(delimPos + 1);

			int doDebug = 0;
			if (NumericFromString<int>(doDebug, temp)) {
				if (doDebug > 0)
					debugCfg = true;
			}
		}
		//Is the line RPM config?
		else if (temp.compare(0, 4, rpmPrefix) == 0)
		{
			temp = temp.substr(4);
			std::string carID, firstLeds, redline;
			
			//Separate cartype from the RPM threshold values
			delimPos = temp.find(" ");
			carID = temp.substr(0, delimPos);

			//Separate and read "firstLeds" and "redline"
			temp = temp.substr(delimPos + 1);
			delimPos = temp.find(" ");
			firstLeds = temp.substr(0, delimPos);
			redline = temp.substr(delimPos + 1);

			float firstLedsF, redlineF;
			if(!NumericFromString<float>(firstLedsF, firstLeds)) {
				//Malformed line, skip it
				continue;
			}
			if(!NumericFromString<float>(redlineF, redline)) {
				//Malformed lide, skip it
				continue;
			}

			rpmCfg[carID].firstLeds = firstLedsF;
			rpmCfg[carID].redline = redlineF;
		}
	}

	return S_OK;
}