#include "outsslconnection.h"
#include <iostream>
#include <memory>
#include <sstream>

OutSSLConnection::OutSSLConnection(const std::string& address, const uint16_t port) :
  SSLConnection(address, port)
{
}

/**
  Creates an "address:port" formatted string from
  m_address and m_port and returns it as char*
  wrapped in shared_ptr.
  
  @param[in] p shared_ptr to wrap it resulting string in
  @return Returns true if the string is created OK, on failure returns false
  */
bool OutSSLConnection::addrPortString(std::shared_ptr<char>& p)
{
  std::string s(m_address);
  /* Convert port from numerical to string representation */
  std::stringstream out;
  out << m_port;
  /* Create "address:port" formatted string */
  s = s + ":" + out.str();
  
  int toAlloc = s.length() + 1; //Don't forget the null terminator!
  try {
    p.reset(new char[toAlloc], std::default_delete<char[]>());
  } catch (std::bad_alloc ex) {
    std::cerr << "Cannot allocate memory for addrPortString!" << std::endl;
    return false;
  }

  memset(p.get(), 0, toAlloc);
  strcpy(p.get(), s.c_str());
  return true;
}

/**
  Initializes the BIO socket and opens connection.
  
  @return Returns true on success, false otherwise.
  */
std::shared_ptr<OutSSLConnection> OutSSLConnection::open(const std::string& addr, const uint16_t port)
{
  std::shared_ptr<OutSSLConnection> me(new OutSSLConnection(addr, port));
  
  std::shared_ptr<char> aps;
  if (!me->addrPortString(aps)) {
    return false;
  }
  me->m_BIO = BIO_new_connect(aps.get());
  if (me->m_BIO == nullptr) {
    std::cerr << "Could not create SSL connection!" << std::endl;
    ERR_print_errors_fp(stderr);
    return false;
  }
  
  std::cerr << "Host:" << BIO_get_conn_hostname(me->m_BIO) << std::endl;
  
  /* Open connection */
  int ret = BIO_do_connect(me->m_BIO);
  if (ret <= 0) {
    std::cerr << "Could not open SSL connection!" << std::endl;
    return nullptr;
  }
  
  return me;
}
