#include "insslconnection.h"
#include <iostream>
#include <memory>
#include <sstream>

InSSLConnection::InSSLConnection(const uint16_t port) :
  m_canRun(true),
  SSLConnection(std::string(""), port)
{
}

/**
  Starts the receving loop on a separate thread.
  */
void InSSLConnection::listen()
{
  m_recvWorker = std::thread{&InSSLConnection::recvLoop, this};
  std::cout << "Listeting on port " << m_port << std::endl;
}

/**
  Initializes the BIO socket and opens connection.
  
  @return Returns true on success, false otherwise.
  */
std::shared_ptr<InSSLConnection> InSSLConnection::open(const uint16_t port)
{
  std::shared_ptr<InSSLConnection> me(new InSSLConnection(port));
  
  /* Convert port from numerical to string representation and make it char* */
  std::stringstream out;
  out << me->m_port;
  int toAlloc = out.str().length() + 1;
  std::shared_ptr<char> ps(new char[toAlloc], std::default_delete<char[]>());
  memset(ps.get(), 0, toAlloc);
  strcpy(ps.get(), out.str().c_str());
  
  me->m_BIO = BIO_new_accept(ps.get());
  if (me->m_BIO == nullptr) {
    std::cerr << "Could not create a socket to listen on" << std::endl;
    return nullptr;
  }
  
  int ret = BIO_do_accept(me->m_BIO);
  if (ret <= 0) {
    std::cerr << "Could not open SSL connection!" << std::endl;
    return nullptr;
  }
  
  return me;
}

/**
  Reads pending data on the socket.
  
  @param[in] bio BIO socket to read from
  @param[in,out] incoming Received data stored as std::string
  @return On success returns the number of bytes read, 0 or -1 on failure.
  */
int InSSLConnection::readIncoming(BIO* bio, std::string& incoming)
{
   if (bio == nullptr) {
    std::cerr << "readIncoming: Socket not OK!" << std::endl;
    return -1;
  }
  char* recvbuf = new char[512];
  
  int ret = BIO_read(bio, recvbuf, 512);
  if (ret <= 0) {
    std::cerr << "Error reading data from socket!" << std::endl;
    delete[] recvbuf;
    return 0;
  }
  
  std::cout << "Received stream: " << recvbuf << std::endl;
  incoming = recvbuf;
  delete[] recvbuf;
  
  return ret;
}

/**
  Receiving loop.
  */
void InSSLConnection::recvLoop()
{
  std::cout << "Entering InSSLConnection receive loop" << std::endl;
  int biofd;
  fd_set readfd, exceptfd;
  struct timespec sel_timeout;
  BIO* cbio;
  
  biofd = BIO_get_fd(m_BIO, &biofd);
  if (biofd <= 0) {
    std::cerr << "Cannot get InSSLConnection socket descriptor" << std::endl;
    return;
  }
  sel_timeout.tv_sec = 1;
  sel_timeout.tv_nsec = 1000;
  
  while (m_canRun) {
    /* Wait for incoming data */
    FD_ZERO(&readfd);
    FD_ZERO(&exceptfd);
    FD_SET(biofd, &readfd);
    FD_SET(biofd, &exceptfd);

    int ret = pselect(biofd + 1, &readfd, NULL, &exceptfd, &sel_timeout, NULL);
    if (ret == 0) // Timeout
      continue;
    else if (ret < 0) {
      std::cout << "Error reading from InSSLConnection" << std::endl;
      break;
    }
    if (FD_ISSET(biofd, &exceptfd)) {
      std::cout << "BIO socket exception in InSSLConnection" << std::endl;
      break;
    }
    
    /* We have data pending on the BIO socket */
    if (BIO_do_accept(m_BIO) <= 0) {
      std::cerr << "Error accepting incoming connection" << std::endl;
      m_canRun = false;
      return;
    }
    
    /* Read the data */
    cbio = BIO_pop(m_BIO);
    std::string incoming;
    readIncoming(cbio, incoming);
    
    std::cout << "Incoming ban info: " << incoming << std::endl;
    
    BIO_free(cbio);
  }
  
  std::cout << "Exiting InSSLConnection receive loop" << std::endl;
}

InSSLConnection::~InSSLConnection()
{
  if (m_recvWorker.joinable()) {
    m_canRun = false;
    m_recvWorker.join();
  }
}