#include "bankeeper.h"

BanKeeper::BanKeeper() :
  m_bandb(new BanDB),
  m_exchanger(m_bandb),
  m_localStorage(m_bandb)
{
}

/**
  Propagate information of a player being banned
  to all BanSystem modules.
  
  @param[in] uname Banned username
  @return True if the information was successfully propagated, false otherwise
  */
bool BanKeeper::banPlayer(std::string& uname)
{
  if (!m_bandb->addBan(uname)) {
    return false;
  }
  
  m_exchanger.announceBan(uname);
  
  return true;
}

/**
  Query local database and check if the given
  username is banned.
  
  @param[in] uname Username to be checked
  @return True if the username is banned, false otherwise
  */
bool BanKeeper::isBanned(std::string& uname)
{
  return m_bandb->isBanned(uname);
}

/**
  Propagate information of a ban being lifted to
  all BanSystem modules.
  
  @param[in] uname Username that was unbanned
  @return True if the information was sucessfully propagated, false otherwise
  */
bool BanKeeper::unbanPlayer(std::string& uname)
{
  return m_bandb->removeBan(uname);
}

/**
  Initialize all BanSystem modules.
  
  @param[in] lPort Port on which the Exchange system will listen
  @return True if all modules were initialized sucessfully, false otherwise
  */
bool BanKeeper::initialize(uint16_t lPort)
{
  if (!m_localStorage.loadBans())
    return false;
  
  m_exchanger.initialize(lPort);
}

/**
  Shutdown BanSystem.
  
  @return True on successful shutdown, false otherwise
  */
bool BanKeeper::shutdown()
{
  m_localStorage.saveBans();
}