#include "banexchanger.h"
#include <fstream>
#include <iostream>
#include <sstream>

BanExchanger::BanExchanger(std::shared_ptr<BanDB> p) :
  m_bandb(p)
{
}

/**
  Announces to all peers that a new user has been banned on this server
  
  @param[in] uname Username of the banned user
  @return True on success, false on failure
  */
bool BanExchanger::announceBan(std::string& uname)
{
  std::vector<std::shared_ptr<OutSSLConnection>>::iterator it;
  for (it = m_connections.begin(); it != m_connections.end(); it++) {
    (*it)->send(uname.c_str(), uname.length());
  }
  
  return true;
}

/**
  Loads a list of all peers we want to exchange bans with.
  
  @return True if the list was read successfully, false otherwise
  */
bool BanExchanger::loadPeers()
{
  std::ifstream peersList;
  peersList.open("peers.txt");
  if (!peersList.is_open()) {
    std::cerr << "Cannot open list of peers" << std::endl;
    return false;
  }
  
  while (peersList.good()) {
    std::string line;
    std::getline(peersList, line);
    
    int idx = line.find(":");
    if (idx == std::string::npos)	// Invalid line
      continue;
    std::string addr = line.substr(0, idx);
    std::string sPort = line.substr(idx + 1);
    uint16_t port = strToNum<uint16_t>(sPort);
    Peer p;
    p.address = addr;
    p.port = port;
    
    m_peers.push_back(p);
    
    std::cout << "Loaded peer " << p.address << ":" << p.port << std::endl;
  }
  
  return true;
}

/**
  Initialize Exchange system.
  
  @return True on success, false otherwise
  */
bool BanExchanger::initialize(uint16_t port)
{
  loadPeers();
  
  m_listening = SSLSpawner::getSpawner().newListen(port);
  if (m_listening == nullptr) {
    std::cerr << "Cannot open socket to listen on" << std::endl;
    return false;
  }
  m_listening->listen();
  
  std::vector<Peer>::iterator it;
  for (it = m_peers.begin(); it != m_peers.end(); it++) {
    std::shared_ptr<OutSSLConnection> conn = SSLSpawner::getSpawner().newRemote(it->address, it->port);
    if (conn == nullptr) {
      std::cout << "Unable to create outgoing connection" << std::endl;
      continue;
    }
    m_connections.push_back(conn);
  }
  
  return true;
}

/**
  Helper function: converts std::string to numerical type.
  
  @param[in] str String to be converted to number
  @return Number
  */
template <typename T> T BanExchanger::strToNum(const std::string& str)
{
  std::stringstream ss(str);
  T result;
  if (!(ss >> result))
    return 0;
  
  return result;
}