﻿# -*- coding: utf_8 -*-

bl_info = {
	'name': 'Import Live for Speed car (.cmx) Blender 2.62 compatible',
	'author': 'Bogey Jammer',
	'version': (1, 0),
	'blender': (2, 6, 2),
	'location': 'File > Import > Live for Speed car (.cmx) Blender 2.62',
	'description': 'Import Live for Speed car',
	'warning': '',
	'category': 'Import-Export'}
	
import bpy
from bpy.props import *

import struct
import os.path

__author__ = 'Bogey Jammer'
__copyright__ = '© Bogey Jammer'
__credits__ = ['Bogey Jammer', 'Scawen Roberts', 'Eric Bailey', 'Victor Van Vlaardingen']
__version__ = '07/07/2012'
__status__ = 'Prototype'


class Graphics_creation:
	"""Create Blender object, mesh, points, bla bla…"""
	
	def __init__(self, object_blocks_list, object_blocks_number, car_name, create_materials, apply_vertices_colors, materials_creator):
		# Variable initialization
		short_car_name = car_name[:3]
		self.create_materials = create_materials
		self.apply_vertices_colors = apply_vertices_colors
		printout_current = 0
		
		for object in object_blocks_list:
			# Add a mesh
			self.the_mesh = bpy.data.meshes.new('{0}_mesh'.format(short_car_name))
			
			# Load materials
			if create_materials == True:
				self._load_materials(
								textureInfo_blocks_list=object.get_textureInfo_blocks_list(),
								materials_creator=materials_creator
								)
			
			# Vertices creation
			self._build_vertices(
							points_number=object.get_points_number(),
							point_blocks_list=object.get_point_blocks_list()
							)
			
			# Faces creation
			self._build_faces(
							triangles_number=object.get_triangles_number(),
							triangle_blocks_list=object.get_triangle_blocks_list(),
							point_blocks_list=object.get_point_blocks_list()
							)
							
			# Material assignation
			if create_materials == True:
				self._apply_materials()
			
			self.the_mesh.update_tag()
			
			# Create the object
			the_objet = bpy.data.objects.new(str(short_car_name), self.the_mesh)
			bpy.context.scene.objects.link(the_objet)
			
			printout_current += 1
			print('    mesh {0}/{1} created.'.format(printout_current, object_blocks_number))
			
	# Private methods ---------------------------------------------------------		
	def _load_materials(self, textureInfo_blocks_list, materials_creator):
		"""Load materials used by the object into the mesh' slots"""
		# List for helping application of materials on faces
		# Format for each material: [(material_index, triangles_number), …]
		self.mat_dependencies = []
		
		# Material loading
		for textureInfo in textureInfo_blocks_list:
			material_index = textureInfo.get_material_index()
			self.the_mesh.materials.append(materials_creator.get_material(material_index))
			
			self.mat_dependencies.append(textureInfo.get_triangles_number())
	
	def _build_vertices(self, points_number, point_blocks_list):
		"""Build current object's vertices only"""
		self.the_mesh.vertices.add(points_number)
		vertices_co = []
		normals = []
		
		self.vertices_definition_list = []
		for point in point_blocks_list:
			coordinates = point.get_coordinates()
			normals_tup = point.get_normals()
			
			vertices_co.append(coordinates)
			normals.append(normals_tup)
			# Construction of the list of vertices' coordinates and normals for later comparison
			self.vertices_definition_list.append((coordinates, normals_tup))
			
		for index, vertex in enumerate(self.the_mesh.vertices):
			vertex.co = vertices_co[index]
			vertex.normal = normals[index]
		
	def _build_faces(self, triangles_number, triangle_blocks_list, point_blocks_list):
		"""Build current object's faces and process them"""
		self.the_mesh.faces.add(triangles_number)
		faces = []
						
		for triangle in triangle_blocks_list:
			# Comparison of vertices' normals and reassignation of vertices index if duplication is detected
			correct_vertices_indexes = []
			for vertice_index in triangle.get_vertices_indexes():
				correct_vertices_indexes.append(self.vertices_definition_list.index((
																		self.vertices_definition_list[vertice_index][0],
																		self.vertices_definition_list[vertice_index][1]))
																		)
				
			faces.extend(correct_vertices_indexes)
			#faces.extend(triangle.get_vertices_indexes())
		
		
		# Integration of faces in the self.the_mesh
		self.the_mesh.faces.foreach_set('vertices_raw', faces)
		
		# Enable faces smoothing
		self.the_mesh.faces.foreach_set('use_smooth', [True] * len(self.the_mesh.faces))
				
		# UV mapping creation and coloration of vertices
		uvMap = self.the_mesh.uv_textures.new(name='LFS_UV_map')
		color_layer = self.the_mesh.vertex_colors.new('LFS_vertices_colors')
		for index in range(len(self.the_mesh.faces)):
			UVtriangle = uvMap.data[index]
			# Ask each triangle block for the point index number, should match created vertices' number in Blender
			indexA = triangle_blocks_list[index].get_point_index('A')
			indexB = triangle_blocks_list[index].get_point_index('B')
			indexC = triangle_blocks_list[index].get_point_index('C')
			
			UVtriangle.uv1 = point_blocks_list[indexA].get_pointUV()
			UVtriangle.uv2 = point_blocks_list[indexB].get_pointUV()
			UVtriangle.uv3 = point_blocks_list[indexC].get_pointUV()
			
			if self.apply_vertices_colors == True:
				colTriangle = color_layer.data[index]
				colTriangle.color1 = point_blocks_list[indexA].get_pointColor()
				colTriangle.color2 = point_blocks_list[indexB].get_pointColor()
				colTriangle.color3 = point_blocks_list[indexC].get_pointColor()
			
		self.the_mesh.update(calc_edges=True)
		
	def _apply_materials(self):
		"""Apply material slot on object's faces"""
		processed_faces = 0
		slot_material_index = 0
		for face in self.the_mesh.faces:
			# Increment the material slot index when no more processed faces require it
			if processed_faces >= self.mat_dependencies[slot_material_index]:
				slot_material_index += 1				
				processed_faces = 0
				
			# Application of the material
			face.material_index = slot_material_index
			processed_faces += 1
			
	
class Materials_creation:
	"""Load texture files and create materials in Blender"""
	
	def __init__(self, texture_blocks_list, fileManipulator, texture_blocks_number):
		self.materials_list = []
		printout_current = 0
		
		for texture in texture_blocks_list:
			# Get data from the texture block
			materialName = texture.get_name()
			fileName = texture.get_filename()
			transparent = texture.get_transparent()
			shiny = texture.get_shiny()
			
			# Load image file in Blender
			if materialName != 'CUSTOM SKIN' and materialName != '':
				image_blendData = bpy.data.images.load(filepath=fileManipulator.get_file_path() + '\\..\\dds\\' + fileName)
				
			# Create Blender texture datablock
			if materialName != '':
				texture_blendData = bpy.data.textures.new(materialName, 'IMAGE')
				if materialName != 'CUSTOM SKIN':
					texture_blendData.image = image_blendData
				if transparent == False:
					texture_blendData.use_alpha = False
					texture_blendData.use_preview_alpha = False
				else:
					texture_blendData.use_alpha = True
					texture_blendData.use_preview_alpha = True
				texture_blendData.use_mipmap = False
				
			# Create Blender material
			material = bpy.data.materials.new(materialName)
			
			if materialName != '':
				texture_slot = material.texture_slots.add()
				texture_slot.texture = texture_blendData
				texture_slot.texture_coords = 'UV'
				texture_slot.uv_layer = 'LFS_UV_map'
				texture_slot.blend_type = 'MULTIPLY'
			
			material.diffuse_intensity = 1
			material.specular_shader = 'PHONG'
			material.specular_intensity = 0
			material.use_cubic = True
			material.use_vertex_color_paint = True
			if shiny == True:
				material.raytrace_mirror.use = True
				material.raytrace_mirror.reflect_factor = 0.05
				
			if transparent == 'ALPHA':
				if texture_blendData.name == 's_glass':
					# Deactivation of the useless glass texture with a quite stupid method
					material.use_textures = (False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False, False)
				material.use_transparency = True
				material.transparency_method = 'RAYTRACE'
				material.alpha = 0.7
				material.raytrace_mirror.reflect_factor = 0.05
			elif transparent == 'ALP24':
				material.use_transparency = True
				material.alpha = 0
				# Add a second texture slot to get the transparent areas
				texture_slot2 = material.texture_slots.add()
				texture_slot2.texture = texture_blendData
				texture_slot2.texture_coords = 'UV'
				texture_slot2.uv_layer = 'LFS_UV_map'
				texture_slot2.use_map_color_diffuse = False
				texture_slot2.use_map_alpha = True
				
			self.materials_list.append(material)
			
			printout_current += 1
			print('    material {0}/{1} created.'.format(printout_current, texture_blocks_number))
			
	# Getters -----------------------------------------------------------------
	
	def get_material(self, index):
		return self.materials_list[index]


class Triangle_block:

	def __init__(self, fileManipulator, startindex):
		"""Read a triangle block and store its data"""
		# Triangle's vertices' indexes
		self.A = fileManipulator.readFile(type='H', seek=startindex)
		self.B = fileManipulator.readFile(type='H')
		self.C = fileManipulator.readFile(type='H')
		
	# Getters -----------------------------------------------------------------
	
	def get_vertices_indexes(self):
		return [self.A, self.B, self.C, 0]
		
	def get_point_index(self, vertex):
		if vertex == 'A':
			return self.A
		elif vertex == 'B':
			return self.B
		elif vertex == 'C':
			return self.C
	
		
class Point_block:

	def __init__(self, fileManipulator, startindex):
		"""Read a point block and store its data"""
		# Coordinates with V coordinate inverted for correction
		self.X = fileManipulator.readFile(type='i', seek=startindex) / 65536
		self.Y = fileManipulator.readFile(type='i') / 65536
		self.Z = fileManipulator.readFile(type='i')	/ 65536
		self.NX = fileManipulator.readFile(type='f')
		self.NY = fileManipulator.readFile(type='f')
		self.NZ = fileManipulator.readFile(type='f')
		self.U = fileManipulator.readFile(type='f')
		self.V = 1 - fileManipulator.readFile(type='f')
		self.color = fileManipulator.readFile(type='4B')
		
	# Getters -----------------------------------------------------------------
	
	def get_coordinates(self):
		return (self.X, self.Y, self.Z)
		
	def get_normals(self):
		return (self.NX, self.NY, self.NZ)
		
	def get_pointUV(self):
		return (self.U, self.V)
		
	def get_pointColor(self):
		return (self.color[2] / 255, self.color[1] / 255, self.color[0] / 255)
		
		
class TextureInfo_block:

	def __init__(self, startindex, fileManipulator):
		"""Read a texture info block and store its data"""
		# Material index
		self.texture_index = fileManipulator.readFile(type='B', seek=startindex+2)
		
		# Number of points and triangles for this texture
		self.points = fileManipulator.readFile(type='H', seek=startindex+4)
		self.triangles = fileManipulator.readFile(type='H')
		
	# Getters -----------------------------------------------------------------
	
	def get_material_index(self):
		return self.texture_index
		
	def get_triangles_number(self):
		return self.triangles
				
		
class Object_block:

	def __init__(self, startindex, fileManipulator, create_materials):
		"""Read an object block and store its data"""
		# Variable and constants initialization
		self.create_materials = create_materials
		self.BLOCKSIZE_OBJECT = 28
		self.BLOCKSIZE_TEXTUREINFO = 8
		self.BLOCKSIZE_POINT = 36
		self.BLOCKSIZE_TRIANGLE = 8
		
		# Number of sub-blocks
		self.textureInfo_blocks_number = fileManipulator.readFile(type='H', seek=startindex+20)
		self.point_blocks_number = fileManipulator.readFile(type='H')
		self.triangle_blocks_number = fileManipulator.readFile(type='H')
		
		# Texture Info blocks reading
		if self.create_materials == True:
			self.textureInfo_blocks_list = []
			subStartindex = startindex + 28
			for index in range(self.textureInfo_blocks_number):
				self.textureInfo_blocks_list.append(TextureInfo_block(fileManipulator=fileManipulator, startindex=subStartindex))
				subStartindex += self.BLOCKSIZE_TEXTUREINFO
		else:
			subStartindex = startindex + 28 + self.BLOCKSIZE_TEXTUREINFO * self.textureInfo_blocks_number
				
		# Point blocks reading
		self.point_blocks_list = []
		for index in range(self.point_blocks_number):
			self.point_blocks_list.append(Point_block(fileManipulator=fileManipulator, startindex=subStartindex))
			subStartindex += self.BLOCKSIZE_POINT
			
		# Triangle blocks reading
		self.triangle_blocks_list = []
		for index in range(self.triangle_blocks_number):
			self.triangle_blocks_list.append(Triangle_block(fileManipulator=fileManipulator, startindex=subStartindex))
			subStartindex += self.BLOCKSIZE_TRIANGLE
			
	# Getters -----------------------------------------------------------------
	
	def get_block_size(self):
		"""Return the size of the current object block"""
		return self.BLOCKSIZE_OBJECT + self.textureInfo_blocks_number * self.BLOCKSIZE_TEXTUREINFO + \
			self.point_blocks_number * self.BLOCKSIZE_POINT + \
			self.triangle_blocks_number * self.BLOCKSIZE_TRIANGLE
			
	def get_textureInfo_blocks_list(self):
		return self.textureInfo_blocks_list
			
	def get_points_number(self):
		return self.point_blocks_number
		
	def get_point_blocks_list(self):
		return self.point_blocks_list
		
	def get_triangles_number(self):
		return self.triangle_blocks_number
		
	def get_triangle_blocks_list(self):
		return self.triangle_blocks_list
		
	def get_textureInfo_blocks_list(self):
		return self.textureInfo_blocks_list
				
		
class Texture_block:

	def __init__(self, fileManipulator):
		"""Read a texture block and store its data"""
		# Variables initialization
		self.fileManipulator = fileManipulator
		
		# Texture name
		self.name = self.fileManipulator.readText(length=16)
		self.filename = self.name + '.dds'
		suffix = self.name[-5:]
		if suffix == 'ALPHA' or suffix == 'ALP24':
			self.transparent = suffix
			self.name = self.name[:-5]
		else:
			self.transparent = False
			
		# Reflectivity of the material
		flags = self.fileManipulator.readFile(type='i')
		if flags / 128 >= 1:
			self.name = 'CUSTOM SKIN'
			flags -= 128
		if flags / 64 >= 1:
			self.shiny = True
		else:
			self.shiny = False
			
	# Getters -------------------------------------------------------------
	
	def get_name(self):
		return self.name
		
	def get_filename(self):
		return self.filename
		
	def get_transparent(self):
		return self.transparent
		
	def get_shiny(self):
		return self.shiny
			
		
class CMX_fileManipulator:

	def __init__(self, filename):
		self.filename = filename
		self.cmx_file = open(self.filename, 'rb')
		
	# Public methods ----------------------------------------------------------
	
	def checkFile(self):
		"""Check and tell if the CMX file is valid"""
		print('Checking if it is a real .cmx file...')
		if self.cmx_file.read(6).decode() == 'LFSCMX':
			print('    The file is ok.\n\nChecking the CMX version...')
			self.cmx_file.seek(8)
			if struct.unpack('B', self.cmx_file.read(1))[0] == 0:
				print('    The CMX version is ok.\n\nChecking the data type...')
				if struct.unpack('B', self.cmx_file.read(1))[0] == 1:
					print('    OK, the data is a car.')
					return True
				else:
					print('    This is not data for a car !')
					return False
			else:
				print('    This CMX version is not supported !')
				return False
		else:
			print('    This is not a CMX file !')
			return False
		
	def readText(self, length, seek=None):
		"""Read a text from the file"""
		if seek != None:
			self.cmx_file.seek(seek)
		text = self.cmx_file.read(length).decode()

		# Remove the x00 caracters at the end of the text
		final = ''
		for letter in text:
			if letter != '\x00':
				final += letter
		return final
	
	def readFile(self, type, seek=None):
		"""Read a value from the file"""
		if seek != None:
			self.cmx_file.seek(seek)
		data = struct.unpack(type, self.cmx_file.read(struct.calcsize(type)))
		
		if len(data) == 1:
			return data[0]
		else:
			return data
			
	# Getters -----------------------------------------------------------------
	
	def get_file_path(self):
		return os.path.dirname(self.filename)
		
	# Setters -----------------------------------------------------------------
	
	def set_seekerIndex(self, seek):
		self.cmx_file.seek(seek)
		

class CmxImport:
	"""At least this is the core of .cmx importation"""
	
	def __init__(self, filename, create_materials, apply_vertices_colors):
		"""Variables and constants initialization, startup of importation"""
		self.create_materials = create_materials
		self.apply_vertices_colors = apply_vertices_colors
		self.BLOCKSIZE_HEADER = 84
		self.BLOCKSIZE_LIGHTSCHEME = 72
		self.BLOCKSIZE_TEXTURE = 20
		
		print('\n*** BEGINNING OF CMX IMPORTATION ***\n')
		
		self.cmxFileManipulator = CMX_fileManipulator(filename)
		
		# Check for the CMX file validity
		if self.cmxFileManipulator.checkFile() == False:
			print('Importation aborted.')
		else:
			self._read()
	
	# Private methods ---------------------------------------------------------
	
	def _read(self):
		"""Read CMX file and store needed values for importation"""
		print('\nReading CMX file...')
		
		# Read header block
		self.car_name = self.cmxFileManipulator.readText(seek=16, length=32)
		print('\nThe car about to be imported is: ' + self.car_name)
		
		# Number of blocks
		self.lightscheme_blocks_number = self.cmxFileManipulator.readFile(seek=68, type='i')
		self.texture_blocks_number = self.cmxFileManipulator.readFile(seek=76, type='i')
		self.object_blocks_number = self.cmxFileManipulator.readFile(type='i')
		
		# Texture blocks reading
		if self.create_materials == True:
			self.texture_blocks_list = []
			startindex = self.BLOCKSIZE_HEADER + self.lightscheme_blocks_number * self.BLOCKSIZE_LIGHTSCHEME
			self.cmxFileManipulator.set_seekerIndex(startindex)
			for index in range(self.texture_blocks_number):
				self.texture_blocks_list.append(Texture_block(fileManipulator=self.cmxFileManipulator))
				startindex += self.BLOCKSIZE_TEXTURE
		else:
			startindex = self.BLOCKSIZE_HEADER + self.lightscheme_blocks_number * self.BLOCKSIZE_LIGHTSCHEME + \
					self.BLOCKSIZE_TEXTURE * self.texture_blocks_number
				
		# Object blocks reading
		self.object_blocks_list = []
		for index in range(self.object_blocks_number):
			object_block = Object_block(
									startindex=startindex,
									fileManipulator=self.cmxFileManipulator,
									create_materials=self.create_materials,
									)
			self.object_blocks_list.append(object_block)
			startindex += object_block.get_block_size()
			
		# If nothing crashed, then build the model
		self._build()
		
	def _build(self):
		"""Create and process Blender entities to make the model"""
		# Materials creation
		if self.create_materials == True:
			print('\nBuilding materials...')
			self.materials_creation = Materials_creation(
													texture_blocks_list=self.texture_blocks_list,
													fileManipulator=self.cmxFileManipulator,
													texture_blocks_number=self.texture_blocks_number)
		else:
			# Antibug for allowing next argument passing even if no materials are to be created
			self.materials_creation = None
			
		# Object creation
		print('\nBuilding meshes...')
		self.graphics_creation = Graphics_creation(
											object_blocks_list=self.object_blocks_list,
											object_blocks_number=self.object_blocks_number,
											car_name=self.car_name,
											create_materials=self.create_materials,
											apply_vertices_colors=self.apply_vertices_colors,
											materials_creator=self.materials_creation
											)
		
		print('\n*** IMPORTATION SUCCEEDED ! ***')
		
		
def getInputFilename(self, filename, create_materials, apply_vertices_colors):
	"""Semi-conventional Blender fonction for import processing startup"""
	checktype = filename.split('\\')[-1].split('.')[1]
	if checktype.lower() != 'cmx':
		print ('Selected file = ',filename)
		raise (IOError, 'The selected input file is not a *.cmx file')
	else:
		# Launch importation
		cmxImport = CmxImport(filename, create_materials, apply_vertices_colors)


class IMPORT_OT_cmx(bpy.types.Operator):
	"""Blender conventional class defining the file picker dialog"""
	
	bl_idname = 'import_scene.cmx'
	bl_label = 'Import CMX'
	bl_space_type = 'PROPERTIES'
	bl_region_type = 'WINDOW'
	bl_options = {'UNDO'}
	
	filepath = StringProperty(
			name='File Path',
			description='Filepath used for importing the cmx file',
			maxlen= 1024,
			subtype='FILE_PATH',
			)
	filter_glob = StringProperty(
			default='*.cmx',
			options={'HIDDEN'},
			)
	create_materials = BoolProperty(
            name="Create materials",
            description="Create and assign LFS-like materials with their textures on the imported model.",
            default=True,
            )
	apply_vertices_colors = BoolProperty(
            name="Apply vertices colors",
            description="Import and apply colors on model's vertices",
            default=True,
            )
			
	def execute(self, context):
		getInputFilename(self, self.filepath, self.create_materials, self.apply_vertices_colors)
		return {'FINISHED'}

	def invoke(self, context, event):
		"""WTF is this method ?"""
		wm = context.window_manager
		wm.fileselect_add(self)
		return {'RUNNING_MODAL'}  


# Blender conventional functions ---------------------------------------------

def menu_func(self, context):
	"""Launches the file selection dialog"""
	self.layout.operator(IMPORT_OT_cmx.bl_idname, text='Live for Speed car (.cmx) Blender 2.62')
	
def register():
	bpy.utils.register_module(__name__)
	bpy.types.INFO_MT_file_import.append(menu_func)
	
def unregister():
	bpy.utils.unregister_module(__name__)
	bpy.types.INFO_MT_file_import.remove(menu_func)
	
if __name__ == '__main__':
	register()