#include <iostream>
#include "area.h"

/** Default constuctor */
Area::Area()
{
  _vertices.reserve(3);
}

/** Copy constructor */
Area::Area(const Area& a)
{
  _vertices = a._vertices;
}

/** Create area from a list of vertices */
Area::Area(std::vector<Point> const& vertices)
{
  _vertices = vertices;
}

/** Add a vertex to the end of the list */
void Area::add_vertex(Point const& p)
{
  _vertices.push_back(p);
}

/** Casts a ray from the tested point onwards
 *  along the X axis to determine if the point
 *  is inside or outside the area.
 *  Odd number of intersections means the point is inside the area.
 */
bool Area::contains(Point const& p)
{
  int pX = p.x();
  int pY = p.y();
  bool inside = false;
  
  /* Start with the segment made by last vertex (j = SIZE - 1) and first vertex (i = 0) of the polygon */
  for (unsigned int j = 0, i = _vertices.size() - 1; j < _vertices.size(); j++) {
    if (pY > _vertices[i].y() != pY > _vertices[j].y()) { /* Can the segment cross the ray vertically? If it cannot, skip it. */
      /* WARNING: Magic ahead.
       * If anybody wonders why this works, imagine that the formula below is nothing but an inverse linear function equation X = K*Y + Q
       * where K = dX / dY and Q is the X coordinate of the starting point of the segment (note that the segments are created backwards).
       * By solving this equation and comparing the result to pX we can tell whether the segment is "ahead of" or "behind" the tested point on the X axis
       * at a given pY coordinate. If it is behind, the casted ray does not intersect the segment.
       * (It's better to grab a piece of paper and visualise this whole solution) */
      float shift = ((float)(_vertices[i].x() - _vertices[j].x()) / (float)(_vertices[i].y() - _vertices[j].y())) * (pY - _vertices[j].y()) + _vertices[j].x();
      if (pX < shift)
		  inside = !inside;
    }
    i = j;
  }
  
  return inside;
}

/** Removes a vertex from the list */
bool Area::remove_vertex_at(unsigned int idx)
{
  if (_vertices.size() >= idx)
    return false;
  
  _vertices.erase(_vertices.begin() + idx);
  return true;
}

/* Modifies a vertex in the list */
void Area::set_vertex_at(Point const& p, unsigned int idx)
{
  _vertices[idx] = p;
}

/* Returns a vertex at given index */
Point Area::vertex_at(unsigned int idx)
{
  return _vertices[idx];
}

/* Returns the whole list of vertices */
std::vector<Point> Area::vertices() const
{
  return _vertices;
}