﻿using System;

namespace VectorMath {
    public class Vector2D {
        const double Pi = 3.14159265359;		/* 180 degrees in radians */
        const double HalfPi = 1.57079632679;		/*  90 degrees in radians */
        const double ThreeHalvesPi = 4.71238898038;	/* 270 degrees in radians */

        public double X { get; private set; }
        public double Y { get; private set; }

        public double Angle {
            get { return Math.Atan(Y / X); }
        }

        public double Magnitude {
            get { return Math.Sqrt(X * X + Y * Y); }
        }

        public Vector2D(double x, double y) {
            X = x;
            Y = y;
        }

        public Vector2D(double angle) {
            if (angle == HalfPi) {
                X = 0;
                Y = 1;
            }
            else if (angle == ThreeHalvesPi) {
                X = 0;
                Y = -1;
            }

            if (angle < HalfPi || angle > ThreeHalvesPi) {
                X = 1;
            }
            else {
                X = -1;
            }

            Y = Math.Tan(angle) * X;
        }

        public static double GetAngle(Vector2D vec1, Vector2D vec2) {
            NormalizeVector(vec1);
            NormalizeVector(vec2);
            return GetAngleNorm(vec1, vec2);
        }

        public static double GetAngleNorm(Vector2D vec1, Vector2D vec2) {
            double scalarMul = (vec1.X * vec2.X) + (vec1.Y * vec2.Y);
            return Math.Acos(scalarMul);
        }

        public static void NormalizeVector(Vector2D vec) {
            if (vec.X == 0) {
                vec.Y = 1;
                return;
            }

            if (vec.Y == 0) {
                vec.X = 1;
                return;
            }

            double tanAlpha = vec.Y / vec.X;
            /* Calculate normalized size of X component */
            double vxNorm = 1 / Math.Sqrt(Math.Pow(tanAlpha, 2) + 1);
            /* Calculate normalized size of Y component */
            double vyNorm = tanAlpha * vxNorm;

            /* Make sure the normalized vector has the same direction as the original one */
            if (!SameSign(vec.X, vxNorm)) {
                vxNorm *= -1;
            }

            if (!SameSign(vec.Y, vyNorm)) {
                vyNorm *= -1;
            }

            vec.X = vxNorm;
            vec.Y = vyNorm;
        }

        public static double DegToRad(double degs) {
            /** Converts degrees to radians */
            return degs * Pi / 180;
        }

        public static double RadToDeg(double rads) {
            /** Converts radians to degrees */
            return rads * 180 / Pi;
        }

        public static bool SameSign(double a, double b) {
            /** Checks if the passed values have the same sign */
            return Math.Sign(a) == Math.Sign(b);
        }
    }
}
