﻿using System;
using System.Linq;
using System.Xml.Linq;

namespace LfsPoints.Projects {
    public class XmlExporter : IExporter {
        private const int XmlVersion = 3;

        public void Export(string fileName, Project project) {
            XDocument xDocument = new XDocument();
            XElement xProject = new XElement("Championship");
            xProject.Add(new XAttribute("Name", project.Name));
            xProject.Add(new XAttribute("Version", XmlVersion));

            XElement xStandings = new XElement("Standings");
            foreach (Standing standing in project.Standings) {
                xStandings.Add(GetStandingXml(standing));
            }
            xProject.Add(xStandings);

            XElement xReplays = new XElement("Replays");
            foreach (Replay replay in project.Replays.OrderBy(r => r.StartTime.Value)) {
                xReplays.Add(GetReplayXml(replay));
            }
            xProject.Add(xReplays);

            xDocument.Add(xProject);
            xDocument.Save(fileName);
        }

        public void Export(string fileName, Replay replay) {
            XDocument xDocument = new XDocument();
            xDocument.Add(GetReplayXml(replay), new XAttribute("Version", XmlVersion));
            xDocument.Save(fileName);
        }

        private static XElement GetStandingXml(Standing standing) {
            XElement xStanding = new XElement("Standing");
            xStanding.Add(new XElement("Position", standing.Position));
            xStanding.Add(new XElement("UserName", standing.UserName));
            xStanding.Add(new XElement("Driver", standing.DisplayName));
            xStanding.Add(new XElement("Points", standing.Points));
            return xStanding;
        }

        private static XElement GetReplayXml(Replay replay) {
            XElement xReplay = new XElement("Replay");
            xReplay.Add(new XElement("Name", replay.Name));
            xReplay.Add(new XElement("Allocation", replay.Allocation.Name));
            xReplay.Add(new XElement("RaceMode", replay.RaceMode));
            xReplay.Add(new XElement("Track", replay.TrackName));
            xReplay.Add(new XElement("Laps", replay.Laps));
            xReplay.Add(new XElement("Hours", replay.Hours));

            TimeSpan timestamp = replay.StartTime.Value - new DateTime(1970, 1, 1);
            xReplay.Add(new XElement("Date", replay.StartTime.Value.ToString("g"), new XAttribute("TimeStamp", timestamp.TotalSeconds)));

            XElement xResults = new XElement("Results");
            foreach (Result result in replay.Results.OrderBy(r => r.Position)) {
                xResults.Add(GetResultXml(result));
            }
            xReplay.Add(xResults);
            return xReplay;
        }

        private static XElement GetResultXml(Result result) {
            XElement xResult = new XElement("Result");
            xResult.Add(new XElement("Position", result.Position));
            xResult.Add(new XElement("UserName", result.UserName));
            xResult.Add(new XElement("Driver", result.DisplayName));
            xResult.Add(new XElement("Car", result.CarName));
            xResult.Add(new XElement("Grid", result.StartPosition, new XAttribute("Climb", result.Climb)));
            xResult.Add(new XElement("Laps", result.LapsDone));
            xResult.Add(new XElement("Time", result.RaceTime.ToLapTimeString(), new XAttribute("Milliseconds", result.RaceTime.TotalMilliseconds)));
            xResult.Add(new XElement("ConfirmFlags", result.ConfirmFlags, new XAttribute("Flags", (int)result.ConfirmFlags)));
            xResult.Add(new XElement("Penalty", result.PenaltyTime.Value.TotalSeconds));
            xResult.Add(new XElement("BestLap", result.BestLapTime.ToLapTimeString(), new XAttribute("Milliseconds", result.BestLapTime.Value.TotalMilliseconds)));
            xResult.Add(new XElement("Points", result.Points));
            return xResult;
        }
    }
}
