﻿using System;
using System.Diagnostics;
using System.Xml.Serialization;

namespace LfsPoints.Projects {
    [Serializable]
    [DebuggerDisplay("PlayerName = {PlayerName}, LapsDone = {LapsDone}, RaceTime = {RaceTime}")]
    public class Result {
        public string ReplayId { get; set; }
        public string PlayerName { get; set; }
        public string UserName { get; set; }
        public string CarName { get; set; }
        public int LapsDone { get; set; }
        public ConfirmFlags ConfirmFlags { get; set; }
        public SerializableTimeSpan PenaltyTime { get; set; }
        public SerializableTimeSpan OverallTime { get; set; }
        public SerializableTimeSpan BestLapTime { get; set; }
        public int StartPosition { get; set; }

        [XmlIgnore]
        public string DisplayName {
            get { return Project.Current.GetDisplayName(UserName); }
        }

        [XmlIgnore]
        public Replay Replay { get; set; }

        [XmlIgnore]
        public int Position { get; set; }

        [XmlIgnore]
        public TimeSpan RaceTime {
            get { return OverallTime.Value + PenaltyTime.Value; }
        }

        [XmlIgnore]
        public int Climb {
            get { return StartPosition - Position; }
        }

        [XmlIgnore]
        public bool IsWinner { get; set; }

        [XmlIgnore]
        public bool IsPoleSitter { get; set; }

        [XmlIgnore]
        public bool IsFastestDriver { get; set; }

        [XmlIgnore]
        public bool IsHighestClimber { get; set; }

        [XmlIgnore]
        public bool IsDisqualified {
            get { return (ConfirmFlags & ConfirmFlags.Disqualified) > 0; }
        }

        [XmlIgnore]
        public bool IsQualifying {
            get { return Replay.RaceMode == RaceMode.Qualifying; }
        }

        [XmlIgnore]
        public double PercentCompleted { get; set; }

        [XmlIgnore]
        public int Points {
            get { return Replay.Allocation.GetTotalPoints(this); }
        }

        [XmlIgnore]
        public object Tag { get; set; }

        public Result() { }

        public static Result FromFile(LfsFileReader reader, Replay replay) {
            Result result = new Result();
            result.ReplayId = replay.ReplayId;
            result.Replay = replay;

            //24    char    0       player name         : text, ends 0, no colours
            result.PlayerName = reader.ReadLfsString(24);
            //8     char    24      number plate        : text, NOTE : NO ZERO AT END
            reader.Skip(8);
            //4     char    32      short car name      : text, ends 0
            result.CarName = reader.ReadString(4);
            //24    byte    36      lfs user name       : text, ends 0
            result.UserName = reader.ReadString(24);
            //1     word    60      laps done           : total laps completed
            result.LapsDone = reader.ReadUInt16();
            //1     word    62      player flags        : driver settings (see NOTES)
            reader.Skip(2);
            //1     byte    64      confirm flags       : penalties (see NOTES)
            result.ConfirmFlags = (ConfirmFlags)reader.ReadByte();
            //1     byte    65      number of stops     : pit stops count
            reader.Skip(1);
            //1     word    66      penalty seconds     : penalty time added
            result.PenaltyTime = new SerializableTimeSpan(reader.ReadSeconds());
            //1     int     68      overall time        : milliseconds
            result.OverallTime = new SerializableTimeSpan(reader.ReadMilliseconds() - result.PenaltyTime.Value);
            //1     int     72      best lap time       : milliseconds
            result.BestLapTime = new SerializableTimeSpan(reader.ReadMilliseconds());
            //1     byte    76      0                   : -
            reader.Skip(1);
            //1     byte    77      start position      : 0 = unknown, 1 = pole, etc.
            result.StartPosition = reader.ReadByte();
            //1     byte    78      handicap mass       : kg
            //1     byte    79      intake restriction  : %
            reader.Skip(2);

            Project.Current.SetDisplayName(result.UserName, result.PlayerName);

            return result;
        }
    }
}
