﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Diagnostics.CodeAnalysis;
using System.IO;
using System.Linq;
using System.Xml.Serialization;

namespace LfsPoints.Projects {
    [Serializable]
    [DebuggerDisplay("Name = {Name}, TrackName = {TrackName}, Results = {Results.Count}")]
    public class Replay {
        private Allocation _allocation;

        public string ReplayId { get; set; }
        public string AllocationId { get; set; }
        public string Name { get; set; }
        public string TrackName { get; set; }
        public int LapsByte { get; set; }
        public SerializableDateTime StartTime { get; set; }
        public List<Result> Results { get; set; }

        [XmlIgnore]
        public Allocation Allocation {
            get { return _allocation; }
            set {
                _allocation = value;
                AllocationId = _allocation.AllocationId;
            }
        }

        [XmlIgnore]
        public int Laps {
            get {
                if (LapsByte >= 1 && LapsByte <= 99) {
                    return LapsByte;
                }
                else if (LapsByte >= 100 && LapsByte <= 190) {
                    return (LapsByte - 100) * 10 + 100;
                }
                else if (Winner != null && Winner.LapsDone > 0) {
                    return Winner.LapsDone;
                }
                return 0;
            }
        }

        [XmlIgnore]
        public int Hours {
            get {
                if (LapsByte >= 191 && LapsByte <= 238) {
                    return LapsByte - 190;
                }
                return 0;
            }
        }

        [XmlIgnore]
        public Result Winner { get; set; }

        [XmlIgnore]
        public Result FastestLap { get; set; }

        [XmlIgnore]
        public Result PolePosition { get; set; }

        [XmlIgnore]
        public Result HighestClimb { get; set; }

        [XmlIgnore]
        public RaceMode RaceMode {
            get {
                if (LapsByte == 0) {
                    return RaceMode.Practice;
                }
                if (Winner != null && Winner.LapsDone == 0) {
                    return RaceMode.Qualifying;
                }
                return RaceMode.Race;
            }
        }

        [XmlIgnore]
        public object Tag { get; set; }

        public Replay() {
            ReplayId = Guid.NewGuid().ToString();
            Results = new List<Result>();
        }

        public int GetTotalPoints(Result result) {
            return Allocation.GetTotalPoints(result);
        }

        public void ReorderResults() {
            var ordered = from r in Results
                          orderby r.LapsDone descending, r.RaceTime
                          select r;

            if (Results.Any()) {
                int position = 0;
                Winner = null;
                FastestLap = null;
                HighestClimb = null;
                PolePosition = null;

                foreach (Result result in ordered) {
                    result.Position = ++position;

                    if (Winner == null && result.Position == 1) {
                        Winner = result;
                    }

                    if (FastestLap == null || result.BestLapTime.Value < FastestLap.BestLapTime.Value) {
                        FastestLap = result;
                    }

                    if (HighestClimb == null || result.Climb > HighestClimb.Climb) {
                        HighestClimb = result;
                    }

                    if (PolePosition == null || result.StartPosition == 1) {
                        PolePosition = result;
                    }

                    if (Winner != null && Winner.LapsDone > 0) {
                        result.PercentCompleted = (result.LapsDone * 100.0) / Winner.LapsDone;
                    }
                }

                Winner.IsWinner = true;
                FastestLap.IsFastestDriver = true;
                HighestClimb.IsHighestClimber = true;
                if (PolePosition != null) {
                    PolePosition.IsPoleSitter = true;
                }
            }
        }

        [SuppressMessage("Microsoft.Usage", "CA2202:Do not dispose objects multiple times")]
        public static Replay FromFile(string fileName, Allocation allocation) {
            const string Header = "LFSMPR";

            using (FileStream input = File.Open(fileName, FileMode.Open, FileAccess.Read, FileShare.Read))
            using (LfsFileReader reader = new LfsFileReader(input)) {
                //6     char    0       LFSMPR              : do not read file if no match
                if (reader.ReadString(Header.Length) != Header) {
                    throw new InvalidOperationException(String.Format(
                        StringResources.InvalidReplayHelperExceptionMessage,
                        fileName));
                }

                Replay replay = new Replay();
                replay.Name = Path.GetFileNameWithoutExtension(fileName);
                replay.Allocation = allocation;

                //1     byte    6       game version        : ignore
                //1     byte    7       game revision       : ignore
                //1     byte    8       MPR version         : ignore
                //1     byte    9       immediate start     : joined already running game
                //1     byte    10      reserved            : -
                //1     byte    11      reserved            : -
                //1     int     12      rules               : -
                //1     int     16      flags               : -
                reader.Skip(14);
                //1     byte    20      laps byte           : laps / hours (see notes)
                replay.LapsByte = reader.ReadByte();
                //1     byte    21      skill               : skill level (0,1,2,3,4)
                //1     byte    22      wind                : 0=off 1=weak 2=strong
                //1     byte    23      num players         : players at start of race
                //8     char    24      LFS version         : text, ends 0
                reader.Skip(11);
                //4     char    32      short track name    : e.g. BL2R
                replay.TrackName = reader.ReadString(4);
                //1     int     36      start time (UTC)    : seconds from 00:00 1/1/1970
                replay.StartTime = new SerializableDateTime(reader.ReadTimestamp());
                //32    char    40      track name          : text, ends 0
                //1     byte    72      config              : 1,2,3.. (first config is 1)
                //1     byte    73      reversed            : 0=no 1=yes
                //1     byte    74      weather             : 0,1,2.. (first weather is 0)
                reader.Skip(35);
                //1     byte    75      num finished (NF)   : players in results table
                int finished = reader.ReadByte();
                //1     int     76      0                   : -
                reader.Skip(4);

                for (int i = 0; i < finished; i++) {
                    replay.Results.Add(Result.FromFile(reader, replay));
                }

                // Figure out the race results. 
                replay.ReorderResults();

                return replay;
            }
        }
    }
}
