﻿using System;
using System.IO;
using System.Linq;
using System.Xml.Linq;
using System.Xml.Xsl;

namespace LfsPoints.Projects {
    public class HtmlExporter : IExporter {
        private static readonly string XslFile = Path.Combine(Program.StartupPath, "LFSPoints.xsl");

        public void Export(string fileName, Project project) {
            // HTML export happens in three stages. We create a temp XML file with the project
            // data, then we perform an XSL transformation on the data to turn it into a HTML
            // file, then we save the HTML file and delete the temp file. Simples!

            string tempXmlFileName = null;
            try {
                tempXmlFileName = GetTempFileName();

                GenerateTempXmlFile(tempXmlFileName, project);

                PerformXslTransformation(tempXmlFileName, fileName);
            }
            finally {
                if (File.Exists(tempXmlFileName)) {
                    File.Delete(tempXmlFileName);
                }
            }
        }

        public void Export(string fileName, Replay replay) {
            throw new NotSupportedException();
        }

        private static string GetTempFileName() {
            return Path.GetTempFileName();
        }

        private static void GenerateTempXmlFile(string tempXmlFileName, Project project) {
            XDocument xDocument = new XDocument();
            XElement xProject = new XElement("Project", new XAttribute("Name", project.Name));

            XElement xStandings = new XElement("Standings");
            foreach (Standing standing in project.Standings) {
                XElement xStanding = new XElement("Standing");
                xStanding.Add(new XElement("Position", standing.Position));
                xStanding.Add(new XElement("Driver", standing.DisplayName));
                xStanding.Add(new XElement("Points", standing.Points));
                xStandings.Add(xStanding);
            }
            xProject.Add(xStandings);

            XElement xReplays = new XElement("Replays");
            foreach (Replay replay in project.Replays.OrderBy(r => r.StartTime.Value)) {
                XElement xReplay = new XElement("Replay");
                xReplay.Add(new XElement("Name", replay.Name));
                xReplay.Add(new XElement("Track", ReplayHelper.GetTrack(replay)));
                xReplay.Add(new XElement("Laps", ReplayHelper.GetLaps(replay)));
                xReplay.Add(new XElement("Date", replay.StartTime.ToString("g")));
                xReplay.Add(new XElement("Points", replay.Allocation.Name));
                xReplay.Add(new XElement("Finished", replay.Results.Count));

                XElement xResults = new XElement("Results");
                foreach (Result result in replay.Results.OrderBy(r => r.Position)) {
                    XElement xResult = new XElement("Result");
                    xResult.Add(new XElement("Position", result.Position));
                    xResult.Add(new XElement("Driver", result.DisplayName));
                    xResult.Add(new XElement("Car", ReplayHelper.GetCar(result)));
                    xResult.Add(new XElement("Grid", result.StartPosition));
                    xResult.Add(new XElement("Time", ReplayHelper.GetRaceTime(result)));
                    xResult.Add(new XElement("Penalty", ReplayHelper.GetPenalty(result)));
                    xResult.Add(new XElement("BestLap", ReplayHelper.GetBestLap(result)));
                    xResult.Add(new XElement("Points", ReplayHelper.GetPoints(result)));
                    xResults.Add(xResult);
                }
                xReplay.Add(xResults);
                xReplays.Add(xReplay);
            }
            xProject.Add(xReplays);

            xDocument.Add(xProject);
            xDocument.Save(tempXmlFileName);
        }

        private static void PerformXslTransformation(string tempXmlFileName, string fileName) {
            XslCompiledTransform xslTransform = new XslCompiledTransform();
            xslTransform.Load(XslFile);
            xslTransform.Transform(tempXmlFileName, fileName);
        }
    }
}
