﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Xml.Serialization;

namespace LfsPoints.Projects {
    [Serializable]
    [DebuggerDisplay("Name = {Name}, Points = {Points.Count}")]
    public class Allocation {
        public string AllocationId { get; set; }
        public string Name { get; set; }
        public List<int> Points { get; set; }
        public int Winner { get; set; }
        public int PolePosition { get; set; }
        public int FastestLap { get; set; }
        public int HighestClimb { get; set; }
        public int PercentCompleted { get; set; }
        public bool OmitDisqualified { get; set; }

        [XmlIgnore]
        public object Tag { get; set; }

        public Allocation() {
            Points = new List<int>();
        }

        public Allocation(string name) {
            AllocationId = Guid.NewGuid().ToString();
            Name = name;
            Points = new List<int>(new int[10]);
            PercentCompleted = 60;
            OmitDisqualified = true;
        }

        private int GetPoints(int position) {
            int index = position - 1;
            if (index < Points.Count) {
                return Points[index];
            }
            return 0;
        }

        public int GetTotalPoints(Result result) {
            if ((OmitDisqualified && result.IsDisqualified) || 
                (result.PercentCompleted < PercentCompleted) || 
                (result.Replay.RaceMode != RaceMode.Race)) {
                return 0;
            }

            int points = GetPoints(result.Position);

            if (result.IsWinner) {
                points += Winner;
            }

            if (result.IsPoleSitter) {
                points += PolePosition;
            }

            if (result.IsFastestDriver) {
                points += FastestLap;
            }

            if (result.IsHighestClimber) {
                points += HighestClimb;
            }

            return points;
        }

        public override string ToString() {
            return Name;
        }

        public static void Save(string fileName, Allocation allocation) {
            SerializationHelper.Serialize(fileName, allocation);
        }

        public static Allocation Load(string fileName) {
            return SerializationHelper.Deserialize<Allocation>(fileName);
        }
    }
}
